/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import com.adobe.xfa.ut.Assertions;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.StringUtils;


/**
 * A common base class to represent XML attributes.
 * 
 * @author John Brinkman
 */
abstract public class Attribute {
	private final String mQName;
	private final String mLocalName;
	private final String mPrefix;
	private String mVal;

	/**
	 * Instantiates an attribute, with the given attribute value.
	 * 
	 * @param qName the attribute name.
	 * @param value the attribute value.
	 */
	public Attribute(String qName, String value) {
		this(null, null, qName, value, true);
	}

	/**
	 * Instantiates an attribute, with the given attribute parameters.
	 * @param NS the namespace of this attribute.
	 * @param localName the local name of this attribute.
	 * @param qName the qualified name of this attribute.
	 * @param value the value of this attribute.
	 */
	public Attribute(String NS, String localName, String qName, String value) {
		this(NS, localName, qName, value, true);
	}
	
	/**
	 * Instantiates an attribute, with the given attribute parameters.
	 * @param NS the namespace of this attribute.
	 * @param localName the local name of this attribute.
	 * @param qName the qualified name of this attribute.
	 * @param value the value of this attribute.
	 * @param internSymbols indicates whether the other parameters must be interned.
	 * 
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES,RCN")
	protected Attribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		if (Assertions.isEnabled) {
			if (!internSymbols) {
				assert NS == null || NS == NS.intern();
				assert localName == null || localName == localName.intern();
				assert qName == null || qName == qName.intern();
			}
		}

		String prefix = "";
		
		assert (value != null);
		mVal = (value == null) ? "" : value;
		
		// If there is a qName, it will take precedence, and we can
		// extract the localName from the qName.
		// One of the goals here is to avoid String.intern() operations
		// wherever possible for performance reasons, so if there is 
		// a non-empty localName, we will use that instead of extracting
		// it from qName. We know that there is existing code that will
		// pass a qualified name into the localName parameter, so we have
		// to guard against that happening.
		
		if (qName != null) {
			
			// Xerces will provide an empty localName for xmlns attributes,
			// so we can't just assume that the presence of a qName implies 
			// that a corresponding localName is provided.
			
			int colon = qName.indexOf(':');
			if (localName != null && localName.length() != 0 && localName.indexOf(':') < 0) {
			
				if (internSymbols) {
					localName = localName.intern();
					qName = qName.intern();
				}
			} 
			else {
				
				if (internSymbols) {
					qName = qName.intern();
				}
				
				localName = (colon >= 0) ? qName.substring(colon + 1).intern() : qName;
			}
			
			if (colon == 5 && qName.startsWith(STRS.XMLNS)) {
				// xmlns:prefix is a common enough case to deal with specially to avoid a new string/intern
				prefix = "xmlns";
			}
			else if (colon > 0) {					
				prefix = qName.substring(0, colon).intern();
			}
		}
		else if (localName != null) {
			
			if (internSymbols) {
				localName = localName.intern();
			}
			
			qName = localName;
		}
		
		mQName = qName;
		mLocalName = localName;
		mPrefix = prefix;
		
		// If this is a name or namespace attribute, intern the value.
		if (mLocalName == XFA.NAME || isNameSpaceAttr()) {
			mVal = mVal.intern();
		}
		
		if (Assertions.isEnabled) {
			assert mQName == null || mQName == qName.intern();
			assert mLocalName == null || mLocalName == mLocalName.intern();
		}
	}
	
	/**
	 * Gets this attribute's value.
	 * @return the attribute value.
	 */
	public final String getAttrValue() {
		return mVal;
	}

	/**
	 * Gets this attribute's local name.
	 * @return The local part of the name (everything after the ":")
	 */
	public final String getLocalName() {
		return mLocalName;
	}
	/**
	 * Gets this attribute's local name.
	 * @return Returns the local name.
	 */
	public final String getName() {
		return mLocalName;
	}

	/**
	 * Gets this attribute's namespace.
	 * @return Returns the namespace.
	 */
	public String getNS() {
		return null;
	}

	/**
	 * Gets this attribute's namespace prefix.
	 * @return the attribute namespace prefix.
	 */
	public final String getPrefix() {
		return mPrefix;
	}

	/**
	 * Gets this attribute's qualified name.
	 * @return the qualified name.
	 */
	public final String getQName() {
		return mQName;
	}
	
	/**
	 * Determines if this attribute's value is empty (or null).
	 * @return true if this attribute's value is empty and false otherwise.
	 */
	public final boolean isEmpty() {
		return StringUtils.isEmpty(getAttrValue());
	}

	/**
	 * Determines if this attribute is a name space attribute.
	 * @return true if this is a name space attribute and false otherwise.
	 */
	public final boolean isNameSpaceAttr() {
		return getQName() == STRS.XMLNS || getPrefix() == STRS.XMLNS;
	}
	
	/**
	 * the URI for xsi has many variants by date e.g.
	 * http://www.w3.org/[this section may vary]/XMLSchema-instance
	 * Pre-2001 used the "null" attribute, the 2001 version rename this to "nil"
	 * We recognize any combination of nil or null with any XMLSchema-instance 
	 * namespace URI.
     *
	 * @return true if this attribute represents the xsi:nil attribute, or one of
	 * its variants.
	 *
	 * @exclude from published api.
	 */
	public final boolean isXSINilAttr() {
		final String localName = getLocalName();
		if (localName == STRS.NIL || localName == STRS.NULL) {
			
			// check the namespace
			final String ns = getNS();
			if (ns == null)
				return false;
			
			return ns == STRS.XSINS || 
				(ns.startsWith(STRS.XSINSPREFIX) && ns.endsWith(STRS.XSINSSUFFIX));
		}
		
		return false;
	}

	/**
	 * Determine if an attribute is a schema attribute.
	 * @return true this is a schema attribute.
	 *
	 * @exclude from published api.
	 */
	public boolean isSchemaAttr() {
		return true;
	}


	/**
	 * Create a new attribute, given a new value
	 * @param value
	 *            the string to use to create the new attribute
	 * @return a new attribute
	 *
	 * @exclude from published api.
	 */
	abstract public Attribute newAttribute(String value);

	/**
	 * Create a new attribute, given all attribute parameters
	 * @param NS the namespace for this attribute
	 * @param localName the local name for this attribute
	 * @param qName the qualified name for this attribute 
	 * @param value the string to use to create the new attribute
	 * @return a new attribute
	 *
	 * @exclude from published api.
	 */
	abstract public Attribute newAttribute(String NS, String localName, String qName, String value);
	
	/**
	 * Create a new attribute, given all attribute parameters
	 * @param NS the namespace for this attribute
	 * @param localName the local name for this attribute
	 * @param qName the qualified name for this attribute 
	 * @param value the string to use to create the new attribute
	 * @param internSymbols indicates whether the symbols in other parameters need to be interned.
	 * @return a new attribute
	 *
	 * @exclude from published api.
	 */
	abstract public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols);
	
	
	/**
	 * Normalizes this attribute.
	 * This base class implementation simply does nothing,
	 * other than allow derived classes the opportunity
	 * to do something.
	 *
	 * @exclude from published api.
	 */
	public void normalize() {
	}
	
	/**
	 * Sets this attribute's value.
	 * <p>
	 * Since attributes are immutable, the value should never really change.
	 * It is acceptable for a derived class to modify the string value to
	 * normalize it since that doesn't really modify the attribute's value.
	 * 
	 * @param value the new attribute value.
	 * 
	 * @exclude from published api.
	 **/
	protected final void setAttrValue(String value) {
		mVal = value;
	}
	
	/**
	 * Generates this attribute's value as a string.
	 * @return the attribute value.
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return mVal;
	}
}
