/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.io.IOException;
import java.io.OutputStream;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;


/**
 * A class to represent the XML processing instruction nodes in the DOM.
 * <p>A processing instruction's name is
 * the target name that follows the <code>&lt;?.
 * <p>A processing instruction's data is all of the characters
 * following the target name and space and ending <code>?&gt;</code>.
 */
public final class ProcessingInstruction extends Node {

	private final static String mClassName = "processingInstruction"; 
	
	private String mData;

	private final String mName;

	/**
	 * The mDoc member will be populated if this is a document-level PI.
	 */
	private final Document mDoc;
	
	/**
	 * Instantiates a processing instruction with the given parameters.
	 * @param parent the node's parent, if any.
	 * @param prevSibling the node's previous sibling, if any.
	 * @param name the node's name.
	 * @param data the node's data.
	 */
	public ProcessingInstruction(Element parent, Node prevSibling, String name, String data) {
		super(parent, prevSibling);
		mName = name;
		mData = data;
		mDoc = null;
	}

	/**
	 * @exclude from published api.
	 */
	public ProcessingInstruction(Document doc, String name, String data) {
		super(null, null);
		mName = name;
		mData = data;
		mDoc = doc;
		if (doc != null)
			mDoc.appendChild(this);
	}
	
	/**
	 * @exclude from published api.
	 */
	public Node clone(Element parent) {
		return new ProcessingInstruction(parent, null, mName, mData);
	}

	/**
	 * @exclude from published api.
	 */
	public String getClassAtom() {
		return mClassName;
	}
	
	/**
	 * @exclude from published api.
	 */
	public String getClassName() {
		return mClassName;
	}

	/**
	 * @see Node#getData()
	 */
	public String getData() {
		return mData;
	}

	/**
	 * Gets this node's name.
	 * @return the processing instruction name.
	 */
	public String getName() {
		return mName;
	}

	/**
	 * @exclude from published api.
	 */
	public boolean isLeaf() {
		return true;
	}

	/**
	 * @exclude from published api.
	 */
	public void postSave() {
		// do nothing
	}

	/**
	 * @exclude from published api.
	 */
	public void preSave(boolean bSaveXMLScript) {
		// do nothing
	}

	/**
	 * @exclude from published api.
	 */
	public void serialize(OutputStream outStream, DOMSaveOptions options, int level, Node prevSibling) throws IOException {
		
		if (level != 0 || prevSibling != null) {
			if (options.getDisplayFormat() == DOMSaveOptions.PRETTY_OUTPUT)
				options.writeIndent(outStream, level);
			else if (level == 0 && (options.getFormatOutside() || options.getDisplayFormat() == DOMSaveOptions.SIMPLE_OUTPUT))
				outStream.write(Document.MarkupReturn);
		}
		outStream.write(Document.MarkupPIStart);
		outStream.write(getName().getBytes(Document.Encoding));
		String sData = getData();
		if (!StringUtils.isEmpty(sData)) {
			outStream.write(Document.MarkupSpace);
			outStream.write(sData.getBytes(Document.Encoding));
		}
		outStream.write(Document.MarkupPIEnd);
	}

	/**
	 * @exclude from published api.
	 */
	public void setData(String data) {
		mData = data;
		setDirty();
	}

	/**
	 * @exclude from published api.
	 */
	public void setScriptProperty(String sPropertyName, Arg propertyValue) {
		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "ProcessingInstruction#setScriptProperty");
	}

}
