/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

/**
 * This class is used to store schema relationships. It stores an array of
 * schema tags along with either a corresponding array of either ChildReln or an
 * Attribute.
 * 
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class SchemaPairs {
	
	public SchemaPairs(SchemaPairs other) {
		mCount = other.mCount;
		meTags = new int[mCount];
		System.arraycopy(other.meTags, 0, meTags, 0, mCount);
		mSchemaObj = new Object[mCount];
		System.arraycopy(other.mSchemaObj, 0, mSchemaObj, 0, mCount);
	}
	
	public SchemaPairs() {
	}
	
	private int mCount;

	private int[] meTags;

	private Object[] mSchemaObj;

	/**
	 * Gets the value for a specific key (tag)
	 * @param eTag the tag to search for
	 * @return the Schema object corresponding to the tag.  null if not found.
	 */
	public Object get(int eTag) {
		for (int i = 0; i < mCount; i++) {
			final int tag = meTags[i];
			if (tag == eTag) {
				return mSchemaObj[i];
			}
			
			if (eTag < tag)
				return null;
		}
		
		return null;
	}
	
	/**
	 * Returns the key (tag) at this position in the list.
	 * 
	 * @param index the position in the list
	 * @return the tag number
	 */
	public int key(int index) {
		return meTags[index];
	}

	/**
	 * Adds a schema pair to the list. We first check if the value exists and if
	 * it does we replace the existing value. We sort the internal list by tag
	 * number.
	 * 
	 * @param eTag the tag of the schema element we're adding
	 * @param schema the schema object - Normally a ChildReln or an Attribute
	 */
	public void put(int eTag, Object schema) {
		// If it already exists, replace it.
		// Otherwise, determine the insertion point.
		int index = 0;
		for (; index < mCount; index++) {
			final int tag = meTags[index];
			if (tag == eTag) {
				mSchemaObj[index] = schema;
				return;
			}
			
			if (eTag < tag)
				break;
		}
		
		// First time in allocation.
		if (meTags == null) {
			meTags = new int[4];
			mSchemaObj = new Object[4];
		}		
		// Re-allocate if necessary
		else if (mCount == meTags.length) {
			final int newSize = mCount * 2;
			final int[] newTags = new int[newSize];
			System.arraycopy(meTags, 0, newTags, 0, mCount);
			final Object[] newObj = new Object[newSize];
			System.arraycopy(mSchemaObj, 0, newObj, 0, mCount);
			meTags = newTags;
			mSchemaObj = newObj;
		}
		
		final int elementsToShift = mCount - index;
		if (elementsToShift != 0) {
			System.arraycopy(meTags, index, meTags, index + 1, elementsToShift);
			System.arraycopy(mSchemaObj, index, mSchemaObj, index + 1, elementsToShift);
		}
		
		meTags[index] = eTag;
		mSchemaObj[index] = schema;
		mCount++;
	}
	
	/**
	 * Returns the size of the list.
	 * 
	 * @return the list size.
	 */
	public int size() {
		return mCount;
	}

	/**
	 * Returns the schema object at this position in the list
	 * 
	 * @param index the position in the list
	 * @return the schema object
	 */
	public Object value(int index) {
		return mSchemaObj[index];
	}
}