package com.adobe.xfa.gfx;

import com.adobe.xfa.ut.UnitSpan;

/**
 * <p>
 * The graphic Line attribute object holds information about attribute
 * values specific for line drawing.
 * @exclude from published api.
 */

public class GFXLineAttr extends GFXAttr {
	public final static UnitSpan DEFAULT_LINEWIDTH = new UnitSpan (UnitSpan.INCHES_72K, 720);	// 0.01in

	public final static GFXLineAttr BLACK_LINE = new GFXLineAttr (DEFAULT_LINEWIDTH, STYLE_SOLID, DEFAULT_SHADESCALE, GFXColour.BLACK, GFXColour.WHITE);
	public final static GFXLineAttr WHITE_LINE = new GFXLineAttr (DEFAULT_LINEWIDTH, STYLE_SOLID, DEFAULT_SHADESCALE, GFXColour.WHITE, GFXColour.WHITE);
	public final static GFXLineAttr LIGHTGRAY_LINE = new GFXLineAttr (DEFAULT_LINEWIDTH, STYLE_SOLID, DEFAULT_SHADESCALE, GFXColour.LIGHTGRAY, GFXColour.WHITE);
	public final static GFXLineAttr GRAY_LINE = new GFXLineAttr (DEFAULT_LINEWIDTH, STYLE_SOLID, DEFAULT_SHADESCALE, GFXColour.GRAY, GFXColour.WHITE);
	public final static GFXLineAttr DARKGRAY_LINE = new GFXLineAttr (DEFAULT_LINEWIDTH, STYLE_SOLID, DEFAULT_SHADESCALE, GFXColour.DARKGRAY, GFXColour.WHITE);

/**
 * enumeration int: Specifies the type of coordinate system to use.
 * <p>
 * <pre>
 *	 HAND_LEFT	: left-handed
 *	 HAND_RIGHT : right-handed
 *	 HAND_EVEN	: even-handed (?)
 * </pre>
 */
	public static final int HAND_LEFT = 0;
	public static final int HAND_RIGHT = 1;
	public static final int HAND_EVEN = 2;

/**
 * enumeration int: Specifies the current device capablilites.
 * <p>
 * <pre>
 *	 CAP_BUTT	:
 *	 CAP_ROUND	:
 *	 CAP_SQUARE :
 * </pre>
 */
	public static final int CAP_BUTT = 0;
	public static final int CAP_ROUND = 1;
	public static final int CAP_SQUARE = 2;

	private UnitSpan moWidth; // Line width.
	private int moHand;
	private int moCap;

/**
 * Default constructor.
 * <p>
 * Creates a new Line attribute with default settings:
 * <pre>
 *	 Width - default width
 *	 Hand  - even
 *	 Cap   - square
 * </pre>
 */
	public GFXLineAttr () {
		moWidth = DEFAULT_LINEWIDTH;
		moHand = HAND_EVEN;
		moCap = CAP_SQUARE;
	}

/**
 * Copy constructor.
 * <p>
 * Creates a new Line attribute with the same settings
 * as the source attribute.
 * @param oSource - Line attribute to copy
 */
	public GFXLineAttr (GFXLineAttr oSource) {
		copyFrom (oSource);
	}

/**
 * Constructor.
 * <p>
 * Creates a new Line attribute with the specified settings.
 * @param oNewWidth - The width for this attribute
 * @param nNewStyle - The style setting for this attribute
 * @param lNewShade - The shade setting for this attribute
 * @param oNewColour - The foreground colour to be used
 * @param oNewColourBg - The background colour to be used
 */
	public GFXLineAttr (UnitSpan oNewWidth, int nNewStyle, int lNewShade, GFXColour oNewColour, GFXColour oNewColourBg) {
		super (nNewStyle, lNewShade, oNewColour, oNewColourBg);
		moWidth = oNewWidth;
		moHand = HAND_EVEN;
		moCap = CAP_SQUARE;
	}

// Some Standard Line Attribute Definitions
/**
 * Get a default Line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width				= default
 *	 Style				= solid
 *	 Shade scale		= maximum
 *	 Foreground colour	= black
 *	 Background colour	= white
 * </pre>
 * @return A Line Attribute with default settings
 */
	public GFXLineAttr defaultLine () {
		return BLACK_LINE;
	}

/**
 * Get a black line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width			   = default
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = black
 *	 Background colour = white
 * </pre>
 * @return A Line Attribute with black line settings
 */
	public GFXLineAttr blackLine () {
		return BLACK_LINE;
	}

/**
 * Get a white line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width			   = default
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = white
 *	 Background colour = white
 * </pre>
 * @return A Line Attribute with white line settings
 */
	public GFXLineAttr whiteLine () {
		return WHITE_LINE;
	}

/**
 * Get a light gray line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width			   = default
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = light gray
 *	 Background colour = white
 * </pre>
 * @return A Line Attribute with light gray line settings
 */
	public GFXLineAttr lightGrayLine () {
		return LIGHTGRAY_LINE;
	}

/**
 * Get a gray line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width			   = default
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = gray
 *	 Background colour = white
 * </pre>
 * @return A Line Attribute with gray line settings
 */
	public GFXLineAttr grayLine () {
		return GRAY_LINE;
	}

/**
 * Get a dark gray line Attribute.
 * <p>
 * This will return a Line Attribute with the following settings:
 * <pre>
 *	 Width			   = default
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = dark gray
 *	 Background colour = white
 * </pre>
 * @return A Line Attribute with dark gray line settings
 */
	public GFXLineAttr darkGrayLine () {
		return DARKGRAY_LINE;
	}

/**
 * Get the width for this attribute.
 * @return The width as a <code>UnitSpan</code> object
 */
	public UnitSpan width () {
		return moWidth;
	}

/**
 * Set the width for this attribute.
 * @param oNewWidth - The new width
 */
	public void width (UnitSpan oNewWidth) {
		moWidth = oNewWidth;
	}

/**
 * Get the hand setting for this attribute.
 * @return The hand setting as a int enumeration
 */
	public int hand () {
		return moHand;
	}

/**
 * Set the hand setting for this attribute.
 * @param oNewCode - The new hand setting
 */
	public void hand (int oNewCode) {
		moHand = oNewCode;
	}

/**
 * Get the capability for this attribute.
 * @return The capability as a int enumeration
 */
	public int cap () {
		return moCap;
	}

/**
 * Set the capability for this attribute.
 * @param oNewCode - The capability value
 */
	public void cap (int oNewCode) {
		moCap = oNewCode;
	}

/**
 * Equality comparison.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param oCompare - Attribute object to compare against.
 * @return TRUE if all members are equal, FALSE otherwise.
 */
	public boolean equivalent (GFXLineAttr oCompare) {
		return (super.equivalent (oCompare)
			&& (moWidth == oCompare.moWidth)
			&& (moHand == oCompare.moHand)
			&& (moCap == oCompare.moCap));
	}

/**
 * Equality comparison operator.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param object - Attribute object to compare against.
 * @return TRUE if the attribute objects are considered equal;
 * FALSE otherwise.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		if (!super.equals(object))
			return false;

		return equivalent ((GFXLineAttr) object);
	}

	public int hashCode() {
		int hash = 37;
		hash = (hash * 31) ^ super.hashCode();
		hash = (hash * 31) ^ moWidth.hashCode();
		hash = (hash * 31) ^ moHand;
		hash = (hash * 31) ^ moCap;
		return hash;
	}

/**
 * Assigment operator.
 * <p>
 * Replace all attribute settings with those from the source object.
 * @param oSource - Source attribute object to copy.
 */
	public void copyFrom (GFXLineAttr oSource) {
		if (this != oSource) {
			super.copyFrom (oSource);
			moWidth = oSource.moWidth;
			moHand = oSource.moHand;
			moCap = oSource.moCap;
		}
	}
}
