/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.ut;

/**
 * The <b>ISODate</b> class derives from <b>LcDate</b> to
 * define ISO8601/XFA date patterns.
 * 
 * <p>Valid ISO8601/XFA date strings are in any one
 * of the following date patterns:
 * <ul>
 * <li> YYYY[-MM[-DD]]
 * <li> YYYY[MM[DD]]
 * </ul>
 * where [...] denotes optional elements.
 * See {@link LcDate} for the meaning of the various metasymbols used above.
 *
 * Here's a snippet of code illustrating the use of
 * {@link ISODate} to reformat an ISO8601/XFA date string
 * <pre><code>
 *      import com.adobe.xfa.ut.ISODate;
 *      import com.adobe.xfa.ut.LcDate;
 *      ...
 *      ISODate d = new ISODate("2000-02-28", "");
 *      String s = "Unknown";
 *      if (d.isValid())
 *          s = d.format(LcDate.XFA_TIME_FMT1);
 *      System.out.println(s);
 * </code></pre>
 *
 * @exclude from published api.
 */
public final class ISODate extends LcDate {
    /**
     * Instantiate an ISODate object from today's date.
     */
    public ISODate() {
    	super(LcLocale.DEFAULT_LOCALE, DEFAULT_CENTURY_SPLIT);
	}

    /**
     * Instantiate an ISODate object from the given number of days
     * from the epoch.  The epoch is such that day 1 corresponds
     * to Jan 1, 1900.
     * @param days the number of days from epoch.
     */
    public ISODate(int days) {
    	super(days, LcLocale.DEFAULT_LOCALE, DEFAULT_CENTURY_SPLIT);
	}

    /**
     * Instantiate a ISODate object from the given date string.
     * @param date an ISO8601/XFA date string.
     * @param locale a locale string.  When empty, it will default
     * to the current locale.
     */
    public ISODate(String date, String locale /* = "" */) {
    	super(locale, DEFAULT_CENTURY_SPLIT);
		//
		// Remove any ISO time component from date string.
		//
		String trimDate = date;
		int tee = date.indexOf('T');
		if (tee >= 0)
			trimDate = date.substring(0, tee);
		//
		// Remove any leading and trailing white spaces from date string.
		//
		trimDate = trimDate.trim();
		mCenturySplit = DEFAULT_CENTURY_SPLIT;
		setDateSymbols(LcLocale.DEFAULT_LOCALE);
		if (parse(trimDate)) {
			mYearOfEra -= 1900;
			if (mDayOfMonth < 0 || mMonthOfYear < 0 || mYearOfEra < 0)
				mDays = 0;
			else
				mDays = epoch(mDayOfMonth, mMonthOfYear, mYearOfEra);
		}
		else {
			mDays = 0;
		}
		setDateSymbols(locale);
		mValid = (mDays > 0);
	}

    /**
     * Instantiate a ISODate object from today's date given a locale string
     * and a possible century split year.
     * @param locale a locale string.  When empty, it will default
     * to the current locale.
     * @param centurySplit a century split year.
     */
    public ISODate(String locale, int centurySplit) {
    	super(locale, centurySplit);
    }


	/*
	 * Parse the given date string.
	 *
	 * @param src the date string to parse.
	 * Valid ISO dates are in one of two formats:
	 * <ul>
	 * <li> YYYY[-MM[-DD]]
	 * <li> YYYY[MM[DD]]
	 * </ul>
	 * @return boolean true if successfully parsed, and false otherwise.
	 */
	boolean parse(String src) {
		boolean needDashes = (src.indexOf('-') >= 0);
		int srcPos = 0;
		int srcLen = src.length();
		int parseRes;
		//
		// Reset parsed date sub-elements.
		//
		mDayOfWeek = mDayOfMonth = mDayOfYear = mMonthOfYear = mYearOfEra = -1;
		//
		// Parse year.
		//
		if (srcPos + 4 > srcLen)
			return false;
		parseRes = subParse(src, srcPos, 'Y', 4);
		if (parseRes < 0)
			return false;
		srcPos = parseRes;
		//
		// Return if we're done.
		//
		if (srcPos == srcLen) {
			mMonthOfYear = 1;
			mDayOfMonth = 1;
			return true;
		}
		if (needDashes) {
			if (! DateTimeUtil.matchChr(src, srcPos, '-', false))
				return false;
			srcPos += 1;
		}
		//
		// Parse any optional month.
		//
		if (srcPos + 2 > srcLen)
			return false;
		parseRes = subParse(src, srcPos, 'M', 2);
		if (parseRes < 0)
			return false;
		srcPos = parseRes;
		//
		// Return if we're done.
		//
		if (srcPos == srcLen) {
			mDayOfMonth = 1;
			return true;
		}
		if (needDashes) {
			if (! DateTimeUtil.matchChr(src, srcPos, '-', false))
				return false;
			srcPos += 1;
		}
		//
		// Parse any optional days.
		//
		if (srcPos + 2 > srcLen)
			return false;
		parseRes = subParse(src, srcPos, 'D', 2);
		if (parseRes < 0)
			return false;
		srcPos = parseRes;
		//
		// Ensure there's no more source to parse.
		//
		if (srcPos != srcLen)
			return false;
		return true;
	}

}
