/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2021 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.replication;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import org.osgi.service.event.Event;

public class ReplicatedAction {

    public static final String PROPERTY_AGENT_IDS = "agentIds";

    public ReplicatedAction(@Nonnull ReplicationAction replicationAction, List<String> agentIds) {
        super();
        this.replicationAction = replicationAction;
        this.agentIds = agentIds != null ? Collections.unmodifiableList(agentIds) : null;
    }

    private final ReplicationAction replicationAction;
    private final List<String> agentIds;

    /**
     * Return the ReplicationAction
     * 
     * @return the ReplicationAction
     */
    public ReplicationAction getReplicationAction() {
        return replicationAction;
    }

    /**
     * Return the agentIds
     * 
     * @return the agentIds
     */
    public List<String> getAgentIds() {
        return agentIds;
    }

    /**
     * Convert an OSGi event to a replication event.
     *
     * @param evt The OSGi event
     * @return The replication event if the OSGi event contained a replication event. Otherwise null is returned.
     */
    @SuppressWarnings("unchecked")
    public static ReplicatedAction fromEvent(Event evt) {
        ReplicationAction replicationAction = ReplicationAction.fromEvent(evt);
        if (replicationAction != null) {
            return new ReplicatedAction(replicationAction, (List<String>) evt.getProperty(PROPERTY_AGENT_IDS));
        }
        return null;
    }

    /**
     * Create an OSGi event out of the replication event.
     * 
     * @param distributeEvents if <code>true</code> a distributed event is created
     * @return A new OSGi event.
     */
    public Event toEvent(boolean distributeEvents) {
        Map<String, Object> properties = new HashMap<>(replicationAction.createEventProperties(distributeEvents));
        if (agentIds != null) {
            properties.put(PROPERTY_AGENT_IDS, agentIds);
        }
        return new Event(ReplicationAction.EVENT_TOPIC, properties);
    }
}
