/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.facets.extractors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import com.day.cq.search.facets.Facet;
import com.day.cq.search.facets.buckets.PredefinedBucket;

/**
 * {@linkplain PredefinedBucketsFacetExtractor} extracts facets based on a list
 * of {@link PredefinedBucket predefined buckets}. This means that the buckets
 * returned will at most be the ones added in the
 * {@link #PredefinedBucketsFacetExtractor(String, Collection) constructor} or
 * through {@link #addPredefinedBucket(PredefinedBucket)}, but could be less,
 * because only buckets with at least one match will be returned in the facet.
 * 
 * @since 5.2
 */
public class PredefinedBucketsFacetExtractor extends PropertyFacetExtractor {

    private List<PredefinedBucket> buckets = new ArrayList<PredefinedBucket>();
    
    private Facet facet = null;
    
    public PredefinedBucketsFacetExtractor(String propertyRelPath) {
        super(propertyRelPath);
    }
    
    public PredefinedBucketsFacetExtractor(String propertyRelPath, Collection<? extends PredefinedBucket> buckets) {
        super(propertyRelPath);
        this.buckets.addAll(buckets);
    }
    
    /**
     * Adds a predefined bucket to this extractor.
     *
     * @param definition the bucket definition to add.
     */
    public void addPredefinedBucket(PredefinedBucket bucket) {
        buckets.add(bucket);
    }

    protected void handleValue(Value value) throws RepositoryException {
        for (PredefinedBucket bucket : buckets) {
            bucket.acceptValue(value);
        }
    }

    public Facet getFacet() {
        if (facet == null) {
            if (buckets == null) {
                return null;
            }
            // remove empty buckets
            for (Iterator<PredefinedBucket> it = buckets.iterator(); it.hasNext(); ) {
                if (it.next().getCount() == 0) {
                    it.remove();
                }
            }
            facet = new FacetImpl(buckets);
        }
        return facet;
    }

    @Override
    protected List<Value> filter(List<Value> values, ValueFactory vf)
            throws RepositoryException {
        // no filtering at all
        return values;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if ((obj == null) || (obj.getClass() != this.getClass())) {
            return false;
        }
        
        PredefinedBucketsFacetExtractor other = (PredefinedBucketsFacetExtractor) obj;
        if (propertyRelPath != other.propertyRelPath && !propertyRelPath.equals(other.propertyRelPath)) {
            return false;
        }
        if (buckets != other.buckets && !buckets.equals(other.buckets)) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + (propertyRelPath == null ? 0 : propertyRelPath.hashCode());
        hash = 31 * hash + (buckets == null ? 0 : buckets.hashCode());
        return hash;
    }
}
