/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import org.xml.sax.Attributes;

import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.EntryTagHandler;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * This tag handler is used to pull out the inline page scripts defined within
 * the script tag(s) of the input HTML document.
 *
 * <p>
 * This tag handler is registered against the markup of type < script >. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.InlineScriptTagHandlerFactory
 */
public class InlineScriptTagHandler implements TagHandler, EntryTagHandler, HTMLContentProvider {

    private boolean isScriptPositionSpecific = false;

    private StringBuffer scriptBuffer = new StringBuffer();

    public void beginHandling(String uri, String localName, String qName, Attributes atts) {

        String posSpecific = atts.getValue("data-cq-inline");
        if (posSpecific != null && posSpecific.equalsIgnoreCase("true")) {
            isScriptPositionSpecific = true;
        }
    }

    public void characters(char[] ch, int start, int length) throws DesignImportException {
        String chars = new String(ch).substring(start, start + length);
        if (chars.length() > 0) {
            scriptBuffer.append(chars);
        }
    }

    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        // Invalid. Script tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public void endHandling(String uri, String localName, String qName) {

    }

    public Object getContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.SCRIPT_INLINE) return scriptBuffer.toString();
        if (htmlContentType == HTMLContentType.MARKUP) return "<script type='text/javascript'>" + scriptBuffer.toString() + "</script>";

        return null;
    }

    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
    }

    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        // Invalid. Script tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public boolean supportsContent(HTMLContentType htmlContentType) {

        if (htmlContentType == HTMLContentType.SCRIPT_INLINE)
            return !isScriptPositionSpecific;
        else if (htmlContentType == HTMLContentType.MARKUP)
            return isScriptPositionSpecific;
        else
            return false;
    }
}
