/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.tags;

import java.io.IOException;
import java.text.Format;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspTagException;
import javax.servlet.jsp.tagext.BodyTagSupport;

import com.adobe.granite.xss.XSSAPI;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.scripting.SlingBindings;
import org.apache.sling.scripting.jsp.util.TagUtil;

import com.day.cq.commons.DiffInfo;
import com.day.cq.commons.DiffService;
import com.day.cq.wcm.api.WCMMode;

/**
 * <code>TextTag</code>...
 */
public class TextTag extends BodyTagSupport {

    public static final String DEFAULT_PLACEHOLDER = "<div><span class=\"cq-text-placeholder-ipe\">&para;</span></div>";

    private Object value;

    private Object oldValue;

    private String def;

    private String placeholder;

    private String propertyName;

    private Format fmt;

    private boolean escapeXml;

    private boolean noDiff;

    private String output;

    private String tagClass;

    private String tagName;

    private static String PROPERTY_TEXT_IS_RICH = "textIsRich";
    private XSSAPI xssAPI;
    private boolean textIsRich;
    private boolean enforceFilterHTML;

    private String[] ALLOWED_TAGS = {"abbr", "acronym", "address", "aricle", "aside", "b", "blockquote", "caption", "cite", "code", "details", "dfn", "dialog", "div", "em", "footer", "h1", "h2", "h3", "h4", "h5", "h6", "i", "kbd", "label", "legend", "main", "mark", "output", "p", "pre", "q", "rp", "rt", "ruby", "s", "samp", "section", "small", "span", "strike", "strong", "sub", "summary", "sup", "u", "var"};
    private String DIV_TAG = "div";

    private void init() {
        value = null;
        oldValue = null;
        def = null;
        placeholder = DEFAULT_PLACEHOLDER;
        propertyName = null;
        noDiff = false;
        fmt = null;
        output = null;
        escapeXml = false;
        enforceFilterHTML = false;
        tagName = null;
        tagClass = null;
    }

    @Override
    public void release() {
        super.release();
        init();
    }

    @Override
    public int doStartTag() throws JspException {

        // clean-up body (just in case container is pooling tag handlers)
        this.bodyContent = null;
        this.output = null;
        if (placeholder == null) {
            placeholder = DEFAULT_PLACEHOLDER;
        }

        // output value if not null
        if (value != null) {
            output = String.valueOf(value);
        }
        Resource resource = TagUtil.getRequest(pageContext).getResource();
        if (output == null && propertyName != null) {
            output = ResourceUtil.getValueMap(resource).get(propertyName, String.class);
        }

        xssAPI = TagUtil.getRequest(pageContext).getResourceResolver().adaptTo(XSSAPI.class);
        textIsRich = ResourceUtil.getValueMap(TagUtil.getRequest(pageContext).getResource()).get(PROPERTY_TEXT_IS_RICH, false);

        // if rich text flag is enforced by setting the enforceTextIsRich attribute in the tag.
        if (enforceFilterHTML) {
            textIsRich = true;
        }

        if (resource != null && !noDiff) {
            DiffInfo diffInfo = resource.adaptTo(DiffInfo.class);
            String diffOutput = null;
            if (diffInfo != null) {
                SlingBindings bindings = (SlingBindings) pageContext.getRequest().getAttribute(SlingBindings.class.getName());
                DiffService diffService = bindings.getSling().getService(DiffService.class);
                String oriText = null;
                if (oldValue != null) {
                    oriText = String.valueOf(oldValue);
                } else if (propertyName != null) {
                    ValueMap map = ResourceUtil.getValueMap(diffInfo.getContent());
                    oriText = map.get(propertyName, "");
                }

                String output_escaped = (escapeXml) ? escapeXSS(output) : output;
                String oritext_escaped = (escapeXml) ? escapeXSS(oriText) : oriText;
                diffOutput = diffInfo.getDiffOutput(diffService, output_escaped, oritext_escaped, !escapeXml);

            } else if (propertyName != null) {
                // handle the case when text component is not inside parsys
                String vLabel = TagUtil.getRequest(pageContext).getParameter(DiffService.REQUEST_PARAM_DIFF_TO);
                if (vLabel != null) {
                    Resource vRes = DiffInfo.getVersionedResource(resource, vLabel);
                    if (vRes != null) {
                        SlingBindings bindings = (SlingBindings) pageContext.getRequest()
                            .getAttribute(SlingBindings.class.getName());
                        DiffService diffService = bindings.getSling().getService(DiffService.class);
                        ValueMap map = ResourceUtil.getValueMap(vRes);
                        String oriText = map.get(propertyName, "");
                        String output_escaped = (escapeXml) ? escapeXSS(output) : output;
                        String oritext_escaped = (escapeXml) ? escapeXSS(oriText) : oriText;
                        diffOutput = diffService.diff(output_escaped, oritext_escaped, false);
                    }
                }
            }
            if (diffOutput != null) {
                output = diffOutput;
                // force html output
                escapeXml = false;
            }
        }

        if (output != null) {
            return SKIP_BODY;
        }

        // TODO: to avoid buffering, can we wrap out in a filter that performs
        // TODO: escaping and use EVAL_BODY_INCLUDE?
        return EVAL_BODY_BUFFERED;
    }

    @Override
    public int doAfterBody() throws JspException {
        output = bodyContent.getString().trim();
        return SKIP_BODY;
    }

    @Override
    public int doEndTag() throws JspException {
        try {
            if (output != null && fmt != null) {
                output = fmt.format(output);
            }
            if (output != null && escapeXml) {
                output = escapeXSS(output);
            }
            if (output == null || (output.length() == 0)) {
                if (WCMMode.fromRequest(pageContext.getRequest()) == WCMMode.EDIT) {
                    output = placeholder;
                } else {
                    output = def;
                }
            }
            if (output != null && output.length() > 0) {
                if (tagClass != null || tagName != null) {
                    if (tagName == null) {
                        tagName = "div";
                    }
                    pageContext.getOut().write("<");
                    pageContext.getOut().write(tagName);
                    if (tagClass != null && tagClass.length() > 0) {
                        pageContext.getOut().write(" class=\"");
                        pageContext.getOut().write(tagClass);
                        pageContext.getOut().write("\"");
                    }
                    pageContext.getOut().write(">");
                }
                pageContext.getOut().write(output);
                if (tagName != null) {
                    pageContext.getOut().write("</");
                    pageContext.getOut().write(tagName);
                    pageContext.getOut().write(">");
                }
            }
        } catch (IOException e) {
            throw new JspTagException(e);
        }
        return EVAL_PAGE;
    }


    public void setValue(Object value) {
        this.value = value;
    }

    public void setDefault(String def) {
        this.def = def;
    }

    public void setProperty(String propertyName) {
        this.propertyName = propertyName;
    }

    public void setFormat(Format fmt) {
        this.fmt = fmt;
    }

    public void setEscapeXml(boolean escapeXml) {
        this.escapeXml = escapeXml;
    }

    public void setEnforceFilterHTML(boolean enforceFilterHTML) {
        this.enforceFilterHTML = enforceFilterHTML;
    }

    public void setNoDiff(boolean noDiff) {
        this.noDiff = noDiff;
    }

    public void setPlaceholder(String def) {
        if(xssAPI == null){
            xssAPI = TagUtil.getRequest(pageContext).getResourceResolver().adaptTo(XSSAPI.class);
        }
        this.placeholder = xssAPI.filterHTML(def);
    }

    public void setOldValue(Object oldValue) {
        this.oldValue = oldValue;
    }

    public void setTagClass(String tagClass) {
        if(xssAPI == null){
            xssAPI = TagUtil.getRequest(pageContext).getResourceResolver().adaptTo(XSSAPI.class);
        }
        this.tagClass = xssAPI.encodeForHTMLAttr(tagClass);
    }

    public void setTagName(String tagName) {
        for (String element:ALLOWED_TAGS) {
            if (element.equals(tagName.toLowerCase())) {
                this.tagName =  tagName;
                return;
            }
        }
        this.tagName = DIV_TAG;
    }

    private String escapeXSS(String text) {
        if(textIsRich) {
            return xssAPI.filterHTML(text);
        } else {
            return xssAPI.encodeForHTML(text);
        }
    }
}
