/*
*
*	File: Cff.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2006 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.ROS;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnderlineMetrics;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.cff.CIDKeyedFont;
import com.adobe.fontengine.font.cff.CFFFont;
import com.adobe.fontengine.font.cff.FontFactory;
import com.adobe.fontengine.font.cff.NameKeyedFont;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to a 'CFF' table. 
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'CFF' tables have major/minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 1.0 tables</li>
 * <li>interprets version 1.x wtables as version 1.0 tables</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 */

final public class Cff extends Table {
  
  protected final CFFFont cffFont;
  
  protected Cff (FontByteArray buffer) 
  throws IOException, InvalidFontException, UnsupportedFontException {

	/* Do NOT call the super constructor that takes the stream.
	 * We never need the data in a Table form and it just causes us to keep
	 * two copies of it around.
	 */
    //super (in, 0);
	super();
    
    CFFFont[] fonts = FontFactory.load (buffer);
    if (fonts == null || fonts.length != 1 || fonts [0] == null) {
      throw new InvalidFontException ("CFFFontSet not appropriate for CFF table"); }
    cffFont = fonts [0];
  }
  
  public CFFFont getCFFFont () {
    return cffFont;
  }
  
  String getGlyphName (int gid) 
  throws InvalidFontException, UnsupportedFontException { 
    return cffFont.getGlyphName (gid);
  }
  
  ROS getROS () {
    return cffFont.getROS();
  }
  
  int getGlyphCid(int glyphID)
  throws InvalidFontException, UnsupportedFontException {
    return cffFont.getGlyphCid(glyphID);
  }
  
  /** Get the bounding box for a glyph. */
  public Rect getGlyphBoundingBoxInMetricSpace (int gid) 
  throws UnsupportedFontException, InvalidFontException {
    return cffFont.getGlyphBBox (gid);
  }

  public void getOutline (int gid, OutlineConsumer consumer) 
  throws UnsupportedFontException, InvalidFontException {
    cffFont.getGlyphOutline (gid, consumer);
  }

 
  public Rect getFontBBox () 
  throws UnsupportedFontException, InvalidFontException {
    return cffFont.getFontBBox ();
  }
  
  
  Permission getEmbeddingPermission (boolean wasEmbedded)
  throws UnsupportedFontException, InvalidFontException {
    return cffFont.getEmbeddingPermission(wasEmbedded);
  }
  
  protected UnderlineMetrics getCoolTypeUnderlineMetrics (double coolTypeUnitsPerEm, double unitsPerEmY)
  throws UnsupportedFontException, InvalidFontException {
    return cffFont.getCoolTypeUnderlineMetrics (coolTypeUnitsPerEm, unitsPerEmY); 
  }

  //----------------------------------------------- subsetting and streaming ---
  
  void subsetAndStream (Subset subset, OutputStream os, boolean preserveROS, Integer fsType) 
  throws InvalidFontException, UnsupportedFontException, IOException {
    cffFont.subsetAndStream (subset, os, preserveROS, fsType);
  }
  
  void subsetAndStream(Subset subset, Map tables, boolean preserveROS, Integer fsType, boolean enableSubrizer)
	  throws InvalidFontException, UnsupportedFontException, IOException
  {
	  OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor stream = new OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor();
	  if (cffFont instanceof NameKeyedFont) {
		  ((NameKeyedFont)cffFont).subsetAndStream(subset, stream, preserveROS, fsType, enableSubrizer);
	  } else if (cffFont instanceof CIDKeyedFont) {
		  ((CIDKeyedFont)cffFont).subsetAndStream(subset, stream, preserveROS, fsType, enableSubrizer);
	  } else {
		  throw new UnsupportedFontException ("subsetAndStream of unsupported CFF font");
	  }
	  tables.put(new Integer(Tag.table_CFF), stream.getBuilder());
  }
  
  void stream(Map tables, Integer fsType) throws InvalidFontException, UnsupportedFontException, IOException
  {
	  OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor stream = new OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor();
	  cffFont.stream(stream, fsType);
	  tables.put(new Integer(Tag.table_CFF), stream.getBuilder());
	  
  }
}
