/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.auth.oauth;

import java.util.Map;

import org.apache.jackrabbit.api.security.user.User;

/**
 * This interface is intended to be implemented by an application which wants to customize the behavior of a
 * {@link Provider} implementation. A ProviderExtension instance is bind to all specific {@link Provider}s supporting
 * the provider extension and for which Provider#getId equals ProviderExtension#getId.
 * Each {@link Provider} can be bind to either 0 or 1 ProviderExtension at any given time and in case the condition
 * above holds for more than one ProviderExtension, then the instance with the highest service ranking is selected.
 */
public interface ProviderExtension {

    /**
     * Unique ID for this provider extension, used to bind a {@link Provider} with the current ProviderExtension instance.
     * @return the provider extension identifier
     */
    public String getId();

    /**
     * Map the provider's user identifier to a unique CRX user identifier.
     * @see Provider#mapUserId(String, java.util.Map)
     * @param userId provider's user identifier
     * @param props map of all provider's properties for the user identifier
     * @return the user identifier or <code>null</code> in order to leave the implementation to the referenced {@link Provider}
     */
    public String mapUserId(String userId, Map<String, Object> props);

    /**
     * Return the node path where the user should be created
     * @param userId provider's user identifier
     * @param clientId client identifier
     * @param props map of all provider's properties for this user identifier
     * @return relative path to store this user within /home/users or <code>null</code> in order to leave the implementation to the referenced {@link Provider}
     */
    public String getUserFolderPath(String userId, String clientId, Map<String, Object> props);

    /**
     * Called after the {@link Provider#onUserUpdate(org.apache.jackrabbit.api.security.user.User)} of the referenced
     * {@link Provider} instance is called.
     * @see Provider#onUserUpdate(org.apache.jackrabbit.api.security.user.User)
     * @param user the user which has been updated
     */
    public void onUserUpdate(User user);

    /**
     * Called after the {@link Provider#onUserCreate(org.apache.jackrabbit.api.security.user.User)} of the referenced
     * {@link Provider} instance is called.
     * @param user the user which has been created
     */
    public void onUserCreate(User user);

}
