/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.haf.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * This annotation can be used on a public field or method. The return type must be a string, an iterator of
 * strings or a collection that is iterable.
 */
@Target({ElementType.FIELD, ElementType.METHOD})
@Retention(RetentionPolicy.RUNTIME)
public @interface ApiLink {
    /**
     * The relation type that this link represents.
     * @return the relation type
     */
    String rel();

    /**
     * The scope in which the link should be serialized. If SCOPE.RESOURCE the link will be serialized for a resource
     * that is the main entity. If SCOPE.INLINE the link will be serialized for a resource that is a sub-entity of
     * another resource. By default the link will be serialized in both cases.
     * @return The scope in which this link should be serialized.
     */
    SCOPE scope() default SCOPE.RESOURCE;

    /**
     * Defines the scope in which the link will be serialized in.
     */
    enum SCOPE {
        /**
         * The link will only be serialized as part of a main entity.
         */
        RESOURCE,
        /**
         * The link will only be serialized as part of a sub-entity.
         */
        INLINE,
        /**
         * The link will be serialized as part of both a main and a sub-entity.
         */
        BOTH;

        /**
         * Determines if the current scope contains the passed in scope value. If the current scope is SCOPE.BOTH true
         * is always returned. Otherwise a direct comparison of the current scope and the passed one is done.
         * @param scope The scope value to test the current scope against.
         * @return True is the current scope contains the passed in scope, false otherwise.
         */
        public boolean contains(SCOPE scope) {
            if (this == SCOPE.BOTH) {
                return true;
            } else {
                return this == scope;
            }
        }
    }

    /**
     * Set this to true if you do not need this path mapped into the API space.
     * @return {@code true} if path is absolute, {@code false} otherwise
     */
    boolean absolute() default false;

    /**
     * The content type returned when following the link.
     * @return The content type
     */
    String contentType() default "";
}
