//----------------------------------------------------------------------
//
//	ADOBE CONFIDENTIAL
//	__________________
//
//		Copyright 1995 - 2004 Adobe Systems Incorporated.  All
//		Rights Reserved.
//
//		NOTICE:  All information contained herein is, and remains
//		the property of Adobe Systems Incorporated and its
//		suppliers, if any.	The intellectual and technical
//		concepts contained herein are proprietary to Adobe Systems
//		Incorporated and its suppliers and may be covered by U.S.
//		and Foreign Patents, patents in process, and are protected
//		by trade secret or copyright law.  Dissemination of this
//		information or reproduction of this material is strictly
//		forbidden unless prior written permission is obtained from
//		Adobe Systems Incorporated.
//
//----------------------------------------------------------------------

package com.adobe.xfa.gfx;

/**
 * The graphic text attribute object holds information about attribute
 * values specific for text drawing.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public class GFXTextAttr extends GFXAttr {
/**
 * enumeration UnderCode.
 * <p>
 * Low four bits are type (unknown, none, all or word).
 * High four bits are style (single, double, strikeout).
 * Combined by using bit-wise OR.
 */
	public final static int UNDER_TYPE = GFXDecorationInfo.DECORATE_TYPE;
	public final static int UNDER_UNKNOWN = GFXDecorationInfo.DECORATE_UNKNOWN;
	public final static int UNDER_NONE = GFXDecorationInfo.DECORATE_NONE;
	public final static int UNDER_ALL = GFXDecorationInfo.DECORATE_ALL;
	public final static int UNDER_WORD = GFXDecorationInfo.DECORATE_WORD;
	public final static int UNDER_STYLE = GFXDecorationInfo.DECORATE_COUNT;
	public final static int UNDER_SINGLE = GFXDecorationInfo.DECORATE_SINGLE;
	public final static int UNDER_DOUBLE = GFXDecorationInfo.DECORATE_DOUBLE;

/**
 * enumeration LineThroughCode.
 * <p>
 * Low four bits are type (unknown, none, all or word).
 * High four bits are style (single, double, strikeout).
 * Combined by using bit-wise OR.
 */
	public final static int LINETHROUGH_TYPE = GFXDecorationInfo.DECORATE_TYPE;
	public final static int LINETHROUGH_UNKNOWN = GFXDecorationInfo.DECORATE_UNKNOWN;
	public final static int LINETHROUGH_NONE = GFXDecorationInfo.DECORATE_NONE;
	public final static int LINETHROUGH_ALL = GFXDecorationInfo.DECORATE_ALL;
	public final static int LINETHROUGH_WORD = GFXDecorationInfo.DECORATE_WORD;
	public final static int LINETHROUGH_STYLE = GFXDecorationInfo.DECORATE_COUNT;
	public final static int LINETHROUGH_SINGLE = GFXDecorationInfo.DECORATE_SINGLE;
	public final static int LINETHROUGH_DOUBLE = GFXDecorationInfo.DECORATE_DOUBLE;

/**
 * enumeration OverCode.
 * <p>
 * Low four bits are type (unknown, none, all or word).
 * High four bits are style (single, double, strikeout).
 * Combined by using bit-wise OR.
 */
	public final static int OVER_TYPE = GFXDecorationInfo.DECORATE_TYPE;
	public final static int OVER_UNKNOWN = GFXDecorationInfo.DECORATE_UNKNOWN;
	public final static int OVER_NONE = GFXDecorationInfo.DECORATE_NONE;
	public final static int OVER_ALL = GFXDecorationInfo.DECORATE_ALL;
	public final static int OVER_WORD = GFXDecorationInfo.DECORATE_WORD;
	public final static int OVER_STYLE = GFXDecorationInfo.DECORATE_COUNT;
	public final static int OVER_SINGLE = GFXDecorationInfo.DECORATE_SINGLE;
	public final static int OVER_DOUBLE = GFXDecorationInfo.DECORATE_DOUBLE;

/**
 * enumeration StrikeoutCode.
 * <p>
 * Legacy enumeration for inconsistent strikeout handling
 */
	public final static int STRIKEOUT_UNKNOWN = LINETHROUGH_UNKNOWN;
	public final static int STRIKEOUT_NONE = LINETHROUGH_NONE;
	public final static int STRIKEOUT_SINGLE = (LINETHROUGH_SINGLE | LINETHROUGH_ALL);
	public final static int STRIKEOUT_DOUBLE = (LINETHROUGH_DOUBLE | LINETHROUGH_ALL);

	public final static GFXTextAttr defaultTextAttr = new GFXTextAttr (UNDER_NONE, STYLE_SOLID, GFXAttr.defaultShadeScale(), GFXColour.black(), GFXColour.white());

	private int mnUnderline; // underline specification.
	private int mnOverline; // overline specification.
	private int mnStrikeout;
	private GFXTextContext mpoContext;

/**
 * Default constructor.
 * <p>
 * Populates the graphic attribute with the following settings:
 * <pre>
 *	 Underline	  = none
 *	 Overline	  = none
 *	 Strikeout	  = none
 * </pre>
 */
	public GFXTextAttr () {
		mnUnderline = UNDER_NONE;
		mnOverline = OVER_NONE;
		mnStrikeout = STRIKEOUT_NONE;
	}

/**
 * Copy constructor.
 * <p>
 * Creates a new Text attribute with the same settings
 * as the source attribute.
 * @param oSource Text attribute to copy
 */
	public GFXTextAttr (GFXTextAttr oSource) {
		copyFrom (oSource);
	}

/**
 * Constructor.
 * <p>
 * Creates a new Text attribute with the specified settings.
 * @param nNewUnderline The underline setting for this attribute
 * @param nNewStyle The style setting for this attribute
 * @param lNewShade The shade setting for this attribute
 * @param oNewColour The foreground colour to be used
 * @param oNewColourBg The background colour to be used
 */
	public GFXTextAttr (int nNewUnderline, int nNewStyle, int lNewShade, GFXColour oNewColour, GFXColour oNewColourBg) {
		super (nNewStyle, lNewShade, oNewColour, oNewColourBg);
		mnUnderline = nNewUnderline;
		mnOverline = OVER_NONE;
		mnStrikeout = STRIKEOUT_NONE;
	}

/**
 * Constructor.
 * <p>
 * Creates a new Text attribute with the specified settings.
 * @param nNewUnderline The underline setting for this attribute
 * @param nNewStyle The style setting for this attribute
 * @param lNewShade The shade setting for this attribute
 * @param oNewColour The foreground colour to be used
 * @param oNewColourBg The background colour to be used
 * @param nNewStrikeout The strikeout setting for this attribute
 * @param nNewOverline The overline setting for this attribute
 */
	public GFXTextAttr (int nNewUnderline, int nNewStyle, int lNewShade, GFXColour oNewColour, GFXColour oNewColourBg, int nNewStrikeout, int nNewOverline) {
		this (nNewUnderline, nNewStyle, lNewShade, oNewColour, oNewColourBg);
		mnUnderline = nNewUnderline;
		mnStrikeout = nNewStrikeout;
	}

/**
 * Get a default Text Attribute.
 * <p>
 * This will return a Text Attribute with the following settings:
 * <pre>
 *	 Underline		   = none
 *	 Style			   = solid
 *	 Shade scale	   = maximum
 *	 Foreground colour = black
 *	 Background colour = white
 * </pre>
 * @return A Text Attribute with default settings
 */
	public static GFXTextAttr defaultText () {
		return defaultTextAttr;
	}

/**
 * Get the underline value for this attribute.
 * @return the underline value as a UnderCode enumeration
 */
	public int underline () {
		return mnUnderline;
	}

/**
 * Set the underline value for this attribute.
 * @param nNewUnderline New underline value
 */
	public void underline (int nNewUnderline) {
		mnUnderline = nNewUnderline;
	}

/**
 * Get the overline value for this attribute.
 * @return the overline value as a OverCode enumeration
 */
	public int overline () {
		return mnOverline;
	}

/**
 * Set the overline value for this attribute.
 * @param nNewOverline New overline value
 */
	public void overline (int nNewOverline) {
		mnOverline = nNewOverline;
	}

/**
 * Get the strikeout value for this attribute.
 * @return The strikeout value as a StrikeoutCode enumeration
 */
	public int strikeout () {
		return mnStrikeout;
	}

/**
 * Set the strikeout value for this attribute.
 * @param nNewStrikeout New strikeout value
 */
	public void strikeout (int nNewStrikeout) {
		mnStrikeout = nNewStrikeout;
	}

/**
 * Return the text context associated with this graphic text attribute
 * object.
 * @return Pointer to the text context.  A null pointer is returned if
 * the context has never been set.
 */
	public GFXTextContext textContext () {
		return mpoContext;
	}

/**
 * Set a new text context for this attribute object.
 * @param poTextContext - New text context to associate with this graphic
 * text attribute object.
 */
	public void textContext (GFXTextContext poTextContext) {
		mpoContext = poTextContext;
	}

/**
 * Equality comparison.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param oCompare Attribute object to compare against.
 * @return TRUE if all members are equal, FALSE otherwise.
 */
	public boolean equivalent (GFXTextAttr oCompare) {
		return super.equivalent (oCompare)
			&& (mnOverline == oCompare.mnOverline)
			&& (mnUnderline == oCompare.mnUnderline)
			&& (mnStrikeout == oCompare.mnStrikeout)
			&& GFXTextContext.match (mpoContext, oCompare.mpoContext);
	}

/**
 * Equality comparison operator.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param object Attribute object to compare against.
 * @return TRUE if the attribute objects are considered equal;
 * FALSE otherwise.
 * <p/>
 * This implementation of Object.equals does not follow standard Java semantics.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		// JavaPort: This implementation uses instanceof to implement equality
		// instead of delegating to the super implementation (which compares class).
		// This means that this implementation does not follow the Java standard of symmetry of equals.
		
		if (! (object instanceof GFXTextAttr)) {
			return false;
		}
		return equivalent ((GFXTextAttr) object);
	}

	public int hashCode() {
		int hash = 41;
		hash = (hash * 31) ^ super.hashCode();
		hash = (hash * 31) ^ mnOverline;
		hash = (hash * 31) ^ mnUnderline;
		hash = (hash * 31) ^ mnStrikeout;
		hash = (hash * 31) ^ mpoContext.hashCode();
		return hash;
	}

/**
 * Non equality comparison operator.
 * <p>
 * Compares on an attribute value basis. Two attributes are only
 * considered equal if their values compare for equality.
 * @param oCompare Attribute object to compare against.
 * @return TRUE if the attribute objects are considered not equal;
 * FALSE otherwise.
 */
	public boolean notEqual (Object oCompare) {
		return ! equals (oCompare);
	}

/**
 * Assigment operator.
 * <p>
 * Replace all attribute settings with those from the source object.
 * @param oSource Source attribute object to copy.
 */
	public void copyFrom (GFXTextAttr oSource) {
		if (this != oSource) {
			super.copyFrom (oSource);
			mnOverline = oSource.mnOverline;
			mnUnderline = oSource.mnUnderline;
			mnStrikeout = oSource.mnStrikeout;
			mpoContext = oSource.mpoContext;
		}
	}

/**
 * Extract text decoration into usable values.
 * <p>
 * Given an underline, line-through or overline value, extract the two
 * component values from it for further processing.
 * @param nDecoration Input text decoration value.
 * @return TRUE if there is any decoration; FALSE if no decoration.
 */
	public static GFXDecorationInfo extractDecoration (int nDecoration) {
		return GFXDecorationInfo.extractDecoration (nDecoration);
	}
}
