/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.replication;

import javax.jcr.Session;
import java.util.Calendar;
import java.util.Map;

import org.osgi.annotation.versioning.ProviderType;

/**
 * This represents a replication agent as defined by an {@link AgentConfig}.
 */
@ProviderType
public interface Agent {

    /**
     * Returns the id of the agent. the id is unique among the agents
     * @return the id of the agent
     */
    String getId();

    /**
     * checks if agent is active
     * @return <code>true</code> if the agent is active
     */
    boolean isEnabled();

    /**
     * checks if agent is valid
     * @return <code>true</code> if the agent is correctly configured
     */
    boolean isValid();

    /**
     * Checks if the agent is valid and throws an exception if not. the message
     * of the exception can be used as hint to the user.
     *
     * @throws IllegalArgumentException if the agent is not valid.
     */
    void checkValid();

    /**
     * Get the configuration.
     * @return configuration
     */
    AgentConfig getConfiguration();

    /**
     * Returns the replication log of this agent
     * @return the replication log.
     */
    ReplicationLog getLog();

    /**
     * Replicates the given content. if the content cannot be delivered,
     * it is queued.
     *
     * @param action the replication action
     * @param content The content for the replication or null if no content.
     * @param options the replication options
     * @throws ReplicationException if an error during replication occurred
     */
    void replicate(ReplicationAction action, ReplicationContent content, ReplicationOptions options)
            throws ReplicationException;

    /**
     * Replicates the given (bulk) content. if the content cannot be delivered,
     * it is queued.
     *
     * @param action the composite replication action
     * @param content The content for the replication or null if no content.
     * @param options the replication options
     * @throws ReplicationException if an error during replication occurred
     */
    void replicate(CompositeReplicationAction action, ReplicationContent content, ReplicationOptions options)
            throws ReplicationException;

    /**
     * Poll modified content from the agent that will be reverse replicated.
     *
     * @param action the replication action
     * @return the polled content or <code>null</code>
     * @throws ReplicationException if an error occurs
     */
    ReverseReplication[] poll(ReplicationAction action) throws ReplicationException;

    /**
     * Builds the content for the respective action using the configured
     * content builder. The returned content is acquired by this agent.
     * @param session jcr session
     * @param action action
     * @return the content or <code>null</code>
     * @throws ReplicationException if an error occurs
     */
    ReplicationContent buildContent(Session session, ReplicationAction action)
            throws ReplicationException;

    /**
     * Builds the content for the respective action using the configured
     * content builder. The returned content is acquired by this agent.
     * @param session jcr session
     * @param action action
     * @param  parameters additional parameters to be supplied to the underlying {@link ContentBuilder}
     * @return the content or <code>null</code>
     * @throws ReplicationException if an error occurs
     */
    ReplicationContent buildContent(Session session, ReplicationAction action, Map<String, Object> parameters)
            throws ReplicationException;

    /**
     * Creates and acquires an existing content given it's facade
     * @param facade the content facade
     * @return the content or <code>null</code>
     * @throws ReplicationException if an error occurs
     */
    ReplicationContent getContent(ReplicationContentFacade facade)
            throws ReplicationException;

    /**
     * Returns the replication queue.
     * @return the replication queue or <code>null</code> if the agents is not
     *         active or valid.
     */
    ReplicationQueue getQueue();

    /**
     * Indicates that this agent is used as cache invalidator. For example
     * as dispatcher flush agent.
     * @return <code>true</code> if this agent is used as a cache invalidator.
     */
    boolean isCacheInvalidator();

    /**
     * Defines the timeline to be used for the next reverse replication poll.
     * @param time the time
     */
    void setNextPollTimeline(Calendar time);

    /**
     * Returns the timeline used for the last reverse replication.
     * @return the timeline or <code>null</code>
     */
    Calendar getLastPollTimeline();

    /**
     * Checks if the agent is in maintenance mode
     * @return <code>true</code> if this agent is in maintenance mode
     */
    boolean isInMaintenanceMode();

    /**
     * Checks if the agent can build content for multiple paths/resources.
     * @return {@code true} if this agent is capable of building bulk content (for multiple paths), {@code false} otherwise.
     */
    boolean supportsBulkContentBuilding();

    /**
     * Builds bulk content for the respective composite action using the configured content builder. The returned
     * content is acquired by this agent.
     * @param session jcr session
     * @param action composite action
     * @param  parameters additional parameters to be supplied to the underlying {@link ContentBuilder}
     * @return the bulk content if {@code #supportsBulkContentBuilding == true} or <code>null</code>
     * @throws ReplicationException if an error occurs
     */
    ReplicationContent buildBulkContent(Session session, CompositeReplicationAction action, Map<String, Object> parameters)
            throws ReplicationException;
}
