/*
*
*	File: Gsub.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;
import java.util.TreeMap;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;
import com.adobe.fontengine.inlineformatting.AttributedRun;
import com.adobe.fontengine.inlineformatting.ElementAttribute;

/** Gives access to the 'GSUB' table. 
 * 
 * <h4>Applying features.</h4>
 * 
 * <p>To apply features, the client first resolves a feature tag to a
 * set of lookups, using the <code>resolveFeatureTag</code> method. Then,
 * it invokes the GSUB engine to apply those lookups to the glyph run,
 * using the <code>applyLookups</code> method. The client passes two objects
 * to <code>applyLookups</code>, implementing the <code>Shaper</code> 
 * and <code>Selector</code> interfaces respectively. The <code>Shaper</code>
 * object is used by the GSUB engine to access the glyphs as well as to 
 * report transformations to be done on the glyph run. The 
 * <code>Selector</code> object is used by the GSUB engine to learn if the 
 * feature should be applied at some positions. Thus, the <code>Shaper</code>
 * and <code>Selector</code> objects are the bulk of the interface between
 * the client and the GSUB engine.
 */

final public class Gsub extends LookupTable {
    
  protected Gsub (FontByteArray buffer)
  throws java.io.IOException, InvalidFontException, UnsupportedFontException  {
    super (buffer);
  }

  static public final ElementAttribute componentCountAttribute = new ElementAttribute ("componentCount");
  static public final ElementAttribute ligatureComponentAttribute = new ElementAttribute ("ligatureComponent");
  
  /** Return the offset of the script list. */
  protected int getScriptListOffset ()
  throws InvalidFontException {
    return this.data.getOffset (0, 4); 
  }

  /** Return the offset of the feature list. */
  protected int getFeatureListOffset ()
  throws InvalidFontException {
    return this.data.getOffset (0, 6);
  }

  /** Return the offset of the lookup list. */
  protected int getLookupListOffset ()
  throws InvalidFontException {
    return this.data.getOffset (0, 8);
  }
  
  /** Apply a lookup subtable at a position in the glyph run.
   * @param lookupType the type of the lookup
   * @param lookupFlag the value of lookupFlag for this lookup
   * @param stOffset the offset of the subtable
   * @param run the AttributedRun to process
   * @param start the first accessible element in the run
   * @param limit the element after the last accessible element in the run
   * @param curGlyph the glyph occurrence to process
   * @param selector to determine whether the lookup applies at some positions
   * @param gdef the GDEF table
   * @return the result the of applying the subtable
   */
  protected LookupResult applyLookupSubtable 
         (int lookupType, int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph, 
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    switch (lookupType) {
      case 1: { 
        return applySingleSubst (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 2: { 
        return applyMultipleSubst (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 3: { 
        return applyAlternateSubst (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 4: { 
        return applyLigatureSubst (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 5: { 
        return applyContextualSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 6: { 
        return applyChainingContextualSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 7: { 
        return applyExtensionSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      default: {
        throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}
  }


  protected LookupResult applySingleSubst 
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applySingleSubstFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      case 2: { 
        return applySingleSubstFormat2 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GSUB single subst format (" + format + ")"); }}
  }
  

  protected LookupResult applySingleSubstFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
   
    if (getCoverageIndex (run.elementAt (curGlyph), coverageOffset) == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, curGlyph)) {
      return lookupNotApplied; }
    
    int deltaGlyphId = this.data.getint16 (stOffset + 4);
    int newGlyphId = (run.elementAt (curGlyph) + deltaGlyphId) & 0xffff;
    run.replace (curGlyph, newGlyphId);
    
    return new LookupResult (true, curGlyph + 1, 0);
  }
  
  protected LookupResult applySingleSubstFormat2 
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci  == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, inPos)) {
      return lookupNotApplied; }
    
    int newGlyphId = this.data.getuint16 (stOffset + 6 + 2*ci);
    run.replace (inPos, newGlyphId);
    return new LookupResult (true, inPos + 1, 0);
  }
  
  protected LookupResult applyMultipleSubst
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyMultipleSubstFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GSUB multiple subst format (" + format + ")"); }}
  }


  protected LookupResult applyMultipleSubstFormat1
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {

    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, inPos)) {
      return lookupNotApplied; }
    
    int sequenceOffset = this.data.getOffset (stOffset, 6 + 2*ci);
    int glyphCount = this.data.getuint16 (sequenceOffset);
    int [] newGlyphIds = new int [glyphCount];
    for (int g = 0; g < glyphCount; g++) {
      newGlyphIds [g] = this.data.getuint16 (sequenceOffset + 2 +  2*g); }
    
    run.replace (inPos, newGlyphIds); 
    
    return new LookupResult (true, inPos + glyphCount, glyphCount - 1);
  }
  
  protected LookupResult applyAlternateSubst 
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyAlternateSubstFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GSUB alternate subst format (" + format + ")"); }}
  }

  protected LookupResult applyAlternateSubstFormat1 
         (int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph, 
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, inPos)) {
      return lookupNotApplied; }
    
    int alternateSetOffset = this.data.getOffset (stOffset, 6 + 2*ci);
    int alternateCount = this.data.getuint16 (alternateSetOffset);
    
    Integer alternate = (Integer) (run.getElementStyle (inPos, ElementAttribute.alternate));
    
    if (alternate != null) {
      int alternateVal = alternate.intValue ();
      if (0 < alternateVal && alternateVal <= alternateCount) {
        int newGlyphId = this.data.getuint16 (alternateSetOffset + 2 + 2*(alternateVal -1));
        run.replace (inPos, newGlyphId); }}
    
    return new LookupResult (true, inPos + 1, 0); 
  }

  protected LookupResult applyLigatureSubst
         (int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph, 
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyLigatureSubstFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GSUB ligature subst format (" + format + ")"); }}
  }
  
  
  protected LookupResult applyLigatureSubstFormat1 
         (int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph, 
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    int ligatureSetOffset = this.data.getOffset (stOffset, 6 + 2*ci);
    int ligatureCount = this.data.getuint16 (ligatureSetOffset);
    
    for (int l = 0; l < ligatureCount; l++) {
      int ligatureOffset = this.data.getOffset (ligatureSetOffset, 2 + 2*l);
      
      int [] matchedPositions = matchOneLigature (lookupFlag, ligatureOffset, run, inPos, limit, selector, gdef);
      
      if (   matchedPositions != noMatch && selector.isApplied (run, matchedPositions)) {

        int componentCount = 0;
        for (int b = 0; b < matchedPositions.length; b++) {
          int m = matchedPositions [b] + 1;
          int upTo = b < matchedPositions.length - 1 ? matchedPositions [b + 1] : limit;
          while (m < upTo && gdef != null && gdef.getGlyphClass (run.elementAt (m)) == 3) {
            { Integer y = (Integer) run.getElementStyle (m, ligatureComponentAttribute);
              int yy = componentCount + (y == null ? 0 : y.intValue ());
              run.setElementStyle (m, ligatureComponentAttribute, new Integer (yy)); }            
            m++; }
          { Integer x = (Integer) run.getElementStyle (matchedPositions [b],
                                                       componentCountAttribute);
            componentCount += x == null ? 1 : x.intValue (); }} 

        int nextPosition = matchedPositions [matchedPositions.length - 1] + 1;
        int newGlyphId = this.data.getuint16 (ligatureOffset);
        run.replace (matchedPositions, newGlyphId);
        run.setElementStyle (matchedPositions [0], componentCountAttribute, new Integer (componentCount));
        nextPosition = nextPosition - matchedPositions.length + 1;
        return new LookupResult (true, nextPosition, -(matchedPositions.length -1)); }}
    
    return lookupNotApplied; 
  }
  
  /** Match one Ligature in a ligature subtable. */
  int[] matchOneLigature (int lookupFlag, int ligatureOffset, 
                          AttributedRun run, int inPos, int limit, 
                          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int compCount = this.data.getuint16 (ligatureOffset + 2);
    int [] matchedPositions = new int [compCount];
    matchedPositions [0] = inPos;
    inPos++;
    
    for (int i = 1; i < compCount; i++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (inPos >= limit || run.elementAt (inPos) != this.data.getuint16 (ligatureOffset + 4 + 2*(i-1))) {
        return noMatch; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    return matchedPositions;
  }
  
  void subsetAndStream(Subset subset, TreeMap lookups, Map tables, int numGlyphs) 
  throws InvalidFontException, UnsupportedFontException
  {
	  LookupTableSubsetter subsetter = new GsubSubsetter(this, numGlyphs);
	  OTByteArrayBuilder newData = subsetter.subsetAndStream(lookups, subset);
	  tables.put(new Integer(Tag.table_GSUB), newData);
	  
  }
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_GSUB), newData);
  }
}
