/*
*
*	File: Head.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'head' table.
 *
 * <h4>Version handling</h4>
 * 
 * <p>'head' tables have a major/minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 1.0 tables,</li>
 * <li>interprets 1.x tables as 1.0 tables,</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Head extends Table {

  private static boolean fontHeaderValidationRequired = true; //default value set to true
  
  public static boolean isFontHeaderValidationRequired() {
	return fontHeaderValidationRequired;
  }

  public static void setFontHeaderValidationRequired(
		boolean fontHeaderValidationRequired) {
	Head.fontHeaderValidationRequired = fontHeaderValidationRequired;
  }

  protected Head (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    
	  super (buffer);

	  if(isFontHeaderValidationRequired()){  // We won't check for the font header attributes when PDFA conversion/validation takes place
		  if (buffer.getSize() < 46) {
			  throw new InvalidFontException ("'head' table must be at least 46 bytes"); }

   
		  int majorVersion = getTableMajorVersion ();
		  if (majorVersion != 1) {
			  throw new UnsupportedFontException ("'head' tables with major version " 
                        						+ majorVersion + " are not supported"); 
	      }
      }
  }

  /** Get the major version. */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
  
  /** Get the minor version. */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }
  
  /** Compute the checksum of the table. 
 * @throws InvalidFontException */
  protected long checksum () 
  throws InvalidFontException 
  {
    // To compute the checksum of a head table, we need to skip 
    // the checkSumAdjustment field.
    return checksum (12, this.data.getSize(), checksum (0, 8, 0L));
  }
  
  /** Get the flags bit 12: "Font converted (produce compatible metrics)"
   */
  public boolean isConverted () throws InvalidFontException {
    return (this.data.getuint16 (16) & 0x1000) != 0;
  }
  
  /** Get the units per em. */
  public int getUnitsPerEm () throws InvalidFontException {
    return this.data.getuint16 (18);
  }
  
  /** Get the indexToLocFormat. */
  public int getIndexToLocFormat () throws InvalidFontException {
    return this.data.getint16 (50);
  }
  
  /** Get the italic macStyle. */
  public boolean isItalic () throws InvalidFontException {
    return (this.data.getuint16 (44) & 0x2) != 0;
  }
  
  /** Get the xMin. */
  public int getXMin () throws InvalidFontException {
    return this.data.getint16 (36);
  }

  /** Get the yMin. */
  public int getYMin () throws InvalidFontException {
    return this.data.getint16 (38);
  }

  /** Get the xMax. */
  public int getXMax () throws InvalidFontException {
    return this.data.getint16 (40);
  }

  /** Get the yMax. */
  public int getYMax () throws InvalidFontException {
    return this.data.getint16 (42);
  }
  
  /** Get the font bbox. */
  public Rect getFontBBox () throws InvalidFontException {
    return new Rect (getXMin (), getYMin (), getXMax (), getYMax ());
  }

  //----------------------------------------------- subsetting and streaming ---
  
  public OTByteArrayBuilder subsetAndStream (Subset subset, Map tables) {
    OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(this.data);
    newData.setint16 (50, 1);      
    tables.put (new Integer (Tag.table_head), newData); 
    return newData;
  }  
  
  public OTByteArrayBuilder stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_head), newData);
	  return newData;
  }
  
  static final void clearChecksumAdjust (OTByteArrayBuilder data) {
    data.setuint32 (8, 0);
  }
  
  static final void setChecksumAdjust (OTByteArrayBuilder data, long l) {
    data.setuint32 (8, (int)l);
  }
}
