/*
 *
 *	File: AFM.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.type1;

import java.io.IOException;
import java.net.URL;
import java.util.Comparator;
import java.util.Arrays;

import com.adobe.fontengine.font.postscript.Token;
import com.adobe.fontengine.font.postscript.Tokenizer;
import com.adobe.fontengine.font.postscript.TokenType;
import com.adobe.fontengine.font.type1.MetricFile;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.InvalidFontException;

/**
 * This class is an in-memory representation of an AFM.
 *
 * <h4>Synchronization</h4>
 * 
 * This class is immutable after construction and contains no mutable
 * static data. It is therefore threadsafe.
 */
final class AFM extends MetricFile{
   private String fontName;
   private static final byte[] kFontName = {'F','o','n','t','N','a','m','e'};
   
   private static final byte[][] kOptionalTokens = {{'S','t','a','r','t','K','e','r','n','P','a','i','r','s'},
           {'S','t','a','r','t','K','e','r','n','P','a','i','r','s', '0'}
   };
   
   private static final byte[][] kKernPairStart = {{'K','P'}, {'K','P','X'}};
   
   private final KernPair[] kernPairs;
   
   private static class KernPair {
       final String leftGName;
       final String rightGName;
       final double kernAmount;
       
       KernPair(String left, String right, double kernAmount)
       {
           this.leftGName = left;
           this.rightGName = right;
           this.kernAmount = kernAmount;
       }
   }
   
   private static final Comparator kpComp  = new Comparator () {
       public int compare(Object left, Object right)
       {
           KernPair leftKP = (KernPair)left;
           KernPair rightKP = (KernPair)right;
           
           int compRes = leftKP.leftGName.compareTo(rightKP.leftGName);
           if (compRes < 0)
               return -1;
           else if (compRes > 0)
               return 1;
           
           compRes = leftKP.rightGName.compareTo(rightKP.rightGName);
           if (compRes < 0)
               return -1;
           else if (compRes > 0)
               return 1;
           
           return 0;
       }
    };
   
        
    public double getKernValue(String leftGlyphName, String rightGlyphName)
    {       
        KernPair thisPair = new KernPair(leftGlyphName, rightGlyphName, 0);
        
        int res = Arrays.binarySearch(kernPairs, thisPair, kpComp);
        if (res >= 0)
            return kernPairs[res].kernAmount;
        return 0;
    }
    
    public String getFamilyName()
    {
        // The afm doesn't contain the face name
        return null;
    }
    
    public String getFontName()
    {
        return fontName;
    }
    
    public int getWeight()
    {
        // the AFM doesn't contain the weight.
        return 0;
    }

    /**
     * @param fontName
     * @param kernPairs a non-null (but may be empty) array of kern pairs.
     * @param url
     * @throws IOException
     * @throws InvalidFontException
     */
    private AFM(String fontName, KernPair[] kernPairs, URL url)
    throws IOException, InvalidFontException
    {
        super(url);
        this.fontName = fontName;
        Arrays.sort(kernPairs, kpComp);
        this.kernPairs = kernPairs;
 
    }
    
    static KernPair[] parseKernPairs(Tokenizer tokenizer, int numKernPairs)
	throws IOException, InvalidFontException
    {
        String leftGName, rightGName;
        double kernVal;
        Token token;
        
        KernPair[] kernPairs = new KernPair[numKernPairs];
        
        for (int i = 0; i < numKernPairs; i++)
        {
            byte[] tokenFound = tokenizer.findOptionalTokensAtStartOfLine(kKernPairStart);
            
            if (tokenFound == null)
            {
                return new KernPair [0];
            }
            
            // both KP and KPX start with the names followed by the x-kerning.
            token = tokenizer.getNextPSToken();
            if (token.tokenType == TokenType.kOPERATOR) // these look like operators
            {
                leftGName = token.stringTokenToString(0, token.tokenLength);
            } else
            {
                return  new KernPair [0];
            }
            
            token = tokenizer.getNextPSToken();
            if (token.tokenType == TokenType.kOPERATOR) // these look like operators
            {
                rightGName = token.stringTokenToString(0, token.tokenLength);
            } else
            {
                return  new KernPair [0];
            }
            
            token = tokenizer.getNextPSToken();
            if (token.tokenType == TokenType.kINTEGER)
                kernVal = token.convertInteger( 0);
            else if (token.tokenType == TokenType.kREAL)
                kernVal = Double.parseDouble(new String(token.buff, 0, token.tokenLength));
            else
            {
                return  new KernPair [0];
            }
            
            kernPairs[i] = new KernPair(leftGName, rightGName, kernVal);
        }
        
        return kernPairs;
    }
    
    /**
     * Determines if str is an AFM. If so, it creates an AFM and returns it
     * @param str The stream to be probed.
     * @return null if str is not an AFM. An AFM object otherwise.
     * @throws IOException thrown if the stream could not be read
     * @throws InvalidFontException thrown if the AFM is not properly formed
     */
    static MetricFile createAFM(FontInputStream str, URL url)
    	throws IOException, InvalidFontException
    {
        String fontName;
        KernPair[] kernPairs = null;
        Tokenizer tokenizer = new Tokenizer(str);
        Token token;
        tokenizer.findToken(kFontName);
        token = tokenizer.getNextPSToken();
        
        char[] chars = new char [token.tokenLength];
        for (int i = 0; i < token.tokenLength; i++) 
        {
          chars [i] = (char) token.buff[i]; 
        }
        
        fontName = new String (chars);
        byte[] foundToken = tokenizer.findOptionalTokensAtStartOfLine(kOptionalTokens);
        if (foundToken != null)
        {
	        token = tokenizer.getNextPSToken();
	        if (token.tokenType == TokenType.kINTEGER)
	        {
	            int numKernPairs = token.convertInteger(0);
	            if (numKernPairs > 0)
	            {
	                kernPairs = parseKernPairs(tokenizer, numKernPairs);
	            }
	        }

        }
        
        if (kernPairs == null) {
          kernPairs = new KernPair [0]; }
        
        return new AFM(fontName, kernPairs, url);
    }
}
