/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import java.io.IOException;
import java.io.OutputStream;


/**
 * A class to represent the children of the XFA app model 
 * that are not otherwise represented by a model.
 */
public final class Packet extends Element implements Element.DualDomNode {
	
	private Element mXmlPeer;

	/**
	 * Instantiates a packet node.
	 *
	 * @exclude from published api.
	 */
	public Packet(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, XFA.PACKET, XFA.PACKET, null,
				XFA.PACKETTAG, XFA.PACKET);

		if (parent != null) {
			
			// If a Packet has a parent, it must be an AppModel
			if (!(parent instanceof AppModel))
				throw new IllegalArgumentException("parent is not an AppModel.");
			
			setDocument(parent.getOwnerDocument());
			setModel((AppModel)parent);
		}
	}

	/**
	 * XFAPackets have no formal schema, so getProperty does not work to return
	 * attributes. Instead, provide an accessor to return arbitrary attribute
	 * values. 
	 * @param sAttrName The name of the attribute to return 
	 * @return the attribute value, or the empty string if it doesn't exist
	 * or is empty.
	 *
	 * @exclude from published api.
	 */
	public String getAttribute(String sAttrName) {
		int oAttr = findAttr(null, sAttrName.intern());
		if (oAttr != -1)
			return getAttrVal(oAttr);

		return "";
	}

	/**
	 * Gets this packet's content.
	 * If this packet has content, return it. Note that for packets that
	 * contain XML content, this should return an empty string.
	 * 
	 * @return the packet content.
	 * If this packet has text content, the text is returned. Note that
	 * for packets that contain XML content, an empty string is returned.
	 */
	public String getContent() {
		Node oChild = getFirstXMLChild();
		while (oChild != null) {
			if (oChild instanceof Chars) {
				Chars oText = (Chars) oChild;
				return oText.getText();
			}
			oChild = oChild.getNextXMLSibling();
		}
		return "";
	}

	/**
	 * Gets this packet's name.
	 */
	public String getName() {
		return getLocalName();
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return PacketScript.getScriptTable();
	}

	/**
	 * Determines if the given attribute tag is a valid attribute
	 * of this packet.
	 * @see Element#isValidAttr(int, boolean, String)
	 *
	 * @exclude from published api.
	 */
	public boolean isValidAttr(int eTag, boolean bReport, String value) {
		return true;
	}

	/**
	 * Gets this element's first child that is an XFA node.
	 * @return null -- the children of packet nodes are not exposed.
	 */
	public Node getFirstXFAChild() {
		// JavaPort: C++ version does not expose packet children.
		return null;
	}
	
	/**
	 * @throws IOException 
	 * @exclude
	 */
	public void serialize(OutputStream outStream, DOMSaveOptions options, int level, Node prevSibling) throws IOException, IOException {
		
		// Ensure that we serialize from the XML DOM side.
		getXmlPeer().serialize(outStream, options, level, prevSibling);
	}

	/**
	 * Removes the specified attribute.
	 * @param sAttrName the name of the attribute to remove.
	 *
	 * @exclude from published api.
	 */
	public void removeAttribute(String sAttrName) {
		String aAttrName = sAttrName.intern();
		super.removeAttr(null, aAttrName);
	}

	/**
	 * XFAPackets have no formal schema, so setProperty does not work to set
	 * attributes. Instead, this sets arbitrary attribute values. If the
	 * attribute exists, we populate it, if it doesn't exist we create it.
	 * @param sValue The value of the attribute to set
	 * @param sAttrName The name
	 * of the attribute to set
	 *
	 * @exclude from published api.
	 */
	public void setAttribute(String sValue, String sAttrName) {
		setAttribute(null, sAttrName, sAttrName, sValue);
		
		// Watson bug 1771483, make sure the doc is marked as dirty
		makeNonDefault(false);
	}

	/**
	 * Sets this packet's content.
	 * 
	 * @param sContent the text content value.
	 */
	public void setContent(String sContent) {

		// Remove all existing children
		Node oChild = getFirstXMLChild();
		while (oChild != null) {
			removeChild(oChild);
			oChild = oChild.getNextXMLSibling();
		}

		getModel().createTextNode(this, null, sContent);

	    // Watson bug 1771483, make sure the doc is marked as dirty
		makeNonDefault(false);
	}
	
	/**
	 * Override of Element.compareVersions.
	 * 
	 * @exclude from published api.
	 */
	protected boolean compareVersions(Node oRollbackElement, Node oContainer, Node.ChangeLogger oChangeLogger , Object oUserData) {
		//
		// XFA doesn't model packet content, so we need to use XML canonicalization here....
		//
		return compareVersionsCanonically(oRollbackElement, oContainer, oChangeLogger, oUserData);
	}
	
	/**
	 * @exclude from public api.
	 */
	public void setXmlPeer(Node peer) {
		mXmlPeer = (Element)peer;
	}

	/**
	 * @exclude from public api.
	 */
	public Node getXmlPeer() {
		return mXmlPeer;
	}
}
