/* ************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2021 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any.  The
 * intellectual and technical concepts contained herein are
 * proprietary to Adobe and its suppliers and are protected
 * by all applicable intellectual property laws, including
 * trade secret and copyright laws.  Dissemination of this
 * information or reproduction of this material is strictly
 * forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.cq.dam.api.smartcrop;

import org.osgi.annotation.versioning.ProviderType;

@ProviderType
public interface SmartCrop {
    /**
     * get the absolute pixel width
     * @return the pixel width of crop
     */
    public int getWidth();
    /**
     * get the absolute pixel height
     * @return the pixel height of crop
     */
    public int getHeight();
    
    /**
     *  Get a rectangle representing the crop boundary.
     *  Contains relative coordinates for the top left corner and relative width &amp; height
     *  @return  {@link NormalizedCropRect}
     */
    public NormalizedCropRect getNormalizedCropRect();

    /**
     * get the source of crop - assetcompute or dynamic media
     * @return {@link Source}
     */
    public Source getSource();
    
    /** the type of crop - banner or swatch
     *  @return {@link CropType}
     */
    public CropType getCropType();

    /**
     * get the name of the crop
     *  @return the name of the crop
     */
    public String getName();


    /**
     * interface for rectangle representing the normalized crop boundary.
     * left + width &lt; 1
     * top + height &lt; 1
     */
    interface NormalizedCropRect {
        /**
         * Get the relative horizontal postition of top left corner
         * @return relative x-position of top left corner for crop
         */
        public double getLeft();

        /**
         *
         * @return relative y-position of top left corner for crop
         */
        public double getTop();

        /**
         *
         * @return relative width of the crop
         */
        public double getWidth();

        /**
         *
         * @return relative height of crop
         */
        public double getHeight();
    }

    /**
     * Generation source for crop - dynamicmedia or assetcompute.
     */
    public enum Source {
        /**
         * Represents crop was generated by dynamic media
         */
        DM,
        /**
         * Represents crop was generated by assetcompute
         */
        NUI;

        /**
         * Parses string value to return corresponding enum value of {@link Source}, defaults to dynamic media
         * @param string value of source
         * @return {@link Source}
         */
        public static Source fromString(String value) {
            if (NUI.name().equalsIgnoreCase(value)) {
                return NUI;
            }
            else {
                return DM;
            }
        }
    }

    /**
     * Type of crop - banner or swatch.
     */
    public enum CropType {
        /**
         * Represents crop is a banner
         */
        BANNER,
        /**
         * Represents crop is swatch
         */
        SWATCH;

        /**
         * Parses string value to return corresponding enum value of {@link CropType}, defaults to banner
         * @param string value of croptype
         * @return {@link CropType}
         */
        public static CropType fromString(String value) {
            if (SWATCH.name().equalsIgnoreCase(value)) {
                return SWATCH;
            }
            else {
                return BANNER;
            }
        }
    }

}
