/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.reporting;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * This interface represents a column, representing the data of a report.
 */
public interface Column {

    // column definition -------------------------------------------------------------------

    /**
     * Gets the path of this instance of the column.
     *
     * @return The path of this column (for example
     *         "/etc/reports/areport/jcr:content/report/col1")
     */
    String getPath();

    /**
     * Gets the name of this instance of the column.
     *
     * @return The name of this column (usually last part of the column's path)
     */
    String getName();

    /**
     * Gets the title of this instance of the column.
     *
     * @return The title of the column
     */
    String getTitle();

    /**
     * Returns the column's data type.
     *
     * @return The column's data type
     */
    String getColumnType();

    /**
     * <p>Returns the column's actual type for the current report setting.</p>
     *
     * <p>Contrary to {@link #getColumnType()}, this method considers aggregation state.</p>
     *
     * <p>The returned value must be supported by the clientside reporting implementation.
     * See {@link com.day.cq.reporting.helpers.DisplayDataType} for a list of officially
     * supported display types. Note that there might be more supported types (which is why
     * a <code>String</code> is returned rather than a <code>DisplayDataType</code>).</p>
     *
     * @return The actual type
     */
    String getActualType();

    /**
     * Get the query engine-specific column definitions.
     *
     * @return The query engine-specific definition
     */
    QueryEngineConfig getColumnDef();

    /**
     * Checks if the column is currently grouped.
     *
     * @return <code>true</code> if the column is grouped
     */
    boolean isGrouped();

    /**
     * Returns the aggregate type defined for the column.
     *
     * @return The aggregate type defined for the column
     */
    String getAggregateType();

    /**
     * Returns the sorting defined for the column.
     *
     * @return The sorting defined for the column
     */
    Sorting getSorting();


    // processing --------------------------------------------------------------------------

    /**
     * Starts the postprocessing of data.
     */
    void startProcessing();

    /**
     * Ends the postprocessing of data.
     */
    void endProcessing();

    /**
     * <p>Preprocesses the specified data row.</p>
     *
     * <p>If required by the <code>type</code> parameter (for example for
     * {@link ProcessingPhase#APPLY}), the preprocessed data gets changed in the
     * <code>DataRow</code> specified.</p>
     *
     * @param type Type of preprocessing to be executed
     * @param row The row to be preprocessed
     * @param isAggregated <code>true</code> if the preprocessing is executed for
     *                     the aggregated column value (if the column actually represents
     *                     an aggregated value)
     * @param isReportGrouped <code>true</code> if the report as a whole is grouped
     * @return The preprocessed data
     */
    CellValue preprocessData(ProcessingPhase type, DataRow row, boolean isAggregated,
                             boolean isReportGrouped);

    /**
     * <p>Resolves the column's value in the specified row if applicable.</p>
     *
     * <p>The resolved value must be set in the row.</p>
     *
     * @param session JCR Session used for resolving
     * @param rowToResolve The row
     * @throws RepositoryException if resolving the value has failed due to a repository
     *                             problem
     */
    void resolve(Session session, DataRow rowToResolve) throws RepositoryException;

    /**
     * Checks if the specified {@link DataRow} matches the current filter settings
     *
     * @param phase The current filtering phase
     * @param rowToCheck The row to be checked against the filter
     * @return <code>true</code> if the row matches the filter and should be kept in the
     *         result set
     */
    boolean matchesFilter(FilteringPhase phase, DataRow rowToCheck);


    /**
     * Cleans up the column after using it.
     */
    void cleanup();

}
