/* ****************************************************************************
 *
 *	File: CosString.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.io.UnsupportedEncodingException;


import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASDate;
import com.adobe.internal.pdftoolkit.core.types.ASHexString;
import com.adobe.internal.pdftoolkit.core.types.ASString;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * Represents a COS string as defined in section 3.2.3 of the PDF Reference
 * Manual version 1.4.
 */
public class CosString extends CosScalar 
{
	private byte[] mValue;			// Decrypted and extracted string data
	private boolean mIsEncrypted;		// True if string content is already encrypted

	// Original data buffer
	//
	private byte[] mBase;			// Original encrypted data buffer
	private int mBegin;			// Starting offset into data buffer
	private int mLength;			// Number of bytes to extract in buffer
	private boolean mWriteHex;
	private boolean mOddBall;		// Ended on odd nibble if hex string
	private boolean mToEncrypt;		// Encrypt only when true
	private CosContainer mParentObj;	// Set to parent container

	/**
	 * Constructs a COS string.
	 *
	 * @param base			String data buffer
	 * @param begin			Starting offset into the buffer
	 * @param length		Number of bytes of string data in the buffer
	 * @param isEncrypted		True if the string data is encrypted
	 */
	private void init(byte[] base, int begin, int length,
			  boolean isEncrypted, boolean writeHex, boolean oddBall, boolean toEncrypt)
	{
		mValue = null;
		mBase = base;
		mBegin = begin;
		mLength = length;
		mIsEncrypted = isEncrypted;
		mWriteHex = writeHex;
		mOddBall = oddBall;
		mToEncrypt = toEncrypt;
	}

	/**
	 * Constructs a COS string.
	 *
	 * @param doc			Document that contains the string
	 * @param base			String data buffer
	 * @param begin			Starting offset into the buffer
	 * @param length		Number of bytes of string data in the buffer
	 * @param isEncrypted		True if the string data is encrypted
	 * @param info			Info for the string object
	 * @param writeHex		This was read as a hex string, so write it that way
	 */
	CosString(CosDocument doc, byte[] base, int begin, int length,
		  boolean isEncrypted, CosObjectInfo info, boolean writeHex)
	{
		super(doc, info);
		init(base, begin, length, isEncrypted, writeHex, false, true);
	}

	/**
	 * Constructs a COS string.
	 *
	 * @param doc			Document that contains the string
	 * @param base			String data buffer
	 * @param begin			Starting offset into the buffer
	 * @param length		Number of bytes of string data in the buffer
	 * @param isEncrypted		True if the string data is encrypted
	 * @param info			Info for the string object
	 * @param writeHex		This was read as a hex string, so write it that way
	 * @param oddBall		If read as hex string, had odd nibble
	 */
	CosString(CosDocument doc, byte[] base, int begin, int length,
		  boolean isEncrypted, CosObjectInfo info, boolean writeHex, boolean oddBall)
	{
		super(doc, info);
		init(base, begin, length, isEncrypted, writeHex, oddBall, true);
	}

	/**
	 * Constructs a COS string.
	 *
	 * @param doc			Document that contains the string
	 * @param base			String data buffer
	 * @param begin			Starting offset into the buffer
	 * @param length		Number of bytes of string data in the buffer
	 * @param isEncrypted		True if the string data is encrypted
	 * @param info			Info for the string object
	 */
	CosString(CosDocument doc, byte[] base, int begin, int length,
		  boolean isEncrypted, CosObjectInfo info)
	{
		super(doc, info);
		init(base, begin, length, isEncrypted, false, false, true);
	}

	/**
	 * @throws PDFSecurityException
	 */
	CosString copy()
		throws PDFSecurityException
	{
		if (isIndirect())
			return this;
		// byteArrayValue() always returns decrypted content which means that
		// isEncrypted flag should always be false.
		CosString copy = new CosString(getDocument(), byteArrayValue(), 0,
					       byteArrayValue().length, false, null, getWriteHex());
		copy.setToEncrypt(getToEncrypt());
		return copy;
	}

	/**
	 *
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_String;
	}

	/**
	 * Decrypts the string data, if necessary, and extracts the desired
	 * portion of the string buffer into the string value buffer. This
	 * method is called on demand by the getValue method.
	 * @param decrypt indicates that the string content should be decrypted if necessary
	 * @throws PDFSecurityException
	 * @see #getValue()
	 */
	private void initValue(boolean decrypt)
		throws PDFSecurityException
	{
		// Simple case of a string spanning the full buffer and not encrypted.
		if (mBegin == 0 && mLength == mBase.length)
			mValue = mBase;
		else {
			// Partial buffer string.
			mValue = new byte[mLength];
			System.arraycopy(mBase, mBegin, mValue, 0, mLength);
		}
		if (mIsEncrypted && decrypt)
			mValue = getDocument().getEncryption().decryptString(this, mValue);
	}

	/**
	 * Set this value if this string is being added to a container.
	 *
	 * @param parent - CosContainer
	 */
	void setParentObj(CosContainer parent)
	{
		mParentObj = parent;
	}

	/**
	 * Get parent's object info.  The value is initialized to null (the free
	 * root object) so if this value is not equal to null it has been set.
	 *
	 * @return parent's CosObjectInfo
	 */
	CosContainer getParentObj()
	{
		return mParentObj;
	}

	/**
	 * Obtains the string data. The first time the method is called, the
	 * string data is decrypted, if necessary, and the desired portion of
	 * the original data buffer is extracted.
	 *
	 * @return Byte array containing the string data.
	 * @throws PDFSecurityException
	 */
	public byte[] byteArrayValue()
	throws PDFSecurityException
	{
		return byteArrayValue(true);
	}

	/**
	 * Obtains the string data. The first time the method is called, the
	 * string data is decrypted, if necessary, and the desired portion of
	 * the original data buffer is extracted.
	 *
	 * @param decrypt indicates that the string content should be decrypted if necessary
	 * @return Byte array containing the string data.
	 * @throws PDFSecurityException
	 */
	private byte[] byteArrayValue(boolean decrypt)
		throws PDFSecurityException
	{
		if (mValue == null)
			initValue(decrypt);
		return mValue;
	}

	@Override
	public ASString stringValue()
		throws PDFSecurityException
	{
		return new ASString(byteArrayValue());
	}

	
	@Override
	public ASHexString hexStringValue()
		throws PDFSecurityException
	{
		return new ASHexString(byteArrayValue());
	}

	/**
	 * Returns String value.
	 * @throws PDFSecurityException
	 */
	public String asString()
		throws PDFSecurityException
	{
		return stringValue().asString();
	}

	
	@Override
	public Object getValue()
		throws PDFSecurityException
	{
		return byteArrayValue();
	}

	/**
	 * Creates an instance of a String object "wrapping" the specified
	 * CosString. This method is called by the Factory class. A Class
	 * implementing the Constructible interface must implement this method.
	 *
	 * @return Newly created String object or null if data is null.
	 * @throws PDFSecurityException
	 * @throws RuntimeException, hard failure, if basic encodings aren't supported
	 *  This represents a "can't happen" situation
	 */
	@Override
	public String textValue()
		throws PDFSecurityException
	{

		String rslt = "";
		byte[] src = byteArrayValue();

		// Determine if the CosString contains Unicode data as described
		// in section 3.8.1 of the PDF Reference Manual version 1.4.
		boolean isUnicode = src.length >= 2 && src[0] == -2 && src[1] == -1;
		try {
			rslt = new String(src, isUnicode ? "UTF-16" : "ISO-8859-1");
		}
		catch (UnsupportedEncodingException e) {
			throw new RuntimeException("Platform does not support encoding.", e);
		}
		return rslt;
	}

	/**
	 * Uses parent container's ObjectInfo for encryption if this is direct.
	 *
	 */
	@Override
	byte[] getObjectEncryptionKey(boolean write)
	{
		byte[] key = new byte[5];
		int objNum;
		if (isIndirect())
			objNum = getInfo().getObjNum();
		else
			objNum = mParentObj.getInfo().getObjNum();
		int gen;
		if (isIndirect())
			gen = getInfo().getObjGen();
		else
			gen = mParentObj.getInfo().getObjGen();
		if (!write) {
			CosLinearization cosLin = getDocument().getLinearization();
			if (cosLin != null) {
				objNum = cosLin.mapNewToOldObjNum(objNum);
				gen = cosLin.mapNewToOldObjGen(objNum);
			}
		}
		System.arraycopy(ByteOps.splitInt2Bytes(objNum, 3), 0, key, 0, 3);
		System.arraycopy(ByteOps.splitInt2Bytes(gen, 2), 0, key, 3, 2);
		return key;
	}

	/**
	 * Mark object as not dirty if it was dirty.
	 * Returns false if not dirty.
	 */
	@Override
	boolean markNotDirty()
	{
		if (isIndirect())
			return super.markNotDirty();
		if (mParentObj != null)
			return mParentObj.markNotDirty();
		return false;
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * Hack for DigSig.
	 * Overwrite string contents with passed byte array.
	 */
	public void setDataInternal(byte[] newData, boolean markDirty)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mValue = null;
		mBase = newData;
		mBegin = 0;
		mLength = newData.length;
		mIsEncrypted = false;
		if (markDirty) {
			try {
				if (isIndirect())
					getInfo().markDirty();
				else {
					if (mParentObj != null)
						mParentObj.getInfo().markDirty();
				}
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		}
	}

	private static final byte[] hexrep = {
		(byte)'0', (byte)'1', (byte)'2', (byte)'3',
		(byte)'4', (byte)'5', (byte)'6', (byte)'7',
		(byte)'8', (byte)'9', (byte)'A', (byte)'B',
		(byte)'C', (byte)'D', (byte)'E', (byte)'F'};

	/**
	 * @throws IOException 
	 */
	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebug)
	throws PDFCosParseException, PDFSecurityException, IOException
	{
			byte[] outValue = byteArrayValue(!inDebug);
			if (mToEncrypt && !inDebug)
				outValue = getDocument().getEncryption().encryptString(this, outValue);
			if (mWriteHex) {
				outStream.write('<');
				for (int i = 0; i < outValue.length; i++) {
					byte b = outValue[i];
					outStream.write(hexrep[(b >> 4) & 0xf]);
					outStream.write(hexrep[b & 0xf]);
				}
				outStream.write('>');
			} else {
				outStream.write('(');
				for (int i = 0; i < outValue.length; i++) {
					byte b = outValue[i];
					if (b == '\n') {
						outStream.write('\\');
						outStream.write('n');
					}
					else if (b == '\r') {
						outStream.write('\\');
						outStream.write('r');
					}
					else if (b == '(') {
						outStream.write('\\');
						outStream.write('(');
					}
					else if (b == ')') {
						outStream.write('\\');
						outStream.write(')');
					}
					else if (b == '\\') {
						outStream.write('\\');
						outStream.write('\\');
					}
					else {
						outStream.write(b);
					}
				}
				outStream.write(')');
			}
			mOddBall = false;
	}
	
	
	@Override
	public String toString()
	{
		boolean showEncrypted = true;
		/** DEBUG 
		 * Debugger plays games when it shows the stream's content. 
		 * It actually decrypts the string's content and then re-encrypts it 
		 * which in some cases can mess up what you're looking for. 
		 * If this is the case then uncomment the next statement. After that Debugger will show
		 * the string un-decrypted (i.e. encrypted but without decryption code executed).
		 * Do not forget to comment it back after you're done debugging!
		 **/ 
//		showEncrypted = false;
		return toString(!showEncrypted);
	}

	/**
	 * Returns this string as {@link ASDate} if it's in proper date format else throws exception.
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 */
	public ASDate asDate()
		throws PDFCosParseException, PDFSecurityException
	{
		try {
			return new ASDate(new ASString(byteArrayValue()));
		} catch (PDFParseException e) {
			throw new PDFCosParseException(e);
		}
	}

	/**
	 * If true is passed here then this string is written in hex format.
	 */
	public void setWriteHex(boolean b)
	{
		mWriteHex = b;
	}

	/**
	 * Returns true if this string is written in hex format.
	 */
	public boolean getWriteHex()
	{
		return mWriteHex;
	}

	/**
	 * Returns true if this is hex string and ended on odd nibble.
	 */
	public boolean isOddBall()
	{
		return mOddBall;
	}

	/**
	 * Sets true if string is to be encrypted.
	 */
	public void setToEncrypt(boolean encrypted)
	{
		mToEncrypt = encrypted;
	}

	/**
	 * Returns true if string is to be encrypted.
	 */
	public boolean getToEncrypt()
	{
		return mToEncrypt;
	}

	/**
	 * Sets the flag which tells whether this string is already encrypted or not.
	 */
	public void setIsEncrypted(boolean encrypted)
	{
		if (encrypted != mIsEncrypted) {
			mValue = null;
			mIsEncrypted = encrypted;
		}
	}

	/**
	 * Tells whether this string is already encrypted or not.
	 */
	public boolean getIsEncrypted()
	{
		return mIsEncrypted;
	}

	public boolean equals(ASString string) 
	{
		return this.equals(string.getBytes());
	}
	
	public boolean equals(CosString string)
	{
		try {
			return this.equals(string.byteArrayValue());
		} catch (PDFSecurityException e) {
			return false;
		}
	}
	
	/**
	 *  This method returns true if both CosStrings have same byte array inside.
	 *  Returns false if passed CosObject is not an instance of CosString or 
	 *  not on same document.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		if(!(value instanceof CosString) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		CosString cosString = (CosString) value;
		return equals(cosString);
	}
	public boolean equals(byte[] b)
	{
		try {
			byte[] myBytes = byteArrayValue();
			if (myBytes.length != b.length)
			{
				return false;
			}
			for (int i = 0; i < myBytes.length; i++)
			{
				if (myBytes[i] != b[i])
				{
					return false;
				}
			}
			return true;
		} catch (PDFSecurityException e) {
			return false;
		}
	}


}
