/*************************************************************************
 *
 *	File: PDF417TextCompactor.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.adobepdf417pmp;

import java.util.ArrayList;
import java.util.List;

//////////////////////////////////////////////////////////////////////
/**
 * Implementation of the Text compactor.
 * 
 * //////////////////////////////////////////////////////////////////////
 * 
 * 
 * Ported from PDF417TextCompactor.cpp
 */
class PDF417TextCompactor {

	enum SetType {
		ALPHA(0), LOWER(1), MIXED(2), PUNCTUATION(3);
		SetType(int value) {
			this.value = value;
		}

		private int value;

		public int getValue() {
			return value;
		}
	}

	final static String alphaSet = // ABCDEFGHIJKLMNOPQRSTUVWXYZ
	"\101\102\103\104\105\106\107\110\111\112\113\114\115\116\117\120\121\122\123\124\125\126\127\130\131\132\040";
	final static String lowerSet = // abcdefghijklmnopqrstuvwxyz
	"\141\142\143\144\145\146\147\150\151\152\153\154\155\156\157\160\161\162\163\164\165\166\167\170\171\172\040";
	final static String mixedSet = // 0123456789&\n\t,:#-.$/+%*=^
	"\060\061\062\063\064\065\066\067\070\071\046\015\011\054\072\043\055\056\044\057\053\045\052\075\136\040";
	final 	static String punctuationSet = // ;<>@[\\]_`~!\n\t,:-.$/\"|*()?{}'
	"\073\074\076\100\133\134\135\137\140\176\041\015\011\054\072\012\055\056\044\057\042\174\052\050\051\077\173\175\047";

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Do the text compaction.
	 * 
	 * @param message
	 *            - The input binary message.
	 * @returns A vector of the of the binary encoded message in PDF417 code
	 *          words.
	 */
	// ////////////////////////////////////////////////////////////////////
	static void compact(List<Character> message, List<Integer> compactedMessage) {
		// Create the output vector
		List<Integer> data = new ArrayList<Integer>();

		// Set the initial sub mode
		SetType subMode = SetType.ALPHA;
		String pCurrentSet = alphaSet;

		int idx = 0;
		// Cycle through the message
		for (idx = 0; idx < message.size(); idx++) {
			char c = message.get(idx);

			// Is the character in the current sub mode
			int index = pCurrentSet.indexOf(c);
			if (index >= 0) {
				// Handle the weirdness of the mixed list.
				if ((subMode == SetType.MIXED) && (index == 25))
					index = 26;
				data.add(index);
			} else {
				SubModeAndIndex subModeAndIndex = getSubModeAndIndex(c);
				if (subModeAndIndex == null)
					throw new IllegalArgumentException(
							"SubMode and index information can't be found");
				SetType newSubMode = subModeAndIndex.subMode;
				index = subModeAndIndex.index;
				List<Integer> subModeSwitch = getSubModeSwitch(subMode,
						newSubMode);

				// Add the sub mode switch
				int jdx = 0;
				for (jdx = 0; jdx < subModeSwitch.size(); jdx++) {
					int v = subModeSwitch.get(jdx);
					data.add(v);
				}
				data.add(index);

				subMode = newSubMode;
				switch (subMode) {
				case ALPHA:
					pCurrentSet = alphaSet;
					break;
				case LOWER:
					pCurrentSet = lowerSet;
					break;
				case MIXED:
					pCurrentSet = mixedSet;
					break;
				case PUNCTUATION:
					pCurrentSet = punctuationSet;
					break;
				}
			}
		}

		// Make sure its even
		if (data.size() % 2 != 0)
			data.add(29);

		// Create the output vector
		compactedMessage.add(PDF417SpecialCode.TextCompactionLatch.getValue());
		for (idx = 0; idx < data.size(); idx += 2) {
			int H = data.get(idx);
			int L = data.get(idx + 1);

			int codeWord = 30 * H + L;
			compactedMessage.add(codeWord);
		}
	}

	static class SubModeAndIndex {
		SetType subMode;
		int index;
	}

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Returns the submode and index of the character.
	 */
	// ////////////////////////////////////////////////////////////////////
	static SubModeAndIndex getSubModeAndIndex(char c) {
		SubModeAndIndex subModeAndIndex = new SubModeAndIndex();
		int index = lowerSet.indexOf(c);
		if (index >= 0) {
			subModeAndIndex.index = index;
			subModeAndIndex.subMode = SetType.LOWER;
			return subModeAndIndex;
		}

		index = alphaSet.indexOf(c);
		if (index >= 0) {
			subModeAndIndex.index = index;
			subModeAndIndex.subMode = SetType.ALPHA;
			return subModeAndIndex;
		}

		index = mixedSet.indexOf(c);
		if (index >= 0) {
			// Handle the weirdness of the mixed list.
			if (index == 25)
				index = 26;
			subModeAndIndex.index = index;
			subModeAndIndex.subMode = SetType.MIXED;
			return subModeAndIndex;
		}

		index = punctuationSet.indexOf(c);
		if (index >= 0) {
			subModeAndIndex.index = index;
			subModeAndIndex.subMode = SetType.PUNCTUATION;
			return subModeAndIndex;
		}
		return null;
	}

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Returns set of characters needed to switch between the submodes.
	 */
	// ////////////////////////////////////////////////////////////////////
	static List<Integer> getSubModeSwitch(SetType subMode, SetType newSubMode) {
		List<Integer> sunModeSwitch = new ArrayList<Integer>();
		switch (subMode) {
		case ALPHA: {
			switch (newSubMode) {
			case ALPHA:
				break;
			case LOWER:
				sunModeSwitch.add(27);
				break;
			case MIXED:
				sunModeSwitch.add(28);
				break;
			case PUNCTUATION:
				sunModeSwitch.add(28);
				sunModeSwitch.add(25);
				break;
			}
		}
			break;
		case LOWER: {
			switch (newSubMode) {
			case ALPHA:
				sunModeSwitch.add(28);
				sunModeSwitch.add(28);
				break;
			case LOWER:
				break;
			case MIXED:
				sunModeSwitch.add(28);
				break;
			case PUNCTUATION:
				sunModeSwitch.add(28);
				sunModeSwitch.add(25);
				break;
			}
		}
			break;
		case MIXED: {
			switch (newSubMode) {
			case ALPHA:
				sunModeSwitch.add(28);
				break;
			case LOWER:
				sunModeSwitch.add(27);
				break;
			case MIXED:
				break;
			case PUNCTUATION:
				sunModeSwitch.add(25);
				break;
			}
		}
			break;
		case PUNCTUATION: {
			switch (newSubMode) {
			case ALPHA:
				sunModeSwitch.add(29);
				break;
			case LOWER:
				sunModeSwitch.add(29);
				sunModeSwitch.add(27);
				break;
			case MIXED:
				sunModeSwitch.add(29);
				sunModeSwitch.add(28);
				break;
			case PUNCTUATION:
				break;
			}
		}
			break;
		}
		return sunModeSwitch;
	}

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Can the compactor handle this byte?
	 * 
	 * @returns TRUE if it can compact this byte. FALSE if it can NOT compact
	 *          this byte.
	 */
	// ////////////////////////////////////////////////////////////////////
	static boolean canHandle(char value) {
		return (alphaSet.indexOf(value) >= 0) || (lowerSet.indexOf(value) >= 0)
				|| (mixedSet.indexOf(value) >= 0)
				|| (punctuationSet.indexOf(value) >= 0);
	}

}
