/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import org.osgi.annotation.versioning.ProviderType;

import com.adobe.granite.crypto.CryptoSupport;
import com.day.cq.commons.inherit.InheritanceValueMap;
import com.day.cq.dam.scene7.api.constants.Scene7PublishMode;


/**
 * Scene7 server access configuration.
 */
@ProviderType
public interface S7Config {

    /**
     * Returns the configuration path of this configuration.
     *
     * @return the s7 cloud configuration path
     */
    String getCloudConfigPath();

    /**
     * Returns the email address used for this configuration.
     *
     * @return email address for the s7 cloud configuration
     */
    String getEmail();

    /**
     * Returns the encrypted password for the S7 service associated with this cloud configuration. To decrypt the password use the
     * {@link CryptoSupport}'s {@code unprotect} method.
     *
     * @return encrypted password for s7 service associated with this cloud configuration
     */
    String getPassword();

    /**
     * Returns the region for this configuration.
     *
     * @return associated s7 server region associated with this cloud configuration
     */
    String getRegion();

    /**
     * Returns the user handle associated with this configuration.
     *
     * @return the user handle associated with the s7 server account
     */
    String getUserHandle();

    /**
     * Returns the company handle associated with this configuration.
     *
     * @return the company handle associated with the s7 server account
     */
    String getCompanyHandle();

    /**
     * Returns the root path of the Scene7 company hierarchy (not to be confused with the actual physical path on the S7 servers).
     *
     * @return virtual root path
     */
    String getRootPath();

    /**
     * Returns the synchronisation status as a string.
     *
     * @return string value ("on/"off") that controls whether previously published folders/assets are kept in sync from cq to s7 when moved
     * or updated
     */
    String isSyncEnabled();

    /**
     * Returns the secure preview feature state support status as a string.
     *
     * @return string value ("on/"off") that controls whether DAM asset
     *         activation state reflects Scene7 asset MarkForPublish state
     */
    @Deprecated
    String isDelayedPublishEnabled();

    /**
     * Returns the secure preview feature state support status as a string.
     *
     * @return string value ("IMMEDIATE/"ML"/"SELECTIVE") that controls whether DAM asset
     *         activation state reflects Scene7 asset MarkForPublish state
     */
    Scene7PublishMode getPublishMode();

    /**
     * Returns the secure preview feature state support status as a string.
     *
     * @return string value ("on/"off") that controls whether DAM asset
     *         activation state reflects Scene7 asset MarkForPublish state
     * @deprecated use {@link #isDelayedPublishEnabled()}
     */
    @Deprecated
    String isPublishEnabled();

    /**
     * Returns the top most folder of the S7 company associated with this configuration.
     *
     * @return company root folder
     */
    String getBasePath();

    /**
     * Returns a property of this configuration with the specified name.
     *
     * @param prop the property's name
     * @return the value of the property
     */
    String get(String prop);

    /**
     * Returns the properties held in this S7 configuration object.
     *
     * @return the properties map
     */
    InheritanceValueMap getPropertiesMap();

    /**
     * Returns the Scene7 configured publish server.
     *
     * @return a String with the server's URL
     */
    String getPublishServer();

    /**
     * Returns the Scene7 configured preview server.
     *
     * @return a String with the server's URL
     */
    String getPreviewServer();

    /**
     * Returns the proper server for previewing, it return previewServer always regardless if immediate mode
     * It was designed to return server by checking immediate mode or not, but envolved into always return new preview server
     *
     * @return a String with the server's URL
     */
    String getPreviewableServer();

    /**
     * Returns the cached jwt token issued by IPS, to be used to authenticate by ImageServing Preview.
     *
     * @return a String with the jwt token
     */
    String getCachedPreviewServerToken();

    /**
     * Returns the configured Scene7 Flash Templates server.
     *
     * @return a String with the server's URL
     */
    @Deprecated
    String getFlashTemplatesServer();

    /**
     * Returns the target path where the Scene7 assets are downloaded into CQ DAM.
     *
     * @return a String containing the full path where the Scene7 assets will be saved into DAM
     */
    String getTargetPath();

    /**
     * Returns the default status of this configuration
     *
     * @return boolean value indicating if this S7 configuration is the default one
     */
    boolean isDefault();

    /**
     * Returns the cached S7 FXG endpoint URL. Deprecated and only return null now.
     *
     * @return a String containing the FXG URL
     */
    @Deprecated
    String getFXGServer();

    /**
     * Returns the Scene7 ad-hoc folder for this configuration.
     *
     * @return a String containing the ad-hoc folder path
     */
    String getAdhocFolder();

    /**
     * Returns the name of the import queue
     *
     * @return a String containing the name of the import queue
     */
    String getImportQueueName();

    /**
     * Returns the topic of the associated import queue
     *
     * @return a String containing the import queue topic
     */
    String getImportQueueTopic();

    /**
     * Returns the Scene7 upload job parameters associated with given MIME type
     *
     * @param mimeType asset MIME type
     * @return a String containing all Scene7 upload job parameters
     */
    String getMimeTypeJobParams(String mimeType);

    /**
     * Returns the Scene7 upload job parameters associated with given MIME type
     *
     * @param mimeType asset MIME type
     * @param addCommonJobParam add common job params
     * @return a String containing all Scene7 upload job parameters
     */
    String getMimeTypeJobParams(String mimeType, Boolean addCommonJobParam);

    /**
     * Returns the Scene7 upload job parameters configured under the scene7 config
     *
     * @return a String containing all Scene7 upload job parameters
     */
    String getAllMimeTypeJobParams();

    /**
     * sets the upload job Params of given mimetype
     * @param mimeType asset MIME type
     * @param jobParams job params
     */
    void setMimeTypeJobParams(String mimeType, String jobParams);

    /**
     * Returns if enable the S7 uploading with given MIME type, by default return true if cannot find the value
     *
     * @param mimeType asset MIME type
     * @return Boolean indicate if we sync such mimetype assets to S7
     */
    Boolean getMimeTypeEnabled(String mimeType);

    public enum SyncMode {
        INCLUDE("include"), EXCLUDE("exclude"), READ_ONLY("read-only") ;

        String mode;

        SyncMode(String mode) {
            this.mode = mode;
        }

        /**
         * Returns SyncMode enum value by parsing a corresponding string
         *
         * @param mode string value of syncmode
         * @return SyncMode
         */
        public static SyncMode fromString(String mode) {
            if (INCLUDE.mode.equalsIgnoreCase(mode)) {
                return INCLUDE;
            } else if (EXCLUDE.mode.equalsIgnoreCase(mode)) {
                return EXCLUDE;
            } else if (READ_ONLY.mode.equalsIgnoreCase(mode)) {
                return READ_ONLY;
            }
            return INCLUDE;
        }
    }

    SyncMode getDefaultSyncMode();

    boolean isSelectiveSyncEnabled();

    /**
     * Close ResourceResolver embedded into this S7Config instance, if necessary.
     *
     * You must call this method explicitly when this S7Config instance was obtained from
     *   factory methods which instantiate the new ResourceResolver
     *   (i.e. when factory method has no input ResourceResolver parameter).
     */
    void close();

    /**
     * Sets the param value at the Jcr:Content level
     * @param property param name
     * @param value param value
     */
    void setJCRContentProperty(String property, String value);
}
