/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import org.xml.sax.Attributes;

import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.EntryTagHandler;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * The style tag handler is used to pull out the inline page styles defined within
 * the style tag of the input HTML document.
 *
 * <p>
 * This tag handler is registered against the markup of type < style >. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.StyleTagHandlerFactory
 */
public class StyleTagHandler implements TagHandler, EntryTagHandler, HTMLContentProvider {

    private StringBuffer cssBuffer = new StringBuffer();
    private boolean keepInline = false;

    public void beginHandling(String uri, String localName, String qName, Attributes atts) {
        String cqInline = atts.getValue("data-cq-inline");
        if (cqInline != null && cqInline.trim().equalsIgnoreCase("true")) {
            keepInline = true;
        }
    }

    public void characters(char[] ch, int start, int length) throws DesignImportException {
        String chars = new String(ch).substring(start, start + length);
        if (chars.length() > 0) {
            cssBuffer.append(chars);
        }
    }

    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        // Invalid. Style tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public void endHandling(String uri, String localName, String qName) {
    }

    public Object getContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.STYLES_INLINE) return cssBuffer.toString();
        if (htmlContentType == HTMLContentType.MARKUP) return "<style>" + cssBuffer.toString() + "</style>";
        return null;
    }

    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
    }

    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        // Invalid. Style tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.STYLES_INLINE)
            return !keepInline;
        else if (htmlContentType == HTMLContentType.MARKUP)
            return keepInline;
        else
            return false;
    }
}
