/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.haf.converter.platform.api;

import com.adobe.granite.haf.annotations.ApiLink;
import com.adobe.granite.haf.annotations.ApiProperty;
import com.adobe.granite.haf.api.OrderByDetails;
import com.adobe.granite.haf.converter.api.ConverterContext;
import com.adobe.granite.haf.converter.api.ConverterResponseBuilder;
import com.adobe.granite.haf.converter.api.HypermediaConverter;
import com.adobe.granite.haf.api.SortOrder;
import com.adobe.granite.haf.converter.api.description.ModelDescription;
import org.apache.sling.api.resource.Resource;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.osgi.annotation.versioning.ProviderType;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

/**
 * Service that provides common functionality that can be used by any Platform API based HypermediaConverter when
 * serializing resources to Platform API style JSON.
 */
@ProviderType
public interface PlatformConverterService {
    String REL_ADOBECLOUD = "http://ns.adobe.com/adobecloud/rel/";
    String REL_ADOBECLOUD_DIRECTORY = "http://ns.adobe.com/adobecloud/rel/directory";
    String REL_ADOBECLOUD_DELETE = "http://ns.adobe.com/adobecloud/rel/delete";
    String REL_SELF = "self";
    String REL_PARENT = "parent";
    String REL_LATEST_VERSION = "latest-version";
    String REL_COLLECTION = "collection";

    String HEADER_LINK = "link";

    String PARAM_START = "start";
    String PARAM_LIMIT = "limit";
    String PARAM_ORDERBY = "orderby";
    String PARAM_SORTORDER = "order";

    String PROP_NEXT = "next";
    String PROP_PAGE = "page";
    String PROP_PROPERTY = "property";
    String PROP_TYPE = "type";
    String PROP_COUNT = "count";
    String PROP_NAME = "name";

    String JSON_NAME_PAGE = "_page";
    String JSON_NAME_LINKS = "_links";
    String JSON_NAME_CHILDREN = "children";
    String JSON_NAME_HREF = "href";

    SortOrder DEFAULT_SORTORDER = SortOrder.ASCENDING;

    /**
     * Builds the response headers as defined in the model using the ApiHeader annotations and sets them in the response
     * builder.
     * @param builder The ConverterResponseBuilder responsible for building the response.
     * @param description The metadata description of the model.
     */
    void populateHeaders(ConverterResponseBuilder builder, ModelDescription description);

    /**
     * Builds the link header for the response based on the ApiHeader annotations in the model, filtered by the scope
     * parameter supplied.
     * @param builder The ConverterResponseBuilder responsible for building the response.
     * @param context The ConverterContext providing information about the request.
     * @param resource The resource being serialized.
     * @param description The metadata description of the model.
     * @param scope The scope a link should satisfy in order to be serialized into the link header.
     * @throws UnsupportedEncodingException If encoding is not supported.
     */
    void populateLinkHeader(ConverterResponseBuilder builder, ConverterContext context, Resource resource,
                            ModelDescription description, ApiLink.SCOPE scope) throws URISyntaxException,
            UnsupportedEncodingException;

    /**
     * Builds the links JSONObject that can be added to the response body.
     * @param context The ConverterContext providing information about the request.
     * @param resource The resource being serialized.
     * @param description The metadata description of the model.
     * @return The JSONObject that is the links object.
     * @throws JSONException If an error occurs when creating the links object.
     * @throws URISyntaxException If an error occurs building the URL.
     * @throws UnsupportedEncodingException If encoding is not supported.
     */
    @Nonnull
    JSONObject buildLinksObject(ConverterContext context, Resource resource, ModelDescription description)
            throws JSONException, URISyntaxException, UnsupportedEncodingException;

    /**
     * Build a map of the properties to be serialized into the response body, filtered by the provided scope.
     * @param resource The resource being serialized.
     * @param description The metadata description of the model.
     * @param scope The scope a property should satisfy in order to be serialized into the Map.
     * @return The Map of properties to be serialized.
     */
    @Nonnull
    Map<String, Object> buildProps(Resource resource, ModelDescription description, ApiProperty.SCOPE scope);

    /**
     * Build the children JSONArray that can be added to the response body.
     * @param context The ConverterContext providing information about the request.
     * @param description The metadata description of the model.
     * @param converter The HypermediaConverter to be used to render the children resources as sub entities.
     * @return The JSONArray that represents the children to be serialized.
     */
    @Nonnull
    JSONArray buildChildren(ConverterContext context, ModelDescription description, HypermediaConverter converter);

    /**
     * Build the page JSONObject that can be added to the response body.
     * @param context The ConverterContext providing information about the request.
     * @param nextValue The value to be used to get the next page.
     * @param resourcePath The path to the resource.
     * @param category The API category.
     * @param count The number of resources contained in the children array.
     * @return The JSONObject that is the page object.
     * @throws JSONException If an error occurs when creating the JSONObject.
     */
    @CheckForNull
    JSONObject buildPageObject(ConverterContext context, String nextValue, String resourcePath, String category,
                               int count) throws JSONException;

    /**
     * Builds a URL for a resource that can be used in the response.
     * @param context The ConverterContext providing information about the request.
     * @param resourcePath The path to the resource.
     * @return The URL to the resource.
     * @throws URISyntaxException If an error occurs building the URL.
     * @throws UnsupportedEncodingException If encoding is not supported.
     */
    @Nonnull
    String buildURL(ConverterContext context, String resourcePath) throws URISyntaxException,
            UnsupportedEncodingException;

    /**
     * Get the API prefix that can be used when building URLs for API managed resources.
     * @param context The ConverterContext providing information about the request.
     * @param category The API category the resource is being managed under.
     * @return The API prefix string.
     */
    @Nonnull
    String getApiPrefix(ConverterContext context, String category);

    /**
     * Get the pagination start value that will be used to build a paginated child entity list.
     * @param request The HTTPServletRequest.
     * @return The pagination start value.
     */
    @CheckForNull
    String getPaginationStartValue(HttpServletRequest request);

    /**
     * Get the pagination limit value that will be used to build a paginated child entity list.
     * @param request The HTTPServletRequest.
     * @return The pagination limit value.
     */
    int getPaginationLimitValue(HttpServletRequest request);

    /**
     * Get the ordering parameters from the request that will be used to build a paginated child entity list.
     * @param request The HTTPServletRequest.
     * @return The pagination ordering parameters.
     */
    @Nonnull
    List<OrderByDetails> getOrderBy(HttpServletRequest request);
}
