/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2012 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/

package com.adobe.granite.ui.components;

import java.util.Collections;
import java.util.Iterator;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.CompositeValueMap;

/**
 * The config properties of a resource.
 */
public class Config {
    @CheckForNull
    private Resource resource;

    /**
     * Current resource properties merged with default properties.
     */
    @Nonnull
    private ValueMap properties;

    /**
     * Current resource properties merge with parent's properties.
     */
    @Nonnull
    private ValueMap propertiesWithParent;

    @CheckForNull
    private Resource parentResource;

    private boolean isParentProcessed;

    /**
     * The resource name for the defaults config ("defaults").
     */
    @Nonnull
    public static String DEFAULTS = "defaults";

    /**
     * The resource name for items ("items").
     */
    @Nonnull
    public static String ITEMS = "items";

    /**
     * The resource name for layout config ("layout").
     */
    @Nonnull
    public static String LAYOUT = "layout";

    /**
     * The resource name for datasource config ("datasource").
     */
    @Nonnull
    public static String DATASOURCE = "datasource";

    /**
     * The resource name for render condition config ("rendercondition").
     */
    @Nonnull
    public static String RENDERCONDITION = "rendercondition";

    /**
     * Alias of {@link #Config(Resource, boolean)} with inherit is {@code false}.
     *
     * @param resource
     *            The resource
     */
    public Config(@CheckForNull Resource resource) {
        this(resource, false);
    }

    /**
     * Create a new Config object for the given resource.
     *
     * If {@code inherit} is {@code true}, the default properties are setup and
     * taken from the parent's {@link #DEFAULTS} sub-resource, and parent properties
     * are used when calling {@link #getInherited(String, Class)}.
     *
     * @param resource
     *            The resource
     * @param inherit
     *            Whether to inherit default properties or not
     * @see #getParentResource()
     */
    public Config(@CheckForNull Resource resource, boolean inherit) {
        this.resource = resource;

        ValueMap prop = resource != null ? resource.getValueMap() : ValueMap.EMPTY;

        ValueMap defaults = null;
        ValueMap parent = null;

        if (inherit) {
            Resource parentResource = getParentResource();
            if (parentResource != null) {
                Resource defaultsResource = parentResource.getChild(DEFAULTS);
                if (defaultsResource != null) {
                    defaults = defaultsResource.getValueMap();
                }
                parent = parentResource.getValueMap();
            }
        }

        properties = new CompositeValueMap(prop, defaults);
        propertiesWithParent = new CompositeValueMap(prop, parent);
    }

    /**
     * Create a new Config object for the given resource.
     *
     * @param resource
     *            The resource
     * @param defaults
     *            The default properties. Pass {@code null} if these are not
     *            required.
     * @param parentProperties
     *            The properties of the parent resource. Pass {@code null} if these
     *            are not required.
     */
    public Config(@CheckForNull Resource resource, @CheckForNull ValueMap defaults,
            @CheckForNull ValueMap parentProperties) {
        this.resource = resource;

        ValueMap prop = resource != null ? resource.getValueMap() : ValueMap.EMPTY;

        properties = new CompositeValueMap(prop, defaults);
        propertiesWithParent = new CompositeValueMap(prop, parentProperties);
    }

    /**
     * Returns the value of property with the given name, converted to string. If
     * the property does not exist it is taken from the defaults. Return an empty
     * string if still not existing can't be converted.
     *
     * @param name
     *            the name of the property
     * @return the value of property with the given name, converted to string
     */
    @Nonnull
    public String get(@Nonnull String name) {
        return get(name, "");
    }

    /**
     * Returns the value of property with the given name, converted into the given
     * type. If the property does not exist it is taken from the defaults. Return
     * the given defaultValue if still non existing or can't be converted.
     *
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist or
     *            cannot be converted to the requested type. The default value is
     *            also used to define the type to convert the value to. If this is
     *            {@code null} any existing property is not converted.
     * @param <T>
     *            The type of the default value
     * @return The value of property with the given name, converted into the given
     *         type
     */
    @Nonnull
    public <T> T get(@Nonnull String name, @Nonnull T defaultValue) {
        return properties.get(name, defaultValue);
    }

    /**
     * Returns the value of property with the given name, converted into the given
     * type. If the property does not exist, it is taken from the defaults. Return
     * {@code null} if non existing or can't be converted.
     *
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type of the submitted type param
     * @return The value of property with the given name, converted into the given
     *         type
     */
    @CheckForNull
    public <T> T get(@Nonnull String name, @Nonnull Class<T> type) {
        return properties.get(name, type);
    }

    /**
     * Returns the value of property with the given name, converted to string. If
     * the property does not exist, it is taken from the parent properties. Return
     * an empty string if still not existing can't be converted.
     *
     * @param name
     *            The name of the property
     * @return The value of property with the given name, converted to string.
     *
     * @see #getParentResource()
     */
    @Nonnull
    public String getInherited(@Nonnull String name) {
        return getInherited(name, "");
    }

    /**
     * Returns the value of property with the given name, converted into the given
     * type. If the property does not exist, it is taken from the parent properties.
     * Return the given defaultValue if still non existing or can't be converted.
     *
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist or
     *            cannot be converted to the requested type. The default value is
     *            also used to define the type to convert the value to. If this is
     *            {@code null} any existing property is not converted.
     * @param <T>
     *            The type to convert the value of the property to
     * @return The value of property with the given name, converted into the given
     *         type
     *
     * @see #getParentResource()
     */
    @Nonnull
    public <T> T getInherited(@Nonnull String name, @Nonnull T defaultValue) {
        return propertiesWithParent.get(name, defaultValue);
    }

    /**
     * Returns the value of property with the given name, converted into the given
     * type. If the property does not exist, it is taken from the parent properties.
     * Return {@code null} if non existing or can't be converted.
     *
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type to convert the value of the property to
     * @return The value of property with the given name, converted into the given
     *         type
     *
     * @see #getParentResource()
     */
    @CheckForNull
    public <T> T getInherited(@Nonnull String name, @Nonnull Class<T> type) {
        return propertiesWithParent.get(name, type);
    }

    /**
     * Alias of {@link #get(String)}.
     *
     * @param name
     *            The name of the property
     * @return The value of property with the given name, converted to string
     */
    @Nonnull
    public String getInheritedDefault(@Nonnull String name) {
        return getInheritedDefault(name, "");
    }

    /**
     * Alias of {@link #get(String, Object)};
     *
     * @return The value of property with the given name, converted to string
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist or
     *            cannot be converted to the requested type. The default value is
     *            also used to define the type to convert the value to. If this is
     *            {@code null} any existing property is not converted.
     * @param <T>
     *            The type to convert the value of the property to
     */
    @Nonnull
    public <T> T getInheritedDefault(@Nonnull String name, @Nonnull T defaultValue) {
        return get(name, defaultValue);
    }

    /**
     * Alias of {@link #get(String, Class)}.
     *
     * @return The value of property with the given name, converted to string
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type to convert the value of the property to
     */
    @CheckForNull
    public <T> T getInheritedDefault(@Nonnull String name, @Nonnull Class<T> type) {
        return get(name, type);
    }

    /**
     * Returns the default properties that will be applied to child elements. Be
     * aware that these are not the defaults that are applied to the resource
     * itself.
     *
     * @return The value map containing the default properties
     */
    @SuppressWarnings("null")
    @Nonnull
    public ValueMap getDefaultProperties() {
        Resource r = resource;

        Resource defaults = null;
        if (r != null) {
            defaults = r.getChild(DEFAULTS);
        }

        return defaults != null ? defaults.getValueMap() : ValueMap.EMPTY;
    }

    /**
     * Returns the properties of the resource merged with the default properties.
     *
     * @return The value map containing the properties merged with the default
     *         properties.
     */
    @Nonnull
    public ValueMap getProperties() {
        return properties;
    }

    /**
     * Returns the child resources of {@link #ITEMS} sub-resource.
     *
     * @return The child resources of the specified {@link #ITEMS} sub-resource
     */
    @Nonnull
    public Iterator<Resource> getItems() {
        return getItems(ITEMS);
    }

    /**
     * Returns the child resources of the sub-resource with the given name.
     *
     * @param name
     *            The name of the resource
     * @return The child resources of the sub-resource with the given name
     */
    @Nonnull
    public Iterator<Resource> getItems(@CheckForNull String name) {
        return getItems(resource, name);
    }

    /**
     * Returns the child resources of {@link #ITEMS} sub-resource of the given
     * resource.
     *
     * @param resource
     *            The resource
     * @return The child resources of the {@link #ITEMS} sub-resource of the given
     *         resource
     */
    @Nonnull
    public Iterator<Resource> getItems(@CheckForNull Resource resource) {
        return getItems(resource, null);
    }

    /**
     * Returns the child resources of the sub resource with the given name of the
     * given resource.
     *
     * @param name
     *            The name of the resource
     * @param resource
     *            The resource
     * @return the child resources of the sub resource with the given name of the
     *         given resource.
     */
    @SuppressWarnings("null")
    @Nonnull
    public Iterator<Resource> getItems(@CheckForNull Resource resource, @CheckForNull String name) {
        if (name == null || name.length() == 0) {
            name = ITEMS;
        }

        if (resource != null) {
            Resource items = resource.getChild(name);
            if (items != null) {
                return items.listChildren();
            }
        }

        return Collections.<Resource>emptyList().iterator();
    }

    /**
     * Returns the child resource with the given name.
     *
     * @param name
     *            The name of the resource
     * @return The child resource
     */
    @CheckForNull
    public Resource getChild(@Nonnull String name) {
        Resource r = resource;
        return r == null ? null : r.getChild(name);
    }

    /**
     * Returns the parent of the resource. If the parent is of type
     * {@link JcrConstants#NT_UNSTRUCTURED}, then the parent is a container (e.g.
     * "items"), then the container's parent is returned instead.
     *
     * @return The parent of the resource
     */
    @CheckForNull
    public Resource getParentResource() {
        if (isParentProcessed) {
            return parentResource;
        }

        Resource r = resource;
        Resource parent = null;

        if (r != null) {
            parent = r.getParent();
        }

        if (parent != null) {
            if (JcrConstants.NT_UNSTRUCTURED.equals(parent.getResourceType())) {
                parent = parent.getParent();
            }
        }

        isParentProcessed = true;
        parentResource = parent;
        return parent;
    }
}
