/* ****************************************************************************
 *
 *	File: EncryptionHandler.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *.
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.securityframework;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;

/**
 * Implements specific encryption and decryption algorithms.      
 */
public interface EncryptionHandler 
{
	
	/**
	 * Encrypts the byte array with the encryption key calculated from the base
	 * encryption key held in the implementation and the <i>addKey</i> which is
	 * an addition to the base key.</P>
	 * <P>The encryption algorithm depends on a particular implementation 
	 * of this class. </P>
	 * @param src the input byte array.
	 * @param addKey the addition to the encryption key.
	 * @return encrypted byte array.
     * @throws PDFSecurityConfigurationException
	 */
	byte[] encrypt(byte[] src, byte[] addKey) 
    throws PDFSecurityConfigurationException;
	

	/**
	 * Encrypts the byte stream with the encryption key calculated from the base
	 * encryption key held in the implementation and the <i>addKey</i> which is
	 * an addition to the base key.</P>
	 * <P>The encryption algorithm depends on a particular implementation 
	 * of this class. </P>
	 * @param src the input byte stream.
	 * @param dest the output stream of bytes
	 * @param addKey the addition to the encryption key.
	 * @throws PDFSecurityConfigurationException
	 * @throws PDFIOException 
	 */
	void encrypt(InputByteStream src, OutputByteStream dest, byte[] addKey) 
    throws PDFIOException, PDFSecurityConfigurationException;

	
	/**
	 * Decrypts the byte array with the encryption key calculated from the base
	 * encryption key held in the implementation and the <i>addKey</i> which is
	 * an addition to the base key.</P>
	 * <P>The encryption algorithm depends on a particular implementation 
	 * of this class. </P>
	 * @param src the input byte array.
	 * @param addKey the addition to the encryption key.
	 * @return decrypted byte array.
     * @throws PDFSecurityConfigurationException
	 */
	byte[] decrypt(byte[] src, byte[] addKey) 
    throws PDFSecurityConfigurationException;
	
	/**
     * <P>Decrypts the byte stream with the encryption key calculated from the base
	 * encryption key held in the implementation and the <i>addKey</i> which is
	 * an addition to the base key.</P>
	 * <P>The encryption algorithm depends on a particular implementation 
	 * of this class. </P>
	 * @param src an input stream of bytes.
	 * @param dest the output stream of bytes
	 * @param addKey the addition to the encryption key.
	 * @throws PDFIOException 
     * @throws PDFSecurityConfigurationException
	 */
	void decrypt(InputByteStream src, OutputByteStream dest, byte[] addKey) 
    throws PDFIOException, PDFSecurityConfigurationException;

	/**
	 * @return EncryptionHandlerState object associated with this EncryptionHandler for
	 * sequential decryption/encryption of a data stream.
	 */
	EncryptionHandlerState createEncryptionHandlerState()
	throws PDFSecurityConfigurationException;
}
