/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.formcalc;


import java.lang.reflect.Method;
import java.util.Locale;



/**
 * This defines additional miscellaneous methods
 * from the ParserImpl class.
 *
 * @author Paul Imerson
 *
 * @exclude from published api.
 */
final class Builtins {

	/*
	 *  Disallow instances of this class.
	 */
	private Builtins() {
	}

	/*
	 *  InitBuiltin
	 *      This function initializes the builtin table.
	 */
	static void init(SymbolTable oTable) {
		// Arithmetic builtins
		add(oTable, "Abs", BuiltinMath.class);
		add(oTable, "Avg", BuiltinMath.class);
		add(oTable, "Ceil", BuiltinMath.class);
		add(oTable, "Count", BuiltinMath.class);
		add(oTable, "Floor", BuiltinMath.class);
		add(oTable, "Max", BuiltinMath.class);
		add(oTable, "Min", BuiltinMath.class);
		add(oTable, "Mod", BuiltinMath.class);
		add(oTable, "Round", BuiltinMath.class);
		add(oTable, "Sum", BuiltinMath.class);
		// Date and Time builtins
		add(oTable, "Date", BuiltinDate.class);
		add(oTable, "Date2Num", BuiltinDate.class);
		add(oTable, "DateFmt", BuiltinDate.class);
		add(oTable, "DateTimeFmt", BuiltinDate.class);
		add(oTable, "IsoDate2Num", BuiltinDate.class);
		add(oTable, "IsoTime2Num", BuiltinDate.class);
		add(oTable, "LocalDateFmt", BuiltinDate.class);
		add(oTable, "LocalTimeFmt", BuiltinDate.class);
		add(oTable, "Num2Date", BuiltinDate.class);
		add(oTable, "Num2GmTime", BuiltinDate.class);
		add(oTable, "Num2Time", BuiltinDate.class);
		add(oTable, "Time", BuiltinDate.class);
		add(oTable, "Time2Num", BuiltinDate.class);
		add(oTable, "TimeFmt", BuiltinDate.class);
		// Financial builtins
		add(oTable, "Apr", BuiltinFinancial.class);
		add(oTable, "Cterm", BuiltinFinancial.class);
		add(oTable, "Fv", BuiltinFinancial.class);
		add(oTable, "Ipmt", BuiltinFinancial.class);
		add(oTable, "Npv", BuiltinFinancial.class);
		add(oTable, "Pmt", BuiltinFinancial.class);
		add(oTable, "Ppmt", BuiltinFinancial.class);
		add(oTable, "Pv", BuiltinFinancial.class);
		add(oTable, "Rate", BuiltinFinancial.class);
		add(oTable, "Term", BuiltinFinancial.class);
		// Logical builtins
		add(oTable, "Choose", BuiltinLogical.class);
		add(oTable, "Exists", BuiltinLogical.class);
		add(oTable, "HasValue", BuiltinLogical.class);
		add(oTable, "If", BuiltinLogical.class);
		add(oTable, "Oneof", BuiltinLogical.class);
		add(oTable, "Throw", BuiltinLogical.class);
		add(oTable, "Within", BuiltinLogical.class);
		// Miscellaneous builtins
		add(oTable, "Eval", BuiltinMisc.class);
		add(oTable, "Ref", BuiltinMisc.class);
		// Scientific builtins
		add(oTable, "Acos", BuiltinScientific.class);
		add(oTable, "Asin", BuiltinScientific.class);
		add(oTable, "Atan", BuiltinScientific.class);
		add(oTable, "Atan2", BuiltinScientific.class);
		add(oTable, "Cos", BuiltinScientific.class);
		add(oTable, "Deg2Rad", BuiltinScientific.class);
		add(oTable, "Exp", BuiltinScientific.class);
		add(oTable, "Log", BuiltinScientific.class);
		add(oTable, "Pi", BuiltinScientific.class);
		add(oTable, "Pow", BuiltinScientific.class);
		add(oTable, "Rad2Deg", BuiltinScientific.class);
		add(oTable, "Sin", BuiltinScientific.class);
		add(oTable, "Sqrt", BuiltinScientific.class);
		add(oTable, "Tan", BuiltinScientific.class);
		// String builtins
		add(oTable, "At", BuiltinString.class);
		add(oTable, "Concat", BuiltinString.class);
		add(oTable, "Format", BuiltinString.class);
		add(oTable, "Left", BuiltinString.class);
		add(oTable, "Len", BuiltinString.class);
		add(oTable, "Lower", BuiltinString.class);
		add(oTable, "Ltrim", BuiltinString.class);
		add(oTable, "NumFmt", BuiltinString.class);
		add(oTable, "Parse", BuiltinString.class);
		add(oTable, "Replace", BuiltinString.class);
		add(oTable, "Right", BuiltinString.class);
		add(oTable, "Rtrim", BuiltinString.class);
		add(oTable, "Space", BuiltinString.class);
		add(oTable, "Str", BuiltinString.class);
		add(oTable, "Stuff", BuiltinString.class);
		add(oTable, "Substr", BuiltinString.class);
		add(oTable, "Uuid", BuiltinString.class);
		add(oTable, "Upper", BuiltinString.class);
		add(oTable, "WordNum", BuiltinString.class);
		// Encode builtins
		add(oTable, "Encode", BuiltinEncode.class);
		add(oTable, "Decode", BuiltinEncode.class);
		// Unit builtins
		add(oTable, "UnitValue", BuiltinUnit.class);
		add(oTable, "UnitType", BuiltinUnit.class);
	}


	/*
	 * addBuiltin
	 *  This function installs a function in the builtin table.
	 */
	static void add(SymbolTable oTable, String name, Class<?> c) {
		CalcSymbol oSym = oTable.install(name.toUpperCase(Locale.US));
		oSym.setType(CalcSymbol.TypeBuiltin);
		try {
			Class<?>[] parms = new Class[] {
				CalcParser.class,
				CalcSymbol[].class
			}; 
			Method func = c.getDeclaredMethod(name, parms);
			oSym.setFuncValue(func);
		} catch (NoSuchMethodException e) {
			assert false;
		}
	}


	/*
	 * MinArgs
	 *  This function limits the minimum number of arguments being passed
	 *  to a builtin function.
	 */ 
	static void minArgs(int nCount, int nMin) {
		if (nCount < nMin)
			throw new CalcException();
	}


	/*
	 * MaxArgs
	 *  This function limits the maxinum number of arguments being passed
	 *  to a builtin function.
	 */
	static void maxArgs(int nCount, int nLimit) {
		if (nCount > nLimit)
			throw new CalcException();
	}


	/*
	 * LimitNullArgs
	 *  This function limits any null-valued arguments being passed
	 *  to a builtin function.
	 */
	static void limitNullArgs(CalcParser oParser, int nArgs, CalcSymbol[] oSymArray) {
		for (int i = 0; i < nArgs; i++) {
			if (oSymArray[i].getType() == CalcSymbol.TypeAccessor) {
				try {
					CalcSymbol[] oSym = oParser.moScriptHost.getItemValue(
						oSymArray[i].getName(), oSymArray[i].getObjValues());
					CalcSymbol.delete(oSymArray[i], oParser);
					oSymArray[i] = oSym[0];
					for (int j = oSym.length - 1; j > 0; j--)
						CalcSymbol.delete(oSym[j], oParser);
				}
				catch (CalcException  e) {
					throw e;
				}
			}
			if (oSymArray[i].getType() == CalcSymbol.TypeNull) {
				throw new CalcException(oSymArray[i]);
			}
		}
	}


	/*
	 * LimitExceptionArgs
	 *	This function limits the error-valued, and
	 *  return-valued arguments being passed to a builtin function.
	 */
	static void limitExceptionArgs(CalcSymbol[] oSymArray) {
		for (int i = 0; i < oSymArray.length; i++) {
			if (oSymArray[i].getType() == CalcSymbol.TypeError)
				throw new CalcException(oSymArray[i]);
		}
		for (int i = 0; i < oSymArray.length; i++) {
			if (oSymArray[i].getType() == CalcSymbol.TypeReturn)
				throw new CalcException(oSymArray[i]);
		}
	}


	/*
	 * LimitAllNullArgs
	 *  This function limits all null-valued arguments
	 *  being passed to a builtin function.
	 */
	static CalcSymbol[] limitAllNullArgs(CalcParser oParser, CalcSymbol[] oSymArray) {
		for (int i = 0; i < oSymArray.length; i++) {
			final CalcSymbol symbol = oSymArray[i];
			if (symbol.getType() == CalcSymbol.TypeAccessor) {
				try {
					CalcSymbol[] oSym = oParser.moScriptHost.getItemValue(
						symbol.getName(), symbol.getObjValues());
					final int nSyms = oSym.length;
					if (nSyms > 1) {
						CalcSymbol[] oNewArray = new CalcSymbol[oSymArray.length + nSyms - 1];						
						System.arraycopy(oSymArray, 0,     oNewArray, 0,         i                       );						
						System.arraycopy(oSymArray, i + 1, oNewArray, nSyms + i, oSymArray.length - i - 1);						
						oSymArray = oNewArray;
					}
					CalcSymbol.delete(symbol, oParser);
					System.arraycopy(oSym, 0, oSymArray, i, nSyms);
				}
				catch(CalcException e) {
					CalcSymbol.delete(symbol, oParser);
					oSymArray[i] = e.getSymbol();
				}
			}
			if (oSymArray[i].getType() != CalcSymbol.TypeNull) {
				return oSymArray;
			}
		}
		throw new CalcException();
	}
}