/*
 * Adobe Confidential
 * ------------------
 * Copyright 2005 Adobe Systems Incorporated.  All Rights Reserved.
 *
 * Notice: all information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any.  The intellectual and 
 * technical concepts contained herein are proprietary to Adobe Systems Inc.
 * and its suppliers and may be covered by U.S. and foreign patents, patents in 
 * process, and are protected by trade secret or copyright law.  Dissemination 
 * of this information or reproduction of this material is strictly forbidden 
 * unless prior written permission is obtained from Adobe Systems Inc.
 */
package com.adobe.xfa.svg;


import com.adobe.xfa.ChildReln;
import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.Schema;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.STRS;
import com.adobe.xfa.SchemaStrings;


/**
 * @exclude from published api.
 */
public final class SVGSchema extends Schema {

	public static final String aSVG_WIDTH = "width";

	public static final String aSVG_XFACAPTION = "Caption";

	public static final String aSVG_XFACONTENT = "Content";

	// These literals must correspond to the SVG tag array.
	private static final String[] gSVGStrings = {
	// ELEMENTS
		SVG.TEXTDATA,
		SVG.DESC,
		SVG.G,
		SVG.LINE,
		SVG.SVG,
		SVG.TEXT,
		SVG.TSPAN,

	// ATTRIBUTES
		SVG.ALTGLYPH,
		SVG.CODEPAGE,
		SVG.FILL,
		SVG.FONTFAMILY,
		SVG.FONTSIZE,
		SVG.FONTSTYLE,
		SVG.FONTWEIGHT,
		SVG.HEIGHT,
		SVG.ID,
		SVG.STROKE,
	//	SVG.STROKEDASHARRAY,
	//	SVG.STROKEDASHOFFSET,
	//	SVG.STROKELINECAP,
		SVG.STROKEWIDTH,
		SVG.TRANSFORM,
		SVG.VIEWBOX,
		SVG.WIDTH,
		SVG.X,
		SVG.X1,
		SVG.X2,
		SVG.XMLNS,
		SVG.Y,
		SVG.Y1,
		SVG.Y2
	};

	private static final SVGSchema goSVG = new SVGSchema();

	
	/**
	 * Get the static version of the SVG Schema
	 * 
	 * @return the SVG schema instance
	 * @exclude
	 */
	public static SVGSchema getSVGSchema() {
		return goSVG;
	}

	/**
	 * @param aSVGName
	 *            the name to lookup
	 * @param bIsElement
	 *            a clue whether we're looking for an element name or attribute
	 *            name
	 * @return the tag if found (-1 if not found)
	 * @exclude
	 */
	static public int getTagImpl(String aSVGName, boolean bIsElement /* = true */) {
		// if the atom is undefined, then there can't be a related tag
		if (aSVGName == null)
			return SVG.SVG_INVALID;

		int nRet;
		if (bIsElement) {
			nRet = getSVGSchema().mElementTags.getInt(aSVGName);
		} else {
			nRet = getSVGSchema().mAttributeTags.getInt(aSVGName);
		}

		return nRet;
	}

	private final SchemaStrings mAttributeTags = new SchemaStrings(SVG.SVG_ATTRIBUTE_COUNT);

	private final SchemaStrings mElementTags = new SchemaStrings(SVG.SVG_ELEMENT_COUNT);

	SVGSchema() {
		super(STRS.SVG_NAMESPACE, SVG.SVG_ATTRIBUTE_MIN, SVG.SVG_ATTRIBUTE_MAX,
				SVG.SVG_ELEMENT_MIN, SVG.SVG_ELEMENT_MAX);
		schemaUsesProtos(false);

		initSchema();
	}

	/**
	 * @see Schema#getAtom(int)
	 * @exclude
	 */
	public String getAtom(int eTag) {
		assert (eTag >= mnElementMin && eTag <= mnAttributeMax);
		return gSVGStrings[eTag - mnElementMin];
	}

	/**
	 * @see Schema#getAttributeTag(String, String)
	 * @exclude
	 */
	public int getAttributeTag(String aNS, String aAttrName) {
		return getTagImpl(aAttrName, false);
	}

	/**
	 * @see Schema#getElementTag(String, String)
	 * @exclude
	 */
	public int getElementTag(String aNS, String aNodeName) {
		return getTagImpl(aNodeName, true);
	}

	protected int getTextTag() {
		return SVG.TEXTDATATAG;
	}

	/**
	 * @see Schema#initSchema()
	 * @exclude
	 */
	protected void initSchema() {
		super.initSchema();

		for (int i = 0; i < gSVGStrings.length; i++) {
			String atom = gSVGStrings[i];
			if (i < SVG.SVG_ELEMENT_COUNT) {
				mElementTags.put(atom, i + mnElementMin);
			} else if (i <= SVG.SVG_KEYWORD_COUNT) {
				mAttributeTags.put(atom, (i - SVG.SVG_ELEMENT_COUNT) + mnAttributeMin);
			}
		}
		// Ensure counts are correct
		assert (gSVGStrings.length == SVG.SVG_KEYWORD_COUNT);

		putAttribute(SVG.SVGTAG, SVG.WIDTHTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.SVGTAG, SVG.HEIGHTTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.SVGTAG, SVG.VIEWBOXTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putElement(SVG.SVGTAG, SVG.GTAG, ChildReln.getZeroOrMore(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putElement(SVG.SVGTAG, SVG.LINETAG, ChildReln.getZeroOrMore(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

		putAttribute(SVG.GTAG, SVG.TRANSFORMTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.GTAG, SVG.WIDTHTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.GTAG, SVG.HEIGHTTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.GTAG, SVG.IDTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putElement(SVG.GTAG, SVG.DESCTAG, ChildReln.getZeroOrOne(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putElement(SVG.GTAG, SVG.LINETAG, ChildReln.getZeroOrMore(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putElement(SVG.GTAG, SVG.TEXTTAG, ChildReln.getZeroOrMore(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

		putElement(SVG.DESCTAG, SVG.TEXTDATATAG, ChildReln.getZeroOrOne(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

		putElement(SVG.TEXTTAG, SVG.TSPANTAG, ChildReln.getZeroOrMore(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.ALTGLYPHTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.FONTFAMILYTAG, new StringAttr(SVG.FONTFAMILY, STRS.COURIERSTD), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		// Javaport: TODO: We have some work to do to fit SVG attributes into the enum schema.
		putAttribute(SVG.TEXTTAG, SVG.FONTSTYLETAG, null /* EnumValue.getEnum(SVG.FONTSTYLETAG, EnumAttr.getEnum(EnumAttr.POSTURE_NORMAL)) */, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.FONTWEIGHTTAG, null /* EnumValue.getEnum(SVG.FONTWEIGHTTAG, EnumAttr.getEnum(EnumAttr.WEIGHT_NORMAL)) */, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.FONTSIZETAG, new StringAttr(SVG.FONTSIZE, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.FILLTAG, new StringAttr(SVG.FILL, "#000000"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.CODEPAGETAG, null /* EnumValue.getEnum(SVG.CODEPAGETAG, EnumAttr.getEnum(EnumAttr.ISO_8859_1)) */, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TEXTTAG, SVG.TRANSFORMTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

		putElement(SVG.TSPANTAG, SVG.TEXTDATATAG, ChildReln.getZeroOrOne(), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.XTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.YTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.FONTFAMILYTAG, new StringAttr(SVG.FONTFAMILY, STRS.COURIERSTD), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
//		putAttribute(SVG.TSPANTAG, EnumValue.getEnum(SVG.FONTSTYLETAG, EnumAttr.getEnum(EnumAttr.POSTURE_NORMAL)), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
//		putAttribute(SVG.TSPANTAG, EnumValue.getEnum(SVG.FONTWEIGHTTAG, EnumAttr.getEnum(EnumAttr.WEIGHT_NORMAL)), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.FONTSIZETAG, new StringAttr(SVG.FONTSIZE, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.FILLTAG, new StringAttr(SVG.FILL, "#000000"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
//		putAttribute(SVG.TSPANTAG, EnumValue.getEnum(SVG.CODEPAGETAG, EnumAttr.getEnum(EnumAttr.ISO_8859_1)), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.TSPANTAG, SVG.ALTGLYPHTAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

		putAttribute(SVG.LINETAG, SVG.X1TAG, new StringAttr(SVG.X1, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.LINETAG, SVG.X2TAG, new StringAttr(SVG.X2, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.LINETAG, SVG.Y1TAG, new StringAttr(SVG.Y1, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.LINETAG, SVG.Y2TAG, new StringAttr(SVG.Y2, "0"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.LINETAG, SVG.STROKETAG, null, 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);
		putAttribute(SVG.LINETAG, SVG.STROKEWIDTHTAG, new StringAttr( SVG.STROKEWIDTH, "1000"), 0, Schema.XFAAVAILABILITY_ALL, /* 0, */ 0);

	}
	
	protected Element newElement(int eTag, Element parent, Node prevSibling) {
		
		switch (eTag) {
		case SVG.SVGTAG: 
		case SVG.GTAG:
		case SVG.DESCTAG:
		case SVG.TEXTTAG:
		case SVG.TSPANTAG:
		case SVG.LINETAG:
		case SVG.TEXTDATATAG:
			return new com.adobe.xfa.svg.SVGNode(parent, prevSibling);
			
		default:
			return null;
		}
	}
}
