/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.ut;


import java.util.ArrayList;
import java.util.List;


/**
 * An exception class derived from {@link ExceptionBase} that
 * simply consists of a collection of exceptions.
 * <p>
 * Use this class to collect multiple errors when different areas of code need
 * to add detail information to errors as they percolate up from several
 * catch/throw blocks.
 */
public class ExFull extends ExceptionBase {

	/*
	 * serialver-generated UID.
	 */
	private static final long serialVersionUID = -2764654020640603225L;

	/*
	 * collection of ExErrItem exceptions.
	 */
	private final List<ExErrItem> moItems = new ArrayList<ExErrItem>();
												

	/**
	 * Instantiate a ExFull object with a single
	 * exception in its collection.
	 *
	 * @exclude from published api.
	 */
	public ExFull() {
	}

	/**
	 * Instantiates an ExFull object with a single
	 * exception in its collection
	 * whose error message text property is
	 * the text of the given Java exception.
	 * @param e the Java exception.
	 */
	public ExFull(Exception e) {
		/*
		 * There are two approaches we could take here: a) We could translate
		 * each Java exception to an XTG equivalent b) We can pass the text of
		 * the Java exception along in an ExFull Since option b) is *much* less
		 * work and will provide a more meaningful result, we'll go with that
		 * strategy for now.
		 */
		moItems.add(new ExErrItem(ResId.SOFTWARE_FAILURE, e.getMessage()));
	}

	/**
	 * Instantiates an ExFull object with a single
	 * exception in its collection.
	 * @param nResId 
	 *            set this object's exception resource property to the
	 *            given resource.
	 *
	 * @exclude from published api.
	 */
	public ExFull(int nResId) {
		moItems.add(new ExErrItem(nResId));
	}

	/**
	 * Instantiates an ExFull object with a single
	 * exception in its collection,
	 * and having the given properties.
	 * @param nResId
	 *            set this object's exception resource property to the
	 *            given resource.
	 * @param bWasReported
	 *            set this object's WasResported property to the given boolean,
	 *            presumably to indicate whether this exception has been
	 *            reported to the user. Defaults to false.
	 *
	 * @exclude from published api.
	 */
	ExFull(int nResId, boolean bWasReported) {
		super(bWasReported);
		moItems.add(new ExErrItem(nResId));
	}

	/**
	 * Instantiates an ExFull object with a single
	 * exception in its collection, whose resource property is the resource
	 * property nResId whose error message text property is the
	 * MsgFormat's string property.
	 * <p>
	 * This c'tor is suitable only for resources whose associated
	 * message text <em>is</em> parameterized.
	 * @param nResId
	 *            set this object's exception resource property to the
	 *            given resource.
	 * @param oFormat
	 *            a Formatted message string.
	 * @param bWasReported
	 *            set this object's WasResported property to the given boolean,
	 *            presumably to indicate whether this exception has been
	 *            reported to the user. Defaults to false.
	 *
	 * @exclude from published api.
	 */
	ExFull(int nResId, MsgFormatPos oFormat, boolean bWasReported) {
		super(bWasReported);
		MsgFormatPos oFmt = new MsgFormatPos(oFormat);
		ExErrItem oItem = new ExErrItem();
		String sFmt = oFmt.toString();
		oItem.text(sFmt);
		oItem.resId(nResId);
		oItem.formatObject(oFormat);
		moItems.add(oItem);
	}

	/**
	 * Constructor that accepts a positioned message format string with one
	 * parameter.
	 * @param nResId
	 *            resource id of the message
	 * @param p1
	 *            the single string parameter.
	 *
	 * @exclude from published api.
	 */
	public ExFull(int nResId, String p1) {
		MsgFormatPos msg = new MsgFormatPos(nResId, p1);
		ExErrItem oItem = new ExErrItem();
		String sFmt = msg.toString();
		oItem.text(sFmt);
		oItem.resId(nResId);
		oItem.formatObject(msg);
		moItems.add(oItem);
	}

	/**
	 * Constructor with resource ID and final string for a single error
	 * item.
	 * <p>
	 * In the C++ implementation, the constructor with resource ID and
	 * string parameters does no message formatting; it simply treats the
	 * given string as the formatted one.  In Java, the constructor with
	 * similar signature uses the resource ID to get a formatting template
	 * and then formats the given string into that.  Enough Java code has
	 * been written based on that behaviour that it cannot be made
	 * consistent with C++ at this point.
	 * </p>
	 * <p>
	 * This constructor provides the C++ behaviour, which is also necessary
	 * in Java.  It reverses the order of the parameters to get a unique
	 * method signature.
	 * </p>
	 * @param text Final (unformatted) text for the single error item
	 * added to the exception.
	 * @param resId Resource ID.
	 * @exclude from published api.
	 */
	public ExFull(String text, int resId) {
		ExErrItem oItem = new ExErrItem();
		oItem.text (text);
		oItem.resId (resId);
		moItems.add (oItem);
	}

	/**
	 * Instantiate a ExFull object with a single
	 * exception
	 * in its collection.
	 * <p>
	 * This c'tor is suitable only for resources whose associated
	 * message text <em>is not</em> parameterized.
	 * @param nResId
	 *            set this object's exception resource property to the
	 *            given resource.
	 * @param sNewText
	 *            set this object's exception error message text
	 *			  property to the given string.
	 * @param bWasReported
	 *            set this object's WasResported property to the given boolean,
	 *            presumably to indicate whether this exception has been
	 *            reported to the user. Defaults to false.
	 */
//	private ExFull // TODO figure out if this constructor makes sense. If so,
//	// make it public. When would we supply the message text
//	// separate from the resid?
//	(int nResId, String sNewText, boolean bWasReported) {
//		super(bWasReported);
//		moItems.add(new ExErrItem(nResId, sNewText));
//	}

	/**
	 * Instantiate a ExFull object with a single
	 * exception in its collection, whose resource property is the MsgFormatPos's
	 * resource property, and whose error message text property is the
	 * MsgFormatPos's string property.
	 * <p>
	 * This c'tor is suitable only for resources whose associated message text
	 * <em>is</em> parameterized.
	 * @param oFormat
	 *            a Formatted message string
	 *
	 * @exclude from published api.
	 */
	public ExFull(MsgFormat oFormat) {
		ExErrItem oItem = new ExErrItem();
		oItem.text(oFormat.toString());
		oItem.resId(oFormat.resId());
		oItem.formatObject(oFormat);
		moItems.add(oItem);
	}

	/**
	 * Instantiate a ExFull object with a single
	 * exception in its collection, whose resource property is the MsgFormatPos's
	 * resource property, and whose error message text property is the
	 * MsgFormatPos's string property.
	 * <p>
	 * This c'tor is suitable only for resources whose associated message text
	 * <em>is</em> parameterized.
	 * @param oFormat
	 *            a Formatted message string
	 *
	 * @exclude from published api.
	 */
	public ExFull(MsgFormatPos oFormat) {
		ExErrItem oItem = new ExErrItem();
		oItem.text(oFormat.toString());
		oItem.resId(oFormat.resId());
		oItem.formatObject(oFormat);
		moItems.add(oItem);
	}

	/**
	 * Counts of the number of exceptions in this object's
	 * collection.
	 * @return the count.
	 */
	public int count() {
		return moItems.size();
	}

	/**
	 * Gets the resource property of the first exception in this
	 * object's collection.
	 * @return the resource.
	 *
	 * @exclude from published api.
	 */
	public int firstResId() {
		if (moItems.size() > 0)
			return moItems.get(0).resId();

		return 0;
	}

	/**
	 * Gets the resource property of the n'th exception in this
	 * object's collection.
	 * @param n
	 *            the index of the zero-based n'th item.
	 * @return the resource.
	 *
	 * @exclude from published api.
	 */
	public int getResId(int n) {
		if ((n > (count() - 1)) || (moItems.size() < 1))
			return 0;
		return moItems.get(n).resId();
	}

	/**
	 * Determines if this object's collection of exceptions
	 * for a resource property that matches the given resource.
	 * @param nResId
	 *            the resource to search for.
	 * @return boolean true if found, and false if not.
	 *
	 * @exclude from published api.
	 */
	public boolean hasResId(int nResId) {
		for (int i = 0; i < moItems.size(); i++) {
			if (moItems.get(i).resId() == nResId)
				return true;
		}
		return false;
	}

	/**
	 * Inserts the given ExFull's collection of exceptions into this
	 * object's collection.
	 * @param source
	 *            the ExFull object to copy from.
	 * @param bAppend
	 *            boolean true to append at the end of this object's collection,
	 *            and false to insert at the beginning of this object's
	 *            collection. The default is to append at the end.
	 */
	public void insert(ExFull source, boolean bAppend /* =true */) {
		for (int i = 0; i < source.moItems.size(); i++) {
			if (bAppend)
				moItems.add(source.moItems.get(i));
			else
				moItems.add(0, source.moItems.get(i));
		}
	}

	/**
	 * References the <i>n</i>'th exception in this object's
	 * collection.
	 * @param n
	 *            the index of zero-based exception to return.
	 * 			  Must be in the range 0 to Count() - 1.
	 * @return the <i>n</i>'th exception.
	 *
	 * @exclude from published api.
	 */
	public ExErrItem item(int n) {
		return moItems.get(n);
	}

	/**
	 * Resolves this object's collection of exceptions,
	 * ensuring that any resource text associated with each of the
	 * exceptions' resource property has been loaded.
	 */
	public void resolve() {
		for (int i = 0; i < moItems.size(); i++)
			moItems.get(i).resolve();
	}

	/**
	 * Concatenates all the error message text properties from this object's
	 * collection of exceptions.
	 * @return the concatenated error message texts. A newline character is
	 * appended to each error message texts.
	 */
	public String toString() {
		if (moItems.size() == 0)
			return "";
		resolve(); // Make sure we've looked up all the resource Id's
		StringBuilder sReturn = new StringBuilder();
		for (int i = 0, n = moItems.size(); i < n; i++) {
			sReturn.append(moItems.get(i).text());
			// MPT: just reset to 0 if want to restore previous behaviour.
			// It seems most tests prefer the previous (mis)behaviour.
			// if (i < n - 1) // Only separate items with NLs!
			sReturn.append('\n');
		}
		return sReturn.toString();
	}
}
