/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.ut;

import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;


/**
 * A class to represent any of the locales known to XFA.
 * 
 * @author Mike P. Tardif
 */
public final class LcLocale {

    /*
     * Locale names who's value is the comprised of the ISO 139-1 language name,
     * optionally followed by, the ISO 3166-1 country name, optionally followed
     * by, a locale modifier, each separated by an underscore. For example
     * LcLocale.Arabic_UAE is "ar_AE".
     */

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic = "ar";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_UAE = "ar_AE";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Bahrain = "ar_BH";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Algeria = "ar_DZ";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Egypt = "ar_EG";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Iraq = "ar_IQ";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Jordan = "ar_JO";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Kuwait = "ar_KW";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Lebanon = "ar_LB";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Libya = "ar_LY";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Morocco = "ar_MA";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Oman = "ar_OM";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Qatar = "ar_QA";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_SaudiArabia = "ar_SA";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Sudan = "ar_SD";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Syria = "ar_SY";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Tunisia = "ar_TN";

	/**
	 * @exclude from published api.
	 */
    public static final String Arabic_Yemen = "ar_YE";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani = "az";

	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani_Azerbaijan = "az_AZ";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani_Cyrillic = "az_Cyrl";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani_Cyrillic_Azerbaijan = "az_Cyrl_AZ";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani_Latin = "az_Latn";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Azerbaijani_Latin_Azerbaijan = "az_Latn_AZ";

	/**
	 * @exclude from published api.
	 */
    public static final String Byelorussian = "be";

	/**
	 * @exclude from published api.
	 */
    public static final String Byelorussian_Belarus = "be_BY";

	/**
	 * @exclude from published api.
	 */
    public static final String Bulgarian = "bg";

	/**
	 * @exclude from published api.
	 */
    public static final String Bulgarian_Bulgaria = "bg_BG";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Bosnian = "bs";

	/**
	 * @exclude from published api.
	 */
    public static final String Bosnian_BosniaHerzegovina = "bs_BA";
    
	/**
	 * @exclude from published api.
	 */
    public static final String C = "c";

	/**
	 * @exclude from published api.
	 */
    public static final String Catalan = "ca";

	/**
	 * @exclude from published api.
	 */
    public static final String Catalan_Spain = "ca_ES";

	/**
	 * @exclude from published api.
	 */
    public static final String Czech = "cs";

	/**
	 * @exclude from published api.
	 */
    public static final String Czech_CzechRepublic = "cs_CZ";

	/**
	 * @exclude from published api.
	 */
    public static final String Danish = "da";

	/**
	 * @exclude from published api.
	 */
    public static final String Danish_Denmark = "da_DK";

	/**
	 * @exclude from published api.
	 */
    public static final String German = "de";

	/**
	 * @exclude from published api.
	 */
    public static final String German_Austria = "de_AT";
    
	/**
	 * @exclude from published api.
	 */
    public static final String German_Belgium = "de_BE";

	/**
	 * @exclude from published api.
	 */
    public static final String German_Switzerland = "de_CH";

	/**
	 * @exclude from published api.
	 */
    public static final String German_Germany = "de_DE";

	/**
	 * @exclude from published api.
	 */
    public static final String German_Liechtenstein = "de_LI";

	/**
	 * @exclude from published api.
	 */
    public static final String German_Luxembourg = "de_LU";

	/**
	 * @exclude from published api.
	 */
    public static final String Greek = "el";

	/**
	 * @exclude from published api.
	 */
    public static final String Greek_Greece = "el_GR";

	/**
	 * @exclude from published api.
	 */
    public static final String English = "en";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Australia = "en_AU";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Belgium = "en_BE";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Canada = "en_CA";

	/**
	 * @exclude from published api.
	 */
    public static final String English_UK = "en_GB";

	/**
	 * @exclude from published api.
	 */
    public static final String English_UK_Euro = "en_GB_EURO";

	/**
	 * @exclude from published api.
	 */
    public static final String English_HongKong = "en_HK";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Ireland = "en_IE";

	/**
	 * @exclude from published api.
	 */
    public static final String English_India = "en_IN";

	/**
	 * @exclude from published api.
	 */
    public static final String English_NewZealand = "en_NZ";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Philippines = "en_PH";

	/**
	 * @exclude from published api.
	 */
    public static final String English_Singapore = "en_SG";
    
	/**
	 * @exclude from published api.
	 */
    public static final String English_US = "en_US";

	/**
	 * @exclude from published api.
	 */
    public static final String English_US_Posix = "en_US_POSIX";

	/**
	 * @exclude from published api.
	 */
    public static final String English_VirginIslands = "en_VI";

	/**
	 * @exclude from published api.
	 */
    public static final String English_SouthAfrica = "en_ZA";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish = "es";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Argentina = "es_AR";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Bolivia = "es_BO";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Chile = "es_CL";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Colombia = "es_CO";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_CostaRica = "es_CR";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_DominicanRepublic = "es_DO";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Ecuador = "es_EC";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Spain = "es_ES";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Guatemala = "es_GT";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Honduras = "es_HN";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Mexico = "es_MX";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Nicaragua = "es_NI";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Panama = "es_PA";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Peru = "es_PE";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_PuertoRico = "es_PR";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Paraguay = "es_PY";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_ElSalvador = "es_SV";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_US = "es_US";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Uruguay = "es_UY";

	/**
	 * @exclude from published api.
	 */
    public static final String Spanish_Venezuela = "es_VE";

	/**
	 * @exclude from published api.
	 */
    public static final String Estonian = "et";

	/**
	 * @exclude from published api.
	 */
    public static final String Estonian_Estonia = "et_EE";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Basque = "eu";

    /**
	 * @exclude from published api.
	 */
    public static final String Basque_Spain = "eu_ES";

    /**
	 * @exclude from published api.
	 */
    public static final String Persian = "fa";

	/**
	 * @exclude from published api.
	 */
    public static final String Persian_Iran = "fa_IR";

	/**
	 * @exclude from published api.
	 */
    public static final String Finnish = "fi";

	/**
	 * @exclude from published api.
	 */
    public static final String Finnish_Finland = "fi_FI";

	/**
	 * @exclude from published api.
	 */
    public static final String French = "fr";

	/**
	 * @exclude from published api.
	 */
    public static final String French_Belgium = "fr_BE";

	/**
	 * @exclude from published api.
	 */
    public static final String French_Canada = "fr_CA";

	/**
	 * @exclude from published api.
	 */
    public static final String French_Switzerland = "fr_CH";

	/**
	 * @exclude from published api.
	 */
    public static final String French_France = "fr_FR";

	/**
	 * @exclude from published api.
	 */
    public static final String French_Luxembourg = "fr_LU";

	/**
	 * @exclude from published api.
	 */
    public static final String Hebrew = "he";

	/**
	 * @exclude from published api.
	 */
    public static final String Hebrew_Israel = "he_IL";

	/**
	 * @exclude from published api.
	 */
    public static final String Hindi = "hi";

	/**
	 * @exclude from published api.
	 */
    public static final String Hindi_India = "hi_IN";

	/**
	 * @exclude from published api.
	 */
    public static final String Croatian = "hr";

	/**
	 * @exclude from published api.
	 */
    public static final String Croatian_Croatia = "hr_HR";

	/**
	 * @exclude from published api.
	 */
    public static final String Hungarian = "hu";

	/**
	 * @exclude from published api.
	 */
    public static final String Hungarian_Hungary = "hu_HU";

	/**
	 * @exclude from published api.
	 */
    public static final String Armenian = "hy";

	/**
	 * @exclude from published api.
	 */
    public static final String Armenian_Armenia = "hy_AM";

	/**
	 * @exclude from published api.
	 */
    public static final String Indonesian = "id";

	/**
	 * @exclude from published api.
	 */
    public static final String Indonesian_Indonesia = "id_ID";

	/**
	 * @exclude from published api.
	 */
    public static final String Icelandic = "is";

	/**
	 * @exclude from published api.
	 */
    public static final String Icelandic_Iceland = "is_IS";

	/**
	 * @exclude from published api.
	 */
    public static final String Italian = "it";

	/**
	 * @exclude from published api.
	 */
    public static final String Italian_Switzerland = "it_CH";

	/**
	 * @exclude from published api.
	 */
    public static final String Italian_Italy = "it_IT";

	/**
	 * @exclude from published api.
	 */
    public static final String Japanese = "ja";

	/**
	 * @exclude from published api.
	 */
    public static final String Japanese_Japan = "ja_JP";

	/**
	 * @exclude from published api.
	 */
    public static final String Kazakh = "kk";

	/**
	 * @exclude from published api.
	 */
    public static final String Kazakh_Kazakhstan= "kk_KZ";

	/**
	 * @exclude from published api.
	 */
    public static final String Khmer = "km";

	/**
	 * @exclude from published api.
	 */
    public static final String Khmer_Cambodia = "km_KH";

	/**
	 * @exclude from published api.
	 */
    public static final String Korean = "ko";

	/**
	 * @exclude from published api.
	 */
    public static final String Korean_Korea = "ko_KR";

	/**
	 * @exclude from published api.
	 */
    public static final String Korean_Korea_Hani = "ko_KR_HANI";

	/**
	 * @exclude from published api.
	 */
    public static final String Lao = "lo";

	/**
	 * @exclude from published api.
	 */
    public static final String Lao_Laos= "lo_LA";

	/**
	 * @exclude from published api.
	 */
    public static final String Lithuanian = "lt";

	/**
	 * @exclude from published api.
	 */
    public static final String Lithuanian_Lithuania = "lt_LT";

	/**
	 * @exclude from published api.
	 */
    public static final String Latvian = "lv";

	/**
	 * @exclude from published api.
	 */
    public static final String Latvian_Latvia = "lv_LV";

	/**
	 * @exclude from published api.
	 */
    public static final String Macedonian = "mk";

	/**
	 * @exclude from published api.
	 */
    public static final String Macedonian_Macedonia = "mk_MK";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Malay = "ms";

	/**
	 * @exclude from published api.
	 */
    public static final String Malay_Malaysia = "ms_MY";

	/**
	 * @exclude from published api.
	 */
    public static final String Norwegian_Bokmal = "nb";

	/**
	 * @exclude from published api.
	 */
    public static final String Norwegian_Bokmal_Norway = "nb_NO";

	/**
	 * @exclude from published api.
	 */
    public static final String Dutch = "nl";

	/**
	 * @exclude from published api.
	 */
    public static final String Dutch_Belgium = "nl_BE";

	/**
	 * @exclude from published api.
	 */
    public static final String Dutch_Netherlands = "nl_NL";

	/**
	 * @exclude from published api.
	 */
    public static final String Norwegian_Nynorsk = "nn";

	/**
	 * @exclude from published api.
	 */
    public static final String Norwegian_Nynorsk_Norway = "nn_NO";

	/**
	 * @exclude from published api.
	 */
    public static final String Polish = "pl";

	/**
	 * @exclude from published api.
	 */
    public static final String Polish_Poland = "pl_PL";

	/**
	 * @exclude from published api.
	 */
    public static final String Portuguese = "pt";

	/**
	 * @exclude from published api.
	 */
    public static final String Portuguese_Brazil = "pt_BR";

	/**
	 * @exclude from published api.
	 */
    public static final String Portuguese_Portugal = "pt_PT";

	/**
	 * @exclude from published api.
	 */
    public static final String Romanian = "ro";

	/**
	 * @exclude from published api.
	 */
    public static final String Romanian_Moldova = "ro_MD";

	/**
	 * @exclude from published api.
	 */
    public static final String Romanian_Romania = "ro_RO";

	/**
	 * @exclude from published api.
	 */
    public static final String Russian = "ru";

	/**
	 * @exclude from published api.
	 */
    public static final String Russian_Moldova = "ru_MD";

	/**
	 * @exclude from published api.
	 */
    public static final String Russian_Russia = "ru_RU";

	/**
	 * @exclude from published api.
	 */
    public static final String Russian_Ukraine = "ru_UA";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbo_Croatian = "sh";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbo_Croatian_BosniaHerzegovina = "sh_BA";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbo_Croatian_SerbiaMontenegro = "sh_CS";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbo_Croatian_Croatia = "sh_HR";

	/**
	 * @exclude from published api.
	 */
    public static final String Slovak = "sk";

	/**
	 * @exclude from published api.
	 */
    public static final String Slovak_Slovakia = "sk_SK";

	/**
	 * @exclude from published api.
	 */
    public static final String Slovenian = "sl";

	/**
	 * @exclude from published api.
	 */
    public static final String Slovenian_Slovenia = "sl_SI";

	/**
	 * @exclude from published api.
	 */
    public static final String Albanian = "sq";

	/**
	 * @exclude from published api.
	 */
    public static final String Albanian_Albania = "sq_AL";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbian = "sr";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbian_Yugoslavia = "sr_CS";
    
	/**
	 * @exclude from published api.
	 */
    public static final String Serbian_Cyrillic = "sr_Cyrl";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbian_Cyrillic_SerbiaMontenegro = "sr_Cyrl_CS";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbian_Latin = "sr_Latn";

	/**
	 * @exclude from published api.
	 */
    public static final String Serbian_Latin_SerbiaMontenegro = "sr_Latn_CS";

	/**
	 * @exclude from published api.
	 */
    public static final String Swedish = "sv";

	/**
	 * @exclude from published api.
	 */
    public static final String Swedish_Finland = "sv_FI";

	/**
	 * @exclude from published api.
	 */
    public static final String Swedish_Sweden = "sv_SE";

	/**
	 * @exclude from published api.
	 */
    public static final String Thai = "th";

	/**
	 * @exclude from published api.
	 */
    public static final String Thai_Thailand = "th_TH";

	/**
	 * @exclude from published api.
	 */
    public static final String Thai_Thailand_Traditional = "th_TH_TH";

    /**
	 * @exclude from published api.
	 */
    public static final String Tagalog = "tl";

    /**
	 * @exclude from published api.
	 */
    public static final String Tagalog_Philippines = "tl_PH";

    /**
	 * @exclude from published api.
	 */
    public static final String Turkish = "tr";

	/**
	 * @exclude from published api.
	 */
    public static final String Turkish_Turkey = "tr_TR";

	/**
	 * @exclude from published api.
	 */
    public static final String Ukrainian = "uk";

	/**
	 * @exclude from published api.
	 */
    public static final String Ukrainian_Ukraine = "uk_UA";

	/**
	 * @exclude from published api.
	 */
    public static final String Vietnamese = "vi";

	/**
	 * @exclude from published api.
	 */
    public static final String Vietnamese_Vietnam = "vi_VN";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese = "zh";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_China = "zh_CN";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_HongKong = "zh_HK";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_Simplified = "zh_Hans";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_Traditional = "zh_Hant";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_Singapore = "zh_SG";

	/**
	 * @exclude from published api.
	 */
    public static final String Chinese_Taiwan = "zh_TW";

    /**
     * The default XFA locale name: <b>en_US</b>.
     */
    public static final String DEFAULT_LOCALE = English_US;

    private final static int MASK_ARABIC                = 0x0001;
    private final static int MASK_BIDI                  = 0x0002;
    private final static int MASK_CHINESE               = 0x0004;
    private final static int MASK_CHINESE_SIMPLIFIED    = 0x0008;
    private final static int MASK_CHINESE_TRADITIONAL   = 0x0010;
    private final static int MASK_DICTIONARY            = 0x0020;
    private final static int MASK_IDEOGRAPHIC           = 0x0040;
    private final static int MASK_JAPANESE              = 0x0080;
    private final static int MASK_KOREAN                = 0x0100;
    private final static int MASK_KOREAN_HANI           = 0x0200;
    private final static int MASK_THAI                  = 0x0400;

    private static class LocaleDesc {
        final String mLocName;
        final String mDescName;
//      final String mScript;
        final String mCharSet;
        final int mMask;

        LocaleDesc(String locName, String descName, /* String script, */ String charSet, int mask) {
            mLocName = locName;
            mDescName = descName;
//          mScript = script;
            mCharSet = charSet;
            mMask = mask;
        }

    }
    
    private final LocaleDesc mLocaleDesc;
    
    private static final LocaleDesc mInvalidLocale = new LocaleDesc("x", "XXX", /* "Zxxx", */ "ibm367", 0x0); 

    /*
     * Runtime (user-defined) store of locale descriptions.
     */
    private final static ThreadLocal<Map<String, LocaleDesc>> mRuntimeMap = new ThreadLocal<Map<String, LocaleDesc>>() {
    	
    	protected Map<String, LocaleDesc> initialValue() {
    		return new HashMap<String, LocaleDesc>();
    	}
    };

    /*
     * Static store of locale descriptions.
     */
    private static final Map<String, LocaleDesc> mStaticMap = new HashMap<String, LocaleDesc>(188) {
    	
    	private static final long serialVersionUID = 1L;

		private void add(LocaleDesc localeDesc) {
    		put(localeDesc.mLocName, localeDesc);
    	}
    	
    	{
	        add(new LocaleDesc(Arabic, "Arabic", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_UAE, "Arabic (U.A.E.)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Bahrain, "Arabic (Bahrain)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Algeria, "Arabic (Algeria)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Egypt, "Arabic (Egypt)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Iraq, "Arabic (Iraq)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Jordan, "Arabic (Jordan)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Kuwait, "Arabic (Kuwait)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Lebanon, "Arabic (Lebanon)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Libya, "Arabic (Libya)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Morocco, "Arabic (Morocco)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Oman, "Arabic (Oman)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Qatar, "Arabic (Qatar)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_SaudiArabia, "Arabic (SaudiArabia)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Sudan, "Arabic (Sudan)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Syria, "Arabic (Syria)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Tunisia, "Arabic (Tunisia)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Arabic_Yemen, "Arabic (Yemen)", /* "Arab", */ "iso88596", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Azerbaijani_Cyrillic, "Azerbaijani_Cyrillic", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Azerbaijani_Cyrillic_Azerbaijan, "Azerbaijani_Cyrillic (Azerbaijan)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Azerbaijani_Latin, "Azerbaijani_Latin", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Azerbaijani_Latin_Azerbaijan, "Azerbaijani_Latin (Azerbaijan)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Byelorussian, "Belarusian", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Byelorussian_Belarus, "Belarusian (Belarus)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Bulgarian, "Bulgarian", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Bulgarian_Bulgaria, "Bulgarian (Bulgaria)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Bosnian, "Bosnian", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Bosnian_BosniaHerzegovina, "Bosnian (Bosnia And Herzegovina)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(C, "C", /* "Zzzz", */ "ibm367", 0x0));
	        add(new LocaleDesc(Catalan, "Catalan", /* "Latn", */ "iso885915", 0x0));
	        add(new LocaleDesc(Catalan_Spain, "Catalan (Spain)", /* "Latn", */ "iso885915", 0x0));
	        add(new LocaleDesc(Czech, "Czech", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Czech_CzechRepublic, "Czech (Czech Republic)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Danish, "Danish", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Danish_Denmark, "Danish (Denmark)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German, "German", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Austria, "German (Austria)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Belgium, "German (Belgium)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Switzerland, "German (Switzerland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Germany, "German (Germany)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Liechtenstein, "German (Liechtenstein)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(German_Luxembourg, "German (Luxembourg)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Greek, "Greek", /* "Latn", */ "iso88597", 0x0));
	        add(new LocaleDesc(Greek_Greece, "Greek (Greece)", /* "Latn", */ "iso88597", 0x0));
	        add(new LocaleDesc(English, "English", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_Australia, "English (Australia)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_Belgium, "English (Belgium)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_Canada, "English (Canada)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_UK, "English (United Kingdom)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_UK_Euro, "English (United Kingdom Euro)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_HongKong, "English (HongKong)", /* "Latn", */ "iso88591", 0x0)); 
	        add(new LocaleDesc(English_Ireland, "English (Ireland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_India, "English (India)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_NewZealand, "English (NewZealand)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_Philippines, "English (Philippines)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_US, "English (United States)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_VirginIslands, "English (U.S. Virgin Islands)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(English_SouthAfrica, "English (South Africa)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish, "Spanish (Spain)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Argentina, "Spanish (Argentina)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Bolivia, "Spanish (Bolivia)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Chile, "Spanish (Chile)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Colombia, "Spanish (Colombia)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_CostaRica, "Spanish (Costa Rica)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_DominicanRepublic, "Spanish (Dominican Republic)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Ecuador, "Spanish (Ecuador)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Spain, "Spanish (Spain)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Guatemala, "Spanish (Guatemala)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Honduras, "Spanish (Honduras)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Mexico, "Spanish (Mexico)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Nicaragua, "Spanish (Nicaragua)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Panama, "Spanish (Panama)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Peru, "Spanish (Peru)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_PuertoRico, "Spanish (Puerto Rico)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Paraguay, "Spanish (Paraguay)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_ElSalvador, "Spanish (El Salvador)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_US, "Spanish (United States)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Uruguay, "Spanish (Uruguay)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Spanish_Venezuela, "Spanish (Venezuela)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Estonian, "Estonian", /* "Latn", */ "utf8", 0x0));
	        add(new LocaleDesc(Estonian_Estonia, "Estonian (Estonia)", /* "Latn", */ "utf8", 0x0));
	        add(new LocaleDesc(Basque, "Basque", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Basque_Spain, "Basque (Spain)", /* "Latn", */ "iso88591", 0x0));
	    	add(new LocaleDesc(Persian, "Persian", /* "Arab", */ "iso88591", MASK_ARABIC | MASK_BIDI)); 
	        add(new LocaleDesc(Persian_Iran, "Persian (Iran)", /* "Arab", */ "iso88591", MASK_ARABIC | MASK_BIDI));
	        add(new LocaleDesc(Finnish, "Finnish", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Finnish_Finland, "Finnish (Finland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French, "French", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French_Belgium, "French (Belgium)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French_Canada, "French (Canada)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French_Switzerland, "French (Switzerland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French_France, "French (France)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(French_Luxembourg, "French (Luxembourg)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Hebrew, "Hebrew", /* "Hebr", */ "iso88598", MASK_BIDI));
	        add(new LocaleDesc(Hebrew_Israel, "Hebrew (Israel)", /* "Hebr", */ "iso88598", MASK_BIDI));
	        add(new LocaleDesc(Hindi, "Hindi", /* "Deva", */ "utf8", 0x0));
	        add(new LocaleDesc(Hindi_India, "Hindi (India)", /* "Deva", */ "utf8", 0x0));
	        add(new LocaleDesc(Croatian, "Croatian", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Croatian_Croatia, "Croatian (Croatia)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Hungarian, "Hungarian", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Hungarian_Hungary, "Hungarian (Hungary)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Armenian, "Armenian", /* "Armn", */ "utf8" , 0x0));
	        add(new LocaleDesc(Armenian_Armenia, "Armenian (Armenia)", /* "Armn", */ "utf8", 0x0)); 
	        add(new LocaleDesc(Indonesian, "Indonesian", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Indonesian_Indonesia, "Indonesian (Indonesia)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Icelandic, "Icelandic", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Icelandic_Iceland, "Icelandic (Iceland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Italian, "Italian", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Italian_Switzerland, "Italian (Switzerland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Italian_Italy, "Italian (Italy)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Japanese, "Japanese", /* "Hani", */ "shift_jis", MASK_JAPANESE | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Japanese_Japan, "Japanese (Japan)", /* "Hani", */ "shift-jis", MASK_JAPANESE | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Kazakh, "Kazakh", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Kazakh_Kazakhstan, "Kazakh-Kazakhstan", /* "Cyrl", */ "iso88595", 0x0)); 
	        add(new LocaleDesc(Khmer, "Khmer", /* "Khmr", */ "utf8", 0x0));
	        add(new LocaleDesc(Khmer_Cambodia, "Khmer (Cambodia)", /* "Khmr", */ "utf8", 0x0)); 
	        add(new LocaleDesc(Korean, "Korean", /* "Hang", */ "ksc_5601", MASK_KOREAN | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Korean_Korea, "Korean (Korea)", /* "Hang", */ "ksc-5601", MASK_KOREAN | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Korean_Korea_Hani, "Korean_Hanja (Korea)", /* "Hang", */ "ksc-5601", MASK_KOREAN | MASK_KOREAN_HANI | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Lao, "Lao", /* "Laoo", */ "utf8", 0x0));
	        add(new LocaleDesc(Lao_Laos, "Lao (Laos)", /* "Laoo", */ "utf8", 0x0)); 
	        add(new LocaleDesc(Lithuanian, "Lithuanian", /* "Latn", */ "iso885913", 0x0));
	        add(new LocaleDesc(Lithuanian_Lithuania, "Lithuanian (Lithuania)", /* "Latn", */ "iso885913", 0x0));
	        add(new LocaleDesc(Latvian, "Latvian", /* "Latn", */ "iso885913", 0x0));
	        add(new LocaleDesc(Latvian_Latvia, "Latvian (Latvia)", /* "Latn", */ "iso885913", 0x0));
	        add(new LocaleDesc(Macedonian, "Macedonian", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Macedonian_Macedonia, "Macedonian (Macedonia)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Malay, "Malay", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Malay_Malaysia, "Malay (Malaysia)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Norwegian_Bokmal, "Norwegian_Bokm�l", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Norwegian_Bokmal_Norway, "Norwegian_Bokm�l (Norway)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Dutch, "Dutch", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Dutch_Belgium, "Dutch (Belgium)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Dutch_Netherlands, "Dutch (Netherlands)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Norwegian_Nynorsk, "Norwegian_Nynorsk", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Norwegian_Nynorsk_Norway, "Norwegian_Nynorsk (Norway)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc("no", "Norwegian", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc("no_NO", "Norwegian (Norway)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc("no_NO_NY", "Norwegian (Norway Nynorsk)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Polish, "Polish", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Polish_Poland, "Polish (Poland)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Portuguese, "Portuguese", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Portuguese_Brazil, "Portuguese (Brazil)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Portuguese_Portugal, "Portuguese (Portugal)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Romanian, "Romanian", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Romanian_Moldova, "Romanian (Moldova)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Romanian_Romania, "Romanian (Romania)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Russian, "Russian", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Russian_Moldova, "Russian (Moldova)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Russian_Russia, "Russian (Russia)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Russian_Ukraine, "Russian (Ukraine)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Serbo_Croatian, "Serbo (Croatian)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Serbo_Croatian_BosniaHerzegovina, "Serbo (Croatian Bosnia And Herzegovina)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Serbo_Croatian_SerbiaMontenegro, "Serbo (Croatian Serbia And Montenegro)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Serbo_Croatian_Croatia, "Serbo (Croatian Croatia)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Slovak, "Slovak", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Slovak_Slovakia, "Slovak (Slovakia)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Slovenian, "Slovenian", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Slovenian_Slovenia, "Slovenian (Slovenia)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Albanian, "Albanian", /* "Latn", */ "iso885915", 0x0));
	        add(new LocaleDesc(Albanian_Albania, "Albanian (Albania)", /* "Latn", */ "iso885915", 0x0));
	        add(new LocaleDesc(Serbian, "Serbian", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Serbian_Yugoslavia, "Serbian (Yugoslavia)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Serbian_Cyrillic, "Serbian_Cyrillic", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Serbian_Cyrillic_SerbiaMontenegro, "Serbian_Cyrillic (Serbia And Montenegro)", /* "Cyrl", */ "iso88595", 0x0));
	        add(new LocaleDesc(Serbian_Latin, "Serbian_Latin", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Serbian_Latin_SerbiaMontenegro, "Serbian_Latin (Serbia And Montenegro)", /* "Latn", */ "iso88592", 0x0));
	        add(new LocaleDesc(Swedish, "Swedish", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Swedish_Finland, "Swedish (Finland)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Swedish_Sweden, "Swedish (Sweden)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Thai, /* "Thai", */ "Thai", "ibm874", MASK_THAI | MASK_DICTIONARY));
	        add(new LocaleDesc(Thai_Thailand, "Thai (Thailand)", /* "Thai", */ "ibm874", MASK_THAI | MASK_DICTIONARY));
	        add(new LocaleDesc(Thai_Thailand_Traditional, "Thai (Thailand Traditional)", /* "Thai", */ "ibm874", MASK_THAI | MASK_DICTIONARY));
	        add(new LocaleDesc(Tagalog, "Tagalog", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Tagalog_Philippines, "Tagalog (Philippines)", /* "Latn", */ "iso88591", 0x0));
	        add(new LocaleDesc(Turkish, "Turkish", /* "Latn", */ "iso88599", 0x0));
	        add(new LocaleDesc(Turkish_Turkey, "Turkish (Turkey)", /* "Latn", */ "iso88599", 0x0));
	        add(new LocaleDesc(Ukrainian, "Ukrainian", /* "Cyrl", */ "ibm1124", 0x0));
	        add(new LocaleDesc(Ukrainian_Ukraine, "Ukrainian (Ukraine)", /* "Cyrl", */ "ibm1124", 0x0));
	        add(new LocaleDesc(Vietnamese, "Vietnamese", /* "Latn", */ "ibm1258", 0x0));
	        add(new LocaleDesc(Vietnamese_Vietnam, "Vietnamese (Vietnam)", /* "Latn", */ "ibm1258", 0x0));
	        add(new LocaleDesc(Chinese, "Chinese", /* "Hani", */ "gbk", MASK_CHINESE | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_China, "Chinese (China)", /* "Hans", */ "gbk", MASK_CHINESE | MASK_CHINESE_SIMPLIFIED | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_Simplified, "Chinese_Hans", /* "Hans", */ "gbk", MASK_CHINESE | MASK_CHINESE_SIMPLIFIED | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_Traditional, "Chinese_Hant", /* "Hant", */ "big5", MASK_CHINESE | MASK_CHINESE_TRADITIONAL | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_HongKong, "Chinese (Hong Kong)", /* "Hant", */ "hkscs-big5", MASK_CHINESE | MASK_CHINESE_TRADITIONAL | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_Singapore, "Chinese (Singapore)", /* "Hans", */ "big5", MASK_CHINESE | MASK_CHINESE_SIMPLIFIED | MASK_IDEOGRAPHIC));
	        add(new LocaleDesc(Chinese_Taiwan, "Chinese (Taiwan)", /* "Hant", */ "big5", MASK_CHINESE | MASK_CHINESE_TRADITIONAL | MASK_IDEOGRAPHIC));
    	}
    };


    /**
     * Instantiates an LcLocale with the given name. 
     * 
     * @param locale an ISO locale name.
     */
    public LcLocale(String locale) {
    	
    	LocaleDesc localeDesc = null;
    	
        if (!StringUtils.isEmpty(locale)) {
            String sNormalizedLocale = normalize(locale);
            
            localeDesc = searchMap(sNormalizedLocale);
            
            if (localeDesc == null)
            	localeDesc = mStaticMap.get(sNormalizedLocale);
            
    	    // if not found Then trim off any locale-qualifier and try again.
            if (localeDesc == null) {
                String sDequalifiedLocale = dequalify(sNormalizedLocale);
                if (!sNormalizedLocale.equals(sDequalifiedLocale)) {
                	
                	localeDesc = searchMap(sDequalifiedLocale);
                }
            }
        }
        
        mLocaleDesc = localeDesc == null ? mInvalidLocale : localeDesc;
    }

	/**
     * Normalizes locale name into xx_XX or xx_Xxxx_XX format.
	 * @exclude from published api.
     */
    public static String normalize(String locale) {
    	
        String sNormal = "";
        if (locale != null) {
            StringBuilder loc = new StringBuilder(locale);
            int pDash = loc.indexOf("-");
			//
			// Remap dashes to underscores.
			//
            while (pDash >= 0) {
                loc.setCharAt(pDash, '_');
				pDash = loc.indexOf("-");
			}
            int pUnderScore = loc.indexOf("_");
            //
            // Camelcase any script code. As per RFC 3066 standard,
            // the script code always consists of 4 characters and
            // the country code never consists of 4 characters.
            //
            if (pUnderScore >= 0) {
                int pNextUnderScore = loc.indexOf("_", pUnderScore + 1);
                if (pNextUnderScore >= 0
                                && pUnderScore + 5 == pNextUnderScore) {
                    int p = pUnderScore + 1;
                    for (int i = 0; i < 4; i++) {
                        char c = loc.charAt(p);
                        if (i == 0 && Character.isLowerCase(c))
                            loc.setCharAt(p, Character.toUpperCase(c));
                        else if (i > 0 && Character.isUpperCase(c))
                            loc.setCharAt(p, Character.toLowerCase(c));
                        p++;
                    }
                    pUnderScore += 5;
                }
                else if (pNextUnderScore < 0
                                    && pUnderScore + 5 == loc.length()) {
                    int p = pUnderScore + 1;
                    for (int i = 0; i < 4; i++) {
                        char c = loc.charAt(p);
                        if (i == 0 && Character.isLowerCase(c))
                            loc.setCharAt(p, Character.toUpperCase(c));
                        else if (i > 0 && Character.isUpperCase(c))
                            loc.setCharAt(p, Character.toLowerCase(c));
                        p++;
                    }
                    pUnderScore += 5;
                }
                else {
                    pUnderScore = 0;
                }
            }
            else {
                pUnderScore = 0;
                //
                // Downcase the language code.
                //
                for (int p = pUnderScore; p < loc.length(); p++) {
                	char c = loc.charAt(p);
                	if (Character.isUpperCase(c))
                        loc.setCharAt(p, Character.toLowerCase(c));
                }
            }
            //
            // Upcase the country code.
            //
            pUnderScore = loc.indexOf("_", pUnderScore);
            if (pUnderScore >= 0) {
                for (int p = pUnderScore + 1; p < loc.length(); p++) {
                    char c = loc.charAt(p);
                    if (Character.isLowerCase(c))
                        loc.setCharAt(p, Character.toUpperCase(c));
                    else if (c == '-')
                        loc.setCharAt(p, '_');
                }
            }
            sNormal = loc.toString();
        }
        return sNormal;
    }

    /*
     * Trim off any locale modifiers, qualifiers, or charset
     * from the given locale.
     */
    private static String dequalify(String locale) {
    	
    	final int length = locale.length();
    	
    	int dot = locale.indexOf('.');
		int at = locale.indexOf('@');
    	
    	int qualifier = Math.min(
    			dot == -1 ? length : dot,
    			at  == -1 ? length : at);
    	
    	return qualifier == length ? locale : locale.substring(0, qualifier);
    }

    /**
     * Determines if this object is a valid -- recognized locale.
     * 
     * @return boolean true if valid, and false otherwise.
     */
    public boolean isValid() {
        return getLocaleDesc() != mInvalidLocale;
    }

    /**
     * Determines if this object represents a Chinese locale.
     * 
     * @return boolean true if this locale is Chinese.
     * 
	 * @exclude from published api.
     */
    public boolean isChinese() {
        return (getLocaleDesc().mMask & MASK_CHINESE) != 0;
    }

    /**
     * Determines if this object represents a Simplified Chinese locale.
     * 
     * @return boolean true if this locale is Simplified Chinese.
     * 
	 * @exclude from published api.
     */
    public boolean isSimplifiedChinese() {
        LocaleDesc oLocaleDesc = getLocaleDesc();
        
        if (oLocaleDesc.mLocName.equals(Chinese_HongKong))
        	return true;
        
        return (oLocaleDesc.mMask & MASK_CHINESE_SIMPLIFIED) != 0;
    }

    /**
     * Determines if this object represents a Traditional Chinese
     * locale.
     * 
     * @return boolean true if this locale is Traditional Chinese.
     * 
	 * @exclude from published api.
     */
    public boolean isTraditionalChinese() {
        LocaleDesc oLocaleDesc = getLocaleDesc();
		
        // JavaPort: Hongkong is misqualified -- MASK_CHINESE_SIMPLIFIED -- but its more important to emulate broken C++ behaviour!
        if (oLocaleDesc.mLocName.equals(Chinese_HongKong))
        	return false;
        
        return (oLocaleDesc.mMask & MASK_CHINESE_TRADITIONAL) != 0;
    }

    /**
     * Determines if this object represents a Japanese locale.
     * 
     * @return boolean true if this locale is Japanese.
     * 
	 * @exclude from published api.
     */
    public boolean isJapanese() {
        return (getLocaleDesc().mMask & MASK_JAPANESE) != 0;
    }

    /**
     * Determines if this object represents a Korean_Hani locale.
     * 
     * @return boolean true if this locale is Korean_Hani.
     * 
	 * @exclude from published api.
     */
    public boolean isKoreanHani() {
        return (getLocaleDesc().mMask & MASK_KOREAN_HANI) != 0;
    }

    /**
     * Determines if this object represents a Korean locale.
     * 
     * @return boolean true if this locale is Korean.
     * 
	 * @exclude from published api.
     */
    public boolean isKorean() {
        return (getLocaleDesc().mMask & MASK_KOREAN) != 0;
    }

    /**
     * Determines if this object represents a Thai locale.
     * 
     * @return boolean true if this locale is Thai.
     * 
	 * @exclude from published api.
     */
    public boolean isThai() {
        return (getLocaleDesc().mMask & MASK_THAI) != 0;
    }

    /**
     * Determines if this object represents a locale were text is
     * displayed bidirectionally.
     * 
     * @return boolean true if this locale is bi-directional, and false
     *         otherwise.
     * 
	 * @exclude from published api.
     */
    public boolean isBIDI() {
    	return (getLocaleDesc().mMask & MASK_BIDI) != 0;
    }

    /**
     * Determines if this object represents an Arabic locale.
     * 
     * @return boolean true if this locale is Arabic.
     * 
	 * @exclude from published api.
     */
    public boolean isArabic() {
    	return (getLocaleDesc().mMask & MASK_ARABIC) != 0;
    }

    /**
     * Determines if this object represents a locale were text is
     * ideographic. Useful in classifying characters with ambiguous
     * East Asian width.
     * 
     * @return boolean true if this locale uses ideographic text, and
     * false otherwise.
     * 
	 * @exclude from published api.
     */
    public boolean isIdeographic() {
    	return (getLocaleDesc().mMask & MASK_IDEOGRAPHIC) != 0;
    }

    /**
     * Determines if this object represents a locale were
     * dictionary-based line breaking is needed.
     * 
     * @return boolean true if this locale is needs breaks, and
     * false otherwise.
     * 
	 * @exclude from published api.
     */
    public boolean needsDictionaryBreaking() {
    	return (getLocaleDesc().mMask & MASK_DICTIONARY) != 0;
    }

    /**
     * Gets this object's name.
     * 
     * @return the name.
     */
    public String getName() {
        return getLocaleDesc().mLocName;
    }

    /**
     * Gets this object's ISO name.
     * 
     * @return the ISO name.
     * 
	 * @exclude from published api.
     */
    public String getIsoName() {
        return getLocaleDesc().mLocName;
    }
    
    private LocaleDesc getLocaleDesc() {
    	return mLocaleDesc;
    }

    /**
     * Gets this object's character set.
     * 
     * @return the character set.
     * 
	 * @exclude from published api.
     */
    public String getCharSet() {
        return getLocaleDesc().mCharSet;
    }

    /**
     * Gets this object's descriptive name.
     * 
     * @return the descriptive name.
     */
    public String getDescName() {
        LocaleDesc oLocaleDesc = getLocaleDesc();
        Locale oLoc = new Locale(oLocaleDesc.mLocName);
        String sDesc = oLoc.getDisplayName();
        if (sDesc.length() == 0
                    || oLocaleDesc.mLocName.equals(normalize(sDesc)))
            sDesc = oLocaleDesc.mDescName;
        return sDesc;
    }

    /**
     * Compares the given LcLocale to this.
     * 
     * @param object the locale to compare with.
     * @return <code>true</code> when the locales compare identically, and <code>false</code>
     *         otherwise.
     * 
	 * @exclude from published api.
     */
    public boolean equals(Object object) {
    	
    	if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
    	LcLocale locale = (LcLocale) object;
    	
    	return getIsoName().equals(locale.getIsoName());
    }
    
	/*
     * Search the locale map for the given ISO 639 locale name.
     */
    private static LocaleDesc searchMap(String name) {
    	// If found in the runtime list Then use the runtime entry
        LocaleDesc localeDesc = mRuntimeMap.get().get(name);
        
        if (localeDesc == null)
        	localeDesc = mStaticMap.get(name);
        
        return localeDesc;
    }
    
	/**
	 * Returns a hash code value for the object. This method is unsupported.
	 * @exclude from published api.
     * 
	 * @exclude from published api.
	 */
	public int hashCode() {
		return getIsoName().hashCode();
	}

    /**
     * Gets the current locale's iso name.
     * 
     * @return the current locale's name.
     * 
	 * @exclude from published api.
     */
    public static String getLocale() {
        return Locale.getDefault().toString();
    }

    /**
     * Set the current default locale to the given iso locale.
     * 
     * @param localeName the new locale's name.
     * 
	 * @exclude from published api.
     */
    public static void setLocale(String localeName) {
        String language = null;
        String country = null;
        String variant = null;
        Locale locale = null;
        int split = localeName.indexOf('_');
        if (split < 0) {
            locale = new Locale(localeName);
        }
        else {
            country = localeName.substring(split + 1);
            language = localeName.substring(0, split);
            split = country.indexOf('_');
            if (split < 0) {
                locale = new Locale(language, country);
            }
            else {
                variant = country.substring(split + 1);
                country = country.substring(0, split);
                locale = new Locale(language, country, variant);
            }
        }
        Locale.setDefault(locale);
    }

    /**
     * Resets LcLocale's runtime map to its internal defaults.
     * 
	 * @exclude from published api.
     */
    public static void reset() {
        mRuntimeMap.get().clear();
    }

    /**
     * Updates LcLocale's runtime map with the given locale definition.
     * 
	 * @exclude from published api.
     */
    public static void update(String sLocale, String sDesc) {
        //
        // Normalize the locale name and extract any charset from that name.
        //
        String sLoc = normalize(sLocale);
        String sCharSet = "";
        int nDot = sLoc.indexOf('.');
        if (nDot >= 0) {
            sCharSet = sLoc.substring(nDot + 1);
            sLoc = sLoc.substring(0, nDot);
        }
        //
        // Classify locale based upon its name
        //
        int nMask = 0x0;
        if (sLoc.startsWith(Chinese)) {
            nMask |= MASK_CHINESE | MASK_IDEOGRAPHIC;
            if (sLoc.startsWith(Chinese_Simplified))
                nMask |= MASK_CHINESE_SIMPLIFIED;
            else if (sLoc.equals(Chinese_China))
                nMask |= MASK_CHINESE_SIMPLIFIED;
            else if (sLoc.equals(Chinese_Singapore))
                nMask |= MASK_CHINESE_SIMPLIFIED;
            else if (sLoc.startsWith(Chinese_Traditional))
                nMask |= MASK_CHINESE_TRADITIONAL;
            else if (sLoc.equals(Chinese_HongKong))
                nMask |= MASK_CHINESE_TRADITIONAL;
            else if (sLoc.equals(Chinese_Taiwan))
                nMask |= MASK_CHINESE_TRADITIONAL;
        }
        else if (sLoc.startsWith(Japanese)) {
            nMask |= MASK_JAPANESE | MASK_IDEOGRAPHIC;
        }
        else if (sLoc.startsWith(Korean)) {
            nMask |= MASK_KOREAN | MASK_IDEOGRAPHIC;
            if (sLoc.startsWith(Korean_Korea_Hani))
                nMask |= MASK_KOREAN_HANI;
        }
        else if (sLoc.startsWith(Thai)) {
            nMask |= MASK_THAI | MASK_DICTIONARY;
        }
        else if (sLoc.startsWith(Arabic)) {
            nMask |= MASK_ARABIC | MASK_BIDI;
        }
        else if (sLoc.startsWith(Persian)) {
            nMask |= MASK_ARABIC | MASK_BIDI;
        }
        else if (sLoc.startsWith(Hebrew)) {
            nMask |= MASK_BIDI;
        }
        LocaleDesc oLcDesc = new LocaleDesc(sLoc, sDesc, /* sScript, */ sCharSet, nMask);
        
        // Add the new locale description to the runtime map.
        mRuntimeMap.get().put(sLoc, oLcDesc);
    }

    /*
     * Get individual symbols for the current locale from the LcData database.
     */

    /**
     * Gets the current locale's currency symbol.
     * 
     * @return the currency symbol.
     * 
	 * @exclude from published api.
     */
    public static String currencySymbol() {
        return getLcData().getCurrencySymbol();
    }

    /**
     * Gets the current locale's negative sign symbol.
     * 
     * @return the negative sign.
     * 
	 * @exclude from published api.
     */
    public static String negativeSign() {
        return getLcData().getNegativeSymbol();
    }

    /**
     * Get the current locale's positive sign symbol.
     * 
     * @return the positive sign.
     * 
	 * @exclude from published api.
     */
    public static String positiveSign() {
        return getLcData().getPositiveSymbol();
    }

    /**
     * Gets the current locale's decimal point symbol.
     * 
     * @return the decimal point.
     * 
	 * @exclude from published api.
     */
    public static String decimalPoint() {
        return getLcData().getRadixSymbol();
    }

    /**
     * Gets the current locale's thousands separator symbol.
     * 
     * @return the thousands separator.
     * 
	 * @exclude from published api.
     */
    public static String thousandsSep() {
        return getLcData().getGroupingSymbol();
    }

    /**
     * Gets the current system locale's decimal point symbol.
     * 
     * @return the decimal point, as per the operating system's database,
     *         which may differ from this database. Nonetheless, this is
     *         important because it is what is used by the Java runtime.
     * 
	 * @exclude from published api.
     */
    public static String systemDecimalPoint() {
        // Watson 1352589 - Avoid using static variables because the locale
        // may get reset between the time the static variables are initialized
        // and the time this method gets called.
        // String tmp = String.format("%2.1f", 1.0);
        StringBuilder radix = new StringBuilder();
    	DecimalFormat f = (DecimalFormat) NumberFormat.getInstance();
        DecimalFormatSymbols s = f.getDecimalFormatSymbols();
        radix.append(s.getDecimalSeparator());
        return radix.toString();
    }

    /**
     * Gets the current system locale's decimal point symbol.
     * 
     * @return the decimal point, as per the operating system's database,
     *         which may differ from this database. Nonetheless, this is
     *         important because it is what is used by the Java runtime.
     * 
	 * @exclude from published api.
     */
    public static char systemDecimalPointAsChar() {
        // Watson 1352589 - Avoid using static variables because the locale
        // may get reset between the time the static variables are initialized
        // and the time this method gets called.
        // In JDK 1.5, use String tmp = String.format("%2.1f", 1.0);
    	DecimalFormat f = (DecimalFormat) NumberFormat.getInstance();
        DecimalFormatSymbols s = f.getDecimalFormatSymbols();
        return s.getDecimalSeparator();
    }

    /**
     * Returns the locale data for the ambient locale. First tries to create a
     * locale from the ambient locale name. If that fails, it creates a locale
     * based on CURRENT_LOCALE. It then creates an LcData instance for the
     * locale's ISO name.
     * 
     * @return LcData instance for the ambient/default locale.
     * 
	 * @exclude from published api.
     */
    private static LcData getLcData() {
        LcLocale oLocale = new LcLocale(LcLocale.getLocale());
        if (! oLocale.isValid()) {
            oLocale = new LcLocale(DEFAULT_LOCALE);
        }
        return new LcData(oLocale.getIsoName());
    }
}