// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

public final class XMPDateTimeUtils 
{
	/** The UTC TimeZone */
	private static final TimeZone UTC = TimeZone.getTimeZone("UTC");

	/**
	 * Obtain the current date and time.
	 *
	 * @return Returns The returned time is UTC, properly adjusted for the local time zone. The
	 *         resolution of the time is not guaranteed to be finer than seconds.
	 */
	public static XMPDateTime getCurrentDateTime()
	{
		return new XMPDateTime(new GregorianCalendar());
	}


	/**
	 * Set to the local time zone (local timezone is provided by the OS).
	 * if adjustTime is false no other existing time zone value is replaced,
	 * the other date/time fields are not adjusted in any way. If adjustTime is true, 
	 * the time is adjusted based on the local time zone.
	 *
	 * @param dateTime the <code>XMPDateTime</code> variable containing the value to be modified.
	 * @param adjustTime flag if the time should be adjusted based on the time
	 * @return Returns an updated <code>XMPDateTime</code>-object.
	 */
	public static XMPDateTime convertToLocalTimeZone(XMPDateTime dateTime, boolean adjustTime )
	{
		if (adjustTime)
		{
			long timeInMillis = dateTime.getCalendar().getTimeInMillis();
			// has automatically local timezone
			GregorianCalendar cal = new GregorianCalendar();
			cal.setTimeInMillis(timeInMillis);
			return new XMPDateTime(cal);
		}
		else
		{
			Calendar cal = dateTime.getCalendar();
			cal.setTimeZone(TimeZone.getDefault());
			return new XMPDateTime(cal);
		}
		
	}


	/**
	 * Make sure a time is UTC. If the time zone is not UTC, the time is
	 * adjusted and the time zone set to be UTC.
	 *
	 * @param dateTime
	 *            the <code>XMPDateTime</code> variable containing the time to
	 *            be modified.
	 * @return Returns an updated <code>XMPDateTime</code>-object.
	 */
	public static XMPDateTime convertToUTCTime(XMPDateTime dateTime)
	{
		long timeInMillis = dateTime.getCalendar().getTimeInMillis();
		GregorianCalendar cal = new GregorianCalendar(UTC);
		cal.setGregorianChange(new Date(Long.MIN_VALUE));
		cal.setTimeInMillis(timeInMillis);
		return new XMPDateTime(cal);
	}
}
