/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.util.HashMap;
import java.util.Map;

import com.day.cq.wcm.designimporter.ErrorCodes;
import com.day.cq.wcm.designimporter.util.TagUtils;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.ccil.cowan.tagsoup.AttributesImpl;
import org.xml.sax.Attributes;

import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * The title component tag handler is used to translate the title component div(or an equivalent markup)
 * into a CQ title component.
 *
 * <p>
 * This tag handler is registered against the markup of type < div|h1-6 data-cq-component="title" >
 * (Read as a div, h1, h2,.., h6 tag with the attribute data-cq-component equal to "title"). That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.TitleComponentTagHandlerFactory
 */
public class TitleComponentTagHandler extends AbstractTagHandler {

    private String resourceType;

    private String type;

    private String suffix;

    private int stackCounter = 0;

    @Override
    public void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        super.beginHandling(uri, localName, qName, atts);

        if (localName.matches("[hH][1-6]")) {
            AttributesImpl modAttibutes = new AttributesImpl(atts);
            int index = atts.getIndex("data-cq-component");
            if (index > -1) modAttibutes.removeAttribute(index);

            startElement(uri, localName, qName, modAttibutes);
        }
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        if (type == null) {
            if (localName.matches("[hH][1-6]")) {
                type = localName.toLowerCase();
            } else {
                throw new UnsupportedTagContentException(ErrorCodes.TITLE_COMPONENT_UNKNOWN_TAG);
            }
        } else {
            throw new UnsupportedTagContentException(ErrorCodes.TITLE_COMPONENT_ONE_HEADING_TAG);
        }

        stackCounter++;
    }

    @Override
    public void characters(char[] ch, int start, int length) throws DesignImportException {
        super.characters(ch, start, length);
        if (stackCounter == 0 && new String(ch, start, length).trim().length() > 0) // Only WS characters allowed. Any text is not
                                                                                    // tolerable.
            throw new UnsupportedTagContentException(
                    ErrorCodes.TITLE_EXACTLY_ONE_HEADING_TAG);
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        super.endElement(uri, localName, qName);
        stackCounter--;
    }

    @Override
    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        super.endHandling(uri, localName, qName);

        if (localName.matches("[hH][1-6]")) {
            endElement(uri, localName, qName);
        }

        Map<String, Object> base = new HashMap<String, Object>();
        base.put("jcr:title", htmlBuffer.toString());
        base.put("type", type);

        ValueMap properties = new ValueMapDecorator(base);

        suffix = designImporterContext.componentSuffixGenerator.getSuffix("title");
        PageComponent titleComponent = pageBuilder.createComponent(resourceType, properties, getNameHint());
        getPageComponents().add(titleComponent);

        // Reset the type after the request completes.
        type = null;
    }

    @Override
    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP)
            return true;
        else
            return false;
    }

    private String getNameHint() {
        return "title" + suffix;
    }

    @Override
    public Object getContent(HTMLContentType htmlContentType) {

        if (htmlContentType == HTMLContentType.MARKUP) {
            String cqIncludeJspTag = "<sling:include path=" + "\"" + getNameHint() + "\"" + "/>";

            return componentDivStartTag + cqIncludeJspTag + TagUtils.getMatchingEndTag(componentDivStartTag);

        }

        return null;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }

}
