/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image;

import java.awt.Paint;
import java.awt.Stroke;
import java.awt.geom.Rectangle2D;
import java.text.CharacterIterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.image.font.AbstractFont;
import com.day.image.font.FontListEntry;
import com.day.image.internal.font.FontHelper;

/**
 * The <code>Font</code> class is a wrapper for the
 * <code>com.day.image.Font</code> class which provides the functionality to
 * seamlessly support multiple font formats and sources. All API methods are
 * delegated to an instance of (an extension of) the
 * <code>com.day.image.Font</code> class which is setup in the constructor.
 * <p>
 * The constructor tries a list of configured font providers to get the
 * delegatee and falls back to the <code>com.day.image.Font</code> class if
 * none of the configured font providers can provide the desired font.
 * <p>
 * While the list of font providers and their order is configurable, the
 * fallback to the base class is hard coded and cannot be switched of. This way
 * it is guaranteed, that instances of this class always contain a valid
 * delegatee.
 * <p>
 * This class cannot be initialized if the {@link FontHelper} and/or the
 * system {@link com.day.cq.contentbus.Ticket} are not available.
 *
 * @version $Revision$, $Date: 2006-05-03 16:47:57 +0200 (Mit, 03 Mai
 *          2006) $
 * @author fmeschbe
 * @since degu
 * @audience wad
 */
public final class Font extends AbstractFont {

    /** default log */
    private final Logger log = LoggerFactory.getLogger(getClass());

    /** The delegatee font object */
    private final AbstractFont delegatee;

    // ---------- static method

    /**
     * Returns the list of available fonts known to all registered font
     * providers and the <code>com.day.image.Font</code> class.
     *
     * @return List of fonts on the platform
     */
    public static java.util.List<FontListEntry> getFontList() {
        return FontHelper.getInstance().getFontList();
    }

    /**
     * Creates an instance of the font wrapper with a delegatee provided by one
     * of the font providers or the base class.
     * <p>
     * The font providers are asked in configuration order for the given font.
     * The first provider not returning <code>null</code> is taken. If none of
     * the providers may provide the desired font, the constructor falls back to
     * asking the base class for the font instance.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file. In fact this MUST
     *            be the name of the Font family. That is "Times New Roman" and
     *            not "Times New Roman Bold" is expected.
     * @param size Size in points for the font to open
     * @param style Style flags for the new font
     * @see "The <code>java.awt.Font</code> javadoc page for details of this
     *      call."
     */
    public Font(String faceName, int size, int style) {

        log.debug("Font: Looking for {}/{}/{}", new Object[] { faceName, size,
            styleToDescription(style) });

        delegatee = FontHelper.getInstance().getFont(faceName, size, style);
        log.debug("Font: Using font {}", delegatee);
    }

    /**
     * Creates a font wrapper for the given font. This constructor is the same
     * as calling {@link #Font(String, int, int)} with a style of
     * <code>Font.PLAIN</code>.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file.
     * @param size Size in points for the font to open
     * @see "The <code>java.awt.Font</code> javadoc page for details of this
     *      call."
     * @see #Font(String, int, int)
     */
    public Font(String faceName, int size) {
        this(faceName, size, PLAIN);
    }

    /**
     * Calculate the bounding box of the text, if it would be rendered with the
     * rendering attributes given. The calculation will be done as if the text
     * would be rendered in the current font object.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the text
     * is broken into several lines, regardless of whether the text would be
     * broken because of the text box width setting.
     *
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param align alignment, rotation and TrueType attributes for the text.
     *            Use {@link #ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing. Use 0 as default value to not add
     *            additional space.
     * @param ls extra line spacing. Use 0 as default value to not add
     *            additional space.
     * @return returns a rectangle representing the bounding box, if the text
     *         would be rendered in this font using the given additional
     *         rendering attributes.
     */
    public Rectangle2D getTextExtent(int x, int y, int width, int height,
            String text, int align, double cs, int ls) {
        return delegatee.getTextExtent(x, y, width, height, text, align, cs, ls);
    }

    /**
     * Render the given text string in the given font to the <code>Layer</code>
     * using the attributes given. Use the default values given for unspecified
     * values.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the text
     * is broken into several lines, regardless of whether the text would be
     * broken because of the text box width setting.
     *
     * @param layer the layer to draw the text into
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param paint The <code>Paint</code> to use for the text drawing.
     * @param stroke The <code>Stroke</code> to use for the text drawing.
     * @param align alignment, rotation and TrueType attributes for the text.
     *            Use {@link Font#ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing. Use 0 as default value to not add
     *            additional space.
     * @param ls extra line spacing. Use 0 as default value to not add
     *            additional space.
     * @return the number of text lines drawn to the layer
     */
    public int drawText(Layer layer, int x, int y, int width, int height,
            String text, Paint paint, Stroke stroke, int align, double cs,
            int ls) {
        return delegatee.drawText(layer, x, y, width, height, text, paint,
            stroke, align, cs, ls);
    }

    /**
     * Returns the calculated font height in pixels.
     *
     * @return the calculated font height in pixels.
     */
    public double getHeight() {
        return delegatee.getHeight();
    }

    /**
     * Returns the ascent of this <code>Font</code> object, which is the
     * maximal value any glyph of this font ascends above the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public double getAscent() {
        return delegatee.getAscent();
    }

    /**
     * Returns the descent of this <code>Font</code> object, which is the
     * maximal value any glyph of this font descends below the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public double getDescent() {
        return delegatee.getDescent();
    }

    /**
     * Checks if this Font has a glyph for the specified character.
     *
     * @param c a unicode character code
     * @return <code>true</code> if this Font can display the character;
     *         <code>false</code> otherwise.
     */
    public boolean canDisplay(char c) {
        return delegatee.canDisplay(c);
    }

    /**
     * Indicates whether or not this Font can display the specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code . If this Font can display all characters,
     * -1 is returned.
     *
     * @param iter a CharacterIterator object
     * @param start the specified starting offset into the specified array of
     *            characters
     * @param limit the specified ending offset into the specified array of
     *            characters
     * @return an offset into the String object that can be displayed by this
     *         Font.
     */
    public int canDisplayUpTo(CharacterIterator iter, int start, int limit) {
        return delegatee.canDisplayUpTo(iter, start, limit);
    }

    /**
     * Indicates whether or not this Font can display a specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code. If the Font can display all characters, -1
     * is returned.
     *
     * @param str a String object
     * @return an offset into str that points to the first character in str that
     *         this Font cannot display; or -1 if this Font can display all
     *         characters in str.
     */
    public int canDisplayUpTo(String str) {
        return delegatee.canDisplayUpTo(str);
    }

    /**
     * Indicates whether or not this Font can display the characters in the
     * specified text starting at start and ending at limit. This method is a
     * convenience overload.
     *
     * @param text the specified array of characters
     * @param start the specified starting offset into the specified array of
     *            characters
     * @param limit the specified ending offset into the specified array of
     *            characters
     * @return an offset into text that points to the first character in text
     *         that this Font cannot display; or -1 if this Font can display all
     *         characters in text.
     */
    public int canDisplayUpTo(char[] text, int start, int limit) {
        return delegatee.canDisplayUpTo(text, start, limit);
    }

    /**
     * Returns a <code>String</code> representation of this object containing
     * the font name, the size and style flags.
     *
     * @return A <code>String</code> representation of this object.
     */
    public String toString() {
        return delegatee.toString();
    }

    @Override
    public java.awt.Font getAwtFont() {
        return delegatee.getAwtFont();
    }

}
