/*
 *
 *	File: AsciiEexecReader.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.type1;

import java.io.IOException;

import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.postscript.Reader;
import com.adobe.fontengine.font.postscript.Tokenizer;


/**
 * A filter to decrypt ascii eexec.
 *
 * <h4>Synchronization</h4>
 * 
 * This class is NOT threadsafe. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 */
final class AsciiEexecReader implements Reader {
    private int hiNib;
    private int cipherR;
    
    AsciiEexecReader() {
        hiNib = -1;
        cipherR = 55665;
    }
    
    private int lastRead;
    private int secondToLastRead;
    private int lastCipher;
    
    public int read(FontInputStream stream)
    	throws IOException, InvalidFontException
    {
        lastCipher = cipherR;
        
        secondToLastRead = stream.read();
		while (secondToLastRead != -1 && Tokenizer.digitValue((byte)secondToLastRead) > 15)
		{
		    secondToLastRead = stream.read();
		} 
		
		lastRead = stream.read();
		while (lastRead != -1 && Tokenizer.digitValue((byte)lastRead) > 15)
		{
		    lastRead = stream.read();
		} 
        
        if (secondToLastRead == -1 || lastRead == -1)
            throw new InvalidFontException("unexpected end of file");
        
        
        decryptByte((byte)secondToLastRead);     
        return decryptByte((byte)lastRead);
    }
    
    public void unreadLast(FontInputStream stream)
    	throws IOException
    {
        cipherR = lastCipher;
        stream.unread(lastRead);
        stream.unread(secondToLastRead);
    }

    
    
    public int decryptByte(byte inputByte)
    {
		if (hiNib == -1) {
		    hiNib = Tokenizer.digitValue(inputByte);
		    return -1;
		}
		else
		{
		    byte dst;
			byte cipher = (byte)(hiNib<<4 | Tokenizer.digitValue(inputByte));
			dst = (byte)(cipher ^ (cipherR>>8));
			cipherR = (((cipher & 0xff) + cipherR)*52845 + 22719)& 0xffff;
			hiNib = -1;
			return (dst&0xff);
		}

    }
    
    int decryptBuffer(byte[] buffer, int numBytesToDecrypt)
    {
        int dest=0;
        
        for (int i = 0; i < numBytesToDecrypt; ++i)
        {
            if (Tokenizer.digitValue(buffer[i]) > 15)
                continue;
            int j = decryptByte(buffer[i]);
            if (j != -1)
                buffer[dest++] = (byte)j;

        }
        
        return dest;
    }
}