/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2019 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
package com.adobe.granite.contexthub.api;

import java.util.Map;

import org.apache.sling.api.SlingHttpServletRequest;

public interface ContextHubCache {

    enum CodeType {
        KERNEL_JS("kernel.js"),
        UI_JS("ui.js"),
        KERNEL_MIN_JS("kernel.min.js"),
        UI_MIN_JS("ui.min.js");

        private final String name;

        CodeType(String name) {
            this.name = name;
        }

        public String fileName(){
            return name;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    /**
     * Retrieves the code for the particular contexthub configuration and corresponding to the given codeType from the cache.
     * The contexthub configuration, for which the code needs to be returned, is retrieved from the given request object.
     *
     * @param request The request object from which the contexthub configuration path is retrieved
     * @param codeType the type of code to return
     * @return {@code String} containing the code if the code is present in the cache<br/>
     *          {@code null} if code is not found in the cache
     */
    String getCode(SlingHttpServletRequest request, CodeType codeType);

    /**
     * Retrieves the metadata for the code stored in the cache - for the particular contexthub configuration and corresponding
     * to the given codeType.
     * The contexthub configuration, for which the metadata needs to be returned, is retrieved from the given request object.
     * @param request The request object from which the contexthub configuration path is retrieved
     * @param codeType the type of code corresponding to which the metadata should be returned
     * @return {@link Map} containing the metadata as key-value pairs if the code and its corresponding metadata is present
     *      in the cache <br/>
     *      The returned map would be empty if the code or its corresponding metadata is not found in the cache
     */
    Map getCodeMetadata(SlingHttpServletRequest request, CodeType codeType);

    /**
     * Stores the supplied code in the cache. The code is mapped and stored for the particular contexthub configuration
     * and corresponding to the given codeType.
     * The contexthub configuration, corresponding to which the code needs to be stored, is retrieved from the given request object.
     * @param request The request object from which the contexthub configuration path is retrieved
     * @param codeType The type of code corresponding to which the code needs to be stored.
     * @param code the code to store
     */
    void cacheCode(SlingHttpServletRequest request, CodeType codeType, String code);

    /**
     * Stores the supplied metadata in the cache. The metadata is mapped and stored for the particular contexthub
     * configuration and corresponding to the given codeType.
     * The contexthub configuration, corresponding to which the metadata needs to be stored, is retrieved from the given request object.
     * @param request The request object from which the contexthub configuration path is retrieved
     * @param codeType The type of code corresponding to which the metadata needs to stored
     * @param metadata The metadata to store
     */
    void setCodeMetadata(SlingHttpServletRequest request, CodeType codeType, Map metadata);
}
