/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2013 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletResponse;

import org.apache.sling.servlets.post.AbstractPostResponse;

import com.adobe.granite.i18n.LocaleUtil;
import com.adobe.granite.xss.XSSAPI;
import com.day.cq.i18n.I18n;

/**
 * The post response which produces HTML containing metadata, such as status
 * code, title, so that the client can parse it.
 */
public class HtmlResponse extends AbstractPostResponse {
    private static final String PN_DESCRIPTION = "description";

    private final XSSAPI xss;

    private final I18n i18n;

    private final Locale locale;

    private final List<Change> changes = new ArrayList<Change>();

    private final List<Link> links = new ArrayList<Link>();

    public HtmlResponse(@Nonnull XSSAPI xss, @Nonnull I18n i18n, @Nonnull Locale locale) {
        this.xss = xss;
        this.i18n = i18n;
        this.locale = locale;
    }

    @Override
    public void onModified(String path) {
        onChange("^", path);
    }

    @Override
    public void onCreated(String path) {
        onChange("+", path);
    }

    @Override
    public void onDeleted(String path) {
        if (path != null) {
            onChange("-", path);
        }
    }

    @Override
    public void onMoved(String srcPath, String dstPath) {
        onChange(">", srcPath, dstPath);
    }

    @Override
    public void onCopied(String srcPath, String dstPath) {
        onChange("*", srcPath, dstPath);
    }

    @SuppressWarnings("null")
    @Override
    public void onChange(String type, String... arguments) {
        changes.add(new Change(type, arguments));
    }

    public void setDescription(@CheckForNull String description) {
        setProperty(PN_DESCRIPTION, description);
    }

    /**
     * Sets the general purpose error message using the given status code. The title
     * and description will be set automatically.
     *
     * @param code
     *            the status code
     */
    public void setGeneralError(int code) {
        setStatus(code, i18n.get("The server has problem processing your request."));
        setTitle(i18n.get("Error"));
        setDescription(i18n.get("The server has problem processing your request."));
    }

    /**
     * Adds a redirect link to indicate where the client should go after the post.
     * Note that you SHOULD call this method once, otherwise would be a duplicate.
     *
     * @param href
     *            the link
     * @param text
     *            the link text
     */
    public void addRedirectLink(@Nonnull String href, @Nonnull String text) {
        addLink("foundation-form-response-redirect", href, text);
    }

    /**
     * Adds a link to the response. It is for the server to give affordance to the
     * client.
     *
     * @param rel
     *            relationship attribute
     * @param href
     *            the link
     * @param text
     *            the link text
     */
    public void addLink(@Nonnull String rel, @Nonnull String href, @Nonnull String text) {
        links.add(new Link(rel, href, text));
    }

    @SuppressWarnings("null")
    @Override
    protected void doSend(HttpServletResponse res) throws IOException {
        res.setContentType("text/html");
        res.setCharacterEncoding("UTF-8");

        Writer out = res.getWriter();

        out.write("<!DOCTYPE html>\n");
        out.write("<html lang='" + xss.encodeForHTMLAttr(LocaleUtil.toRFC4646(locale)) + "'>\n");
        out.write("<head>\n");

        Object title = getProperty(PN_TITLE);

        if (title != null) {
            write(out, "title", title.toString(), true);
        }

        out.write("</head>\n");
        out.write("<body>\n");

        if (title != null) {
            write(out, "h1", title.toString(), true);
        }

        out.write("<dl>\n");

        writeDefinition(out, "foundation-form-response-status-code", i18n.get("Status"), getProperty(PN_STATUS_CODE));
        writeDefinition(out, "foundation-form-response-status-message", i18n.get("Message"),
                getProperty(PN_STATUS_MESSAGE));
        writeDefinition(out, "foundation-form-response-path", i18n.get("Path"), getProperty(PN_PATH));
        writeDefinition(out, "foundation-form-response-title", i18n.get("Title"), title);
        writeDefinition(out, "foundation-form-response-description", i18n.get("Description"),
                getProperty(PN_DESCRIPTION));

        // TODO add other properties

        if (!changes.isEmpty()) {
            out.write("<dt class='foundation-form-response-changelog'>");
            out.write(i18n.get("Change Log"));
            out.write("</dt>\n");

            out.write("<dd>\n");
            out.write("<ol>\n");

            for (Change change : changes) {
                out.write("<li>");
                writeChange(out, change);
                out.write("</li>\n");
            }

            out.write("</ol>\n");
            out.write("</dd>\n");
        }

        out.write("</dl>\n");

        if (!links.isEmpty()) {
            write(out, "h2", i18n.get("Links"), true);
            out.write("<ul class='foundation-form-response-links'>\n");

            for (Link link : links) {
                out.write("<li>");
                writeLink(out, link);
                out.write("</li>\n");
            }

            out.write("</ul>\n");
        }

        out.write("</body>\n");
        out.write("</html>\n");
    }

    private void writeDefinition(@Nonnull Writer out, @Nonnull String rel, @Nonnull String term,
            @CheckForNull Object value) throws IOException {
        if (value == null) {
            return;
        }

        out.write("<dt class='" + xss.encodeForHTMLAttr(rel) + "'>");
        out.write(xss.encodeForHTML(term));
        out.write("</dt>\n");

        out.write("<dd>");
        out.write(xss.filterHTML(value.toString()));
        out.write("</dd>\n");
    }

    @SuppressWarnings("null")
    private void writeChange(@Nonnull Writer out, @Nonnull Change change) throws IOException {
        write(out, "span", change.type, false);

        out.write("(");
        for (int i = 0; i < change.arguments.length; i++) {
            if (i > 0) {
                out.write(", ");
            }

            write(out, "span", change.arguments[i], false);
        }
        out.write(")");
    }

    private void writeLink(@Nonnull Writer out, @Nonnull Link link) throws IOException {
        out.write("<a class='" + xss.encodeForHTMLAttr(link.rel) + "' href='" + xss.getValidHref(link.href) + "'>");
        out.write(xss.encodeForHTML(link.text));
        out.write("</a>");
    }

    private void write(@Nonnull Writer out, @Nonnull String tag, @Nonnull String text, boolean newline)
            throws IOException {
        out.write("<" + tag + ">" + xss.encodeForHTML(text) + "</" + tag + ">");

        if (newline) {
            out.write("\n");
        }
    }

    private class Change {
        @Nonnull
        String type;

        @Nonnull
        String[] arguments;

        public Change(@Nonnull String type, @Nonnull String... arguments) {
            this.type = type;
            this.arguments = arguments;
        }
    }

    private class Link {
        @Nonnull
        String rel;

        @Nonnull
        String href;

        @Nonnull
        String text;

        public Link(@Nonnull String rel, @Nonnull String href, @Nonnull String text) {
            this.rel = rel;
            this.href = href;
            this.text = text;
        }
    }
}
