/* ****************************************************************************
 *
 *	File: ASDictionary.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnableToCompleteOperationException;

public class ASDictionary extends ASObject implements Map<ASName, ASObject>
{
	private Map<ASName, ASObject> map;
	
	public ASDictionary() {this.map = new LinkedHashMap<ASName, ASObject>();}

	/**
	 * Replaces the "name" key value with the object passed.
	 * @return 
	 */
	public ASObject replace(ASName name, ASObject clientObject) {return map.put(name, clientObject);}
	
	/**
	 * Returns true if this key exists in this dictionary else false.
	 */
	public boolean exists(ASName name) {return map.containsKey(name);}

	/**
	 * Removes this key from the dictionary.
	 */
	public Object remove(ASName name) {return map.remove(name);}

	
	@Override
	public String toString()
	{
		StringBuilder str = new StringBuilder("<<");
		Set keySet = map.keySet();
		Iterator iter = keySet.iterator();
		while(iter.hasNext()) {
			ASName key = (ASName)(iter.next());
			Object value = map.get(key);
			str.append(' ');
			str.append(key.toString());
			str.append(' ');
			str.append(value.toString());
		}
		str.append(">>");
		return str.toString();
	}

	/**
	 * Writes the ASDictionary in to the given OutputStream in the format expected by the PDF Spec.
	 *
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			Set keySet = map.keySet();
			outputByteStream.write('<');
			outputByteStream.write('<');
			Iterator iter = keySet.iterator();
			while(iter.hasNext()) {
	
				// write the key
				ASName key = (ASName)(iter.next());
				key.write(outputByteStream);
				outputByteStream.write(' ');
	
				// write the value
				ASObject value = map.get(key);
				value.write(outputByteStream);
				if (iter.hasNext())
					outputByteStream.write(' ');
			}
			outputByteStream.write('>');
			outputByteStream.write('>');
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public ASNumber getNumber(ASName key) throws PDFUnableToCompleteOperationException
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASNumber))
			throw new PDFUnableToCompleteOperationException("ASNumber Expected");
		return (ASNumber)obj;
	}

	public ASBoolean getBoolean(ASName key) throws PDFUnableToCompleteOperationException
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASBoolean))
			throw new PDFUnableToCompleteOperationException("ASBoolean Expected");
		return (ASBoolean)obj;
	}

	public ASName getName(ASName key) throws PDFUnableToCompleteOperationException
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASName))
			throw new PDFUnableToCompleteOperationException("ASName Expected");
		return (ASName)obj;
	}

	public ASString getString(ASName key) throws PDFUnableToCompleteOperationException 
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASString))
			throw new PDFUnableToCompleteOperationException("ASString Expected");
		return (ASString)obj;
	}

	public ASArray getArray(ASName key) throws PDFUnableToCompleteOperationException
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASArray))
			throw new PDFUnableToCompleteOperationException("ASArray Expected");
		return (ASArray)obj;
	}

	public ASDictionary getDictionary(ASName key) throws PDFUnableToCompleteOperationException
	{
		ASObject obj = map.get(key);
		if (obj != null && !(obj instanceof ASDictionary))
			throw new PDFUnableToCompleteOperationException("ASDictionary Expected");
		return (ASDictionary)obj;
	}

	/* (non-Javadoc)
	 * @see java.util.Map#size()
	 */
	public int size() {
		return map.size();
	}

	/* (non-Javadoc)
	 * @see java.util.Map#clear()
	 */
	public void clear() {
		map.clear();
	}

	/* (non-Javadoc)
	 * @see java.util.Map#isEmpty()
	 */
	public boolean isEmpty() {
		return map.isEmpty();
	}

	/* (non-Javadoc)
	 * @see java.util.Map#containsKey(java.lang.Object)
	 */
	public boolean containsKey(Object key) {
		return map.containsKey(key);
	}

	
	/* (non-Javadoc)
	 * @see java.util.Map#containsValue(java.lang.Object)
	 */
	public boolean containsValue(Object value) {
		return map.containsValue(value);
	}

	
	/* (non-Javadoc)
	 * @see java.util.Map#values()
	 */
	public Collection<ASObject> values() {
		return map.values();
	}

	
	/* (non-Javadoc)
	 * @see java.util.Map#putAll(java.util.Map)
	 */
	public void putAll(Map<? extends ASName, ? extends ASObject> t) {
		map.putAll(t);
	}

	
	/* (non-Javadoc)
	 * @see java.util.Map#entrySet()
	 */
	public Set<Entry<ASName, ASObject>>  entrySet() {
		return map.entrySet();
	}

	
	/* (non-Javadoc)
	 * @see java.util.Map#keySet()
	 */
	public Set<ASName> keySet() {
		return map.keySet();
	}

	/* (non-Javadoc)
	 * @see java.util.Map#get(java.lang.Object)
	 */
	public ASObject get(Object key) {
		return map.get(key);
	}

	/* (non-Javadoc)
	 * @see java.util.Map#remove(java.lang.Object)
	 */
	public ASObject remove(Object key) {
		return map.remove(key);
	}

	/* (non-Javadoc)
	 * @see java.util.Map#put(java.lang.Object, java.lang.Object)
	 */
	public ASObject put(ASName key, ASObject value) {
		return map.put(key, value);
	}
}
