/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.personalization;

import org.apache.jackrabbit.util.Text;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.settings.SlingSettingsService;

import javax.servlet.http.Cookie;
import java.util.HashMap;
import java.util.Map;

public class ContextSessionPersistence {

    private static final String DEFAULT_COOKIE_NAME = "SessionPersistence";

    /**
     * Returns the value from the default session persistence cookie for the
     * specified {@code key}.
     * 
     * @param request The SlingHttpServletRequest
     * @param key Key
     * @param settingsService The SlingSettingsService
     * @return Value or an empty string
     * @deprecated since 5.6. Use
     *             {@link #get(org.apache.sling.api.SlingHttpServletRequest, String)}
     *             instead.
     */
    public static String get(SlingHttpServletRequest request, String key, SlingSettingsService settingsService) {
        return get(request, key);
    }

    /**
     * Returns the value from the default session persistence cookie for the
     * specified {@code key}.
     * 
     * @param request The SlingHttpServletRequest
     * @param key Key
     * @return Value or an empty string
     */
    public static String get(SlingHttpServletRequest request, String key) {
        return get(request, key, getCookieName());
    }

    /**
     * Returns the value from the specified {@code cookieName cookie} for the
     * specified {@code key}.
     * 
     * @param request The SlingHttpServletRequest
     * @param key Key
     * @param cookieName Cookie name
     * @return Value or an empty string
     */
    public static String get(SlingHttpServletRequest request, String key, String cookieName) {
        String pairsMap = getMap(request, cookieName);

        String value = "";
        if (pairsMap != null) {
            int index = pairsMap.indexOf(key + ":=");
            if (index != -1) {
                int end = pairsMap.indexOf("|", index + 2);
                if (end == -1) {
                    end = pairsMap.length();
                }
                value = pairsMap.substring(index + (key + ":=").length(), end);
            }
        }
        value = (value == null || "null".equals(value) ? "" : value);
        return Text.unescape(value);
    }

    /**
     * Returns a key/value map of the default session persistence cookie.
     * 
     * @param request The SlingHttpServletRequest
     * @param settingsService The SlingSettingsService
     * @return Map with key/value pairs
     * @deprecated since 5.6. Use
     *             {@link #getMap(org.apache.sling.api.SlingHttpServletRequest)}
     *             instead.
     */
    public static String getMap(SlingHttpServletRequest request, SlingSettingsService settingsService) {
        return getMap(request);
    }

    /**
     * Returns a key/value map of the default session persistence cookie.
     * 
     * @param request The SlingHttpServletRequest
     * @return Map with key/value pairs
     */
    public static String getMap(SlingHttpServletRequest request) {
        return getMap(request, getCookieName());
    }

    /**
     * Returns a key/value map of named cookie.
     * 
     * @param request The SlingHTTPServletRequest
     * @param cookieName Cookie name
     * @return Map with key/value pairs
     */
    public static String getMap(SlingHttpServletRequest request, String cookieName) {
        Cookie cookie = request.getCookie(cookieName);
        if (cookie != null) {
            String pairsMap = request.getCookie(cookieName).getValue();
            if (pairsMap != null) {
                // DON'T USE URLDecoder.decode()! It won't round-trip unicode
                // characters with Text.escape()
                // or Javascript's encodeURIComponent() (used client-side)
                return Text.unescape(pairsMap);
            }
        }
        return "";
    }

    /**
     * Add a name/value pair to the default session persistence cookie.
     * 
     * @param request The SlingHTTPServletRequest
     * @param response The SlingHTTPServletResponse
     * @param key Key
     * @param value Value
     * @param settingsService The SlingSettingsService
     * @deprecated since 5.6. Use
     *             {@link #put(org.apache.sling.api.SlingHttpServletRequest, org.apache.sling.api.SlingHttpServletResponse, String, String)}
     *             instead.
     */
    public static void put(SlingHttpServletRequest request, SlingHttpServletResponse response, String key, String value,
            SlingSettingsService settingsService) {
        put(request, response, key, value);
    }

    /**
     * Add a name/value pair to the default session persistence cookie.
     * 
     * @param request The SlingHTTPServletRequest
     * @param response The SlingHTTPServletResponse
     * @param key Key
     * @param value Value
     */
    public static void put(SlingHttpServletRequest request, SlingHttpServletResponse response, String key,
            String value) {
        put(request, response, key, value, getCookieName());
    }

    /**
     * Add a name/value pair to a named cookie.
     * 
     * @param request The SlingHTTPServletRequest
     * @param response The SlingHTTPServletResponse
     * @param key Key
     * @param value Value
     * @param cookieName Cookie name
     */
    public static void put(SlingHttpServletRequest request, SlingHttpServletResponse response, String key, String value,
            String cookieName) {
        Cookie cookie = request.getCookie(cookieName);
        if (cookie == null) {
            cookie = new Cookie(cookieName, "");
        }
        internalAddCookie(request, response, internalPut(request, response, key, value, cookie));
    }

    private static Cookie internalPut(SlingHttpServletRequest request, SlingHttpServletResponse response, String key,
            String value, Cookie cookie) {
        if (cookie == null) {
            return null;
        }
        String pairsMap = cookie.getValue();
        if (pairsMap == null) {
            pairsMap = "";
        } else {
            // DON'T USE URLDecoder.decode()! It won't round-trip unicode
            // characters with Text.escape()
            // or Javascript's encodeURIComponent() (used client-side)
            pairsMap = Text.unescape(pairsMap);

            int index = pairsMap.indexOf(key + ":=");
            if (index != -1) {
                int end = pairsMap.indexOf("|", index + 2);
                if (end == -1) {
                    pairsMap = pairsMap.substring(0, index);
                } else {
                    pairsMap = pairsMap.substring(0, index) + pairsMap.substring(end + 1, pairsMap.length());
                }
            }
        }
        if (pairsMap.length() > 0 && !pairsMap.endsWith("|")) {
            pairsMap += "|";
        }
        pairsMap += key + ":=" + Text.escape(value) + "|";

        // DON'T USE URLEncoder.encode()! It's not compatible with Javascript's
        // decodeURIComponent(),
        // which we use client-side (https://issues.adobe.com/browse/CQ5-18583).
        pairsMap = Text.escape(pairsMap);
        cookie.setValue(pairsMap);

        return cookie;
    }

    private static Map<String, String> parse(String str) {
        Map<String, String> obj = new HashMap<String, String>();
        if (str != null) {
            String[] array = str.split(",");
            for (String t : array) {
                String[] entry = t.split("=");
                if (entry.length == 2) {
                    obj.put(entry[0], entry[1]);
                }
            }
        }
        return obj;
    }

    private static String serialize(Map<String, String> store) {
        String result = "";
        String[] keys = store.keySet().toArray(new String[store.size()]);
        for (int i = 0; i < keys.length; i++) {
            if (i > 0) result += ",";
            result += keys[i] + "=" + store.get(keys[i]);
        }
        return result;
    }

    /**
     * Return a single store (by its key) from the default session persistence
     * cookie.
     * 
     * @param request The SlingHTTPServletRequest
     * @param key Key referencing the store
     * @param settingsService The SlingSettingsService
     * @return the store
     * @deprecated since 5.6. Use
     *             {@link #getStore(org.apache.sling.api.SlingHttpServletRequest, String)}
     *             instead.
     */
    public static Map<String, String> getStore(SlingHttpServletRequest request, String key,
            SlingSettingsService settingsService) {
        return getStore(request, key);
    }

    /**
     * Return a single store (by its key) from the default session persistence
     * cookie.
     * 
     * @param request The SlingHttpServletRequest
     * @param key Key referencing the store
     * @return the store
     */
    public static Map<String, String> getStore(SlingHttpServletRequest request, String key) {
        return getStore(request, key, getCookieName());
    }

    /**
     * Return a single store (by its key) from a named cookie.
     * 
     * @param request The SlingHttpServletRequest
     * @param key Key referencing the store
     * @param cookieName Cookie name
     * @return the store
     */
    public static Map<String, String> getStore(SlingHttpServletRequest request, String key, String cookieName) {
        return parse(get(request, key, cookieName));
    }

    /**
     * Persists a single store under the given key.
     * 
     * @param request The SlingHttpServletRequest
     * @param response The SlingHttpServletResponse
     * @param key Key
     * @param store A map representing the store
     * @param settingsService The SlingSettingsService
     * @deprecated since 5.6. Use
     *             {@link #putStore(org.apache.sling.api.SlingHttpServletRequest, org.apache.sling.api.SlingHttpServletResponse, String, java.util.Map)}
     *             instead.
     */
    public static void putStore(SlingHttpServletRequest request, SlingHttpServletResponse response, String key,
            Map<String, String> store, SlingSettingsService settingsService) {
        putStore(request, response, key, store);
    }

    /**
     * Persists a single store under the given key.
     * <p>
     * <b>Note: Don't use this method to persist multiple stores. Use
     * {@link #putStores(SlingHttpServletRequest, SlingHttpServletResponse, Map, String)}
     * instead.</b>
     * 
     * @param request The SlingHttpServletRequest
     * @param response The SlingHttpServletResponse
     * @param key Key 
     * @param store A map representing the store
     */
    public static void putStore(SlingHttpServletRequest request, SlingHttpServletResponse response, String key,
            Map<String, String> store) {
        Map<String, Map<String, String>> stores = new HashMap<String, Map<String, String>>();
        stores.put(key, store);
        putStores(request, response, stores, getCookieName());
    }

    /**
     * Put multiple stores at once without overwriting stores previously put to
     * response. The method takes a map of stores where the key corresponds to
     * the store key.
     * 
     * @param request The request
     * @param response The response
     * @param stores The map of stores
     * @param cookieName The cookie's name
     */
    public static void putStores(SlingHttpServletRequest request, SlingHttpServletResponse response,
            Map<String, Map<String, String>> stores, String cookieName) {
        Cookie cookie = request.getCookie(cookieName);
        if (cookie == null) {
            cookie = new Cookie(cookieName, "");
        }
        for (String key : stores.keySet()) {
            cookie = internalPut(request, response, key, serialize(stores.get(key)), cookie);
        }
        internalAddCookie(request, response, cookie);
    }

    /**
     * Add a cookie to the response. Make sure the secure flag is set if the
     * request is https, and that the cookie path respects the context-path.
     * 
     * @param request The current request.
     * @param response The current response.
     * @param cookie The cookie to add.
     */
    private static void internalAddCookie(SlingHttpServletRequest request, SlingHttpServletResponse response,
            Cookie cookie) {
        // ensure the cookie is secured if this is an https request
        if (request.isSecure()) {
            cookie.setSecure(true);
        }

        // set the cookie path to the contextPath, or "/" if no contextPath is
        // set
        String contextPath = request.getContextPath();
        String cookiePath = (contextPath == null || contextPath.length() == 0) ? "/" : contextPath;
        cookie.setPath(cookiePath);

        response.addCookie(cookie);
    }

    /**
     * Return the name of the default session persistence cookie.
     * 
     * @return Cookie name
     */
    public static String getCookieName() {
        return DEFAULT_COOKIE_NAME;
    }

    /**
     * Return the name of the cookie.
     * 
     * @param settingsService The SlingSettingsService
     * @return Cookie name
     * @deprecated since 5.6. Use {@link #getCookieName()} instead.
     */
    public static String getCookieName(SlingSettingsService settingsService) {
        return getCookieName();
    }

    /**
     * Return the name of the cookie.
     * 
     * @param cookiePrefix The cookie prefix
     * @param settingsService The SlingSettingsService
     * @return Cookie name
     * @deprecated since 5.6. Use {@link #getCookieName()} instead.
     */
    public static String getCookieName(String cookiePrefix, SlingSettingsService settingsService) {
        return cookiePrefix;
    }
}