package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;

class GposHarvester extends LookupTableHarvester{
	private final static NoopHarvester noopHarvester = new NoopHarvester();
	private final NCoveragesHarvester twoCoveragesHarvester = new NCoveragesHarvester(2, 2, 1);
	private final PairPosFormat1Harvester pairPosHarvester = new PairPosFormat1Harvester();
	
	GposHarvester(Gpos gpos, int numGlyphs)
	{
		super(gpos, numGlyphs);
	}
	
	 /**
	  * Harvest the lookups that are relevant to the supplied gids.  
	  * @param gids The gids to harvest
	  * @return A map of lookup indices and subtable indices that can (possibly recursively) apply to the supplied gids.
	  * @throws InvalidFontException
	 * @throws UnsupportedFontException 
	  */
	  TreeMap /*<Integer lookup index, List subtables> */ gatherPossibleLookups(Subset gids) throws InvalidFontException, UnsupportedFontException
	  {
		  TreeMap applicableLookups = new TreeMap();
		  IntGrowableArray featuresInFeatureList = getAllFeatureListFeatures();
		  harvestAllFeatures(featuresInFeatureList, applicableLookups, gids);	
		  return applicableLookups;
	  }
	  
	  abstract private static class GposLookupHarvester implements CoverageLookupHarvester
	  {
			public boolean keepGoing() {
				return false;
			}
	  }
	  
	  private static class NoopHarvester extends GposLookupHarvester
	  {
		  public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) {
			return true;
		}
		  
	  }
	  
	  private class PairPosFormat1Harvester extends GposLookupHarvester {
		  public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) throws InvalidFontException, UnsupportedFontException {
			int pairSetOffset = lookupTable.data.getOffset(stOffset, 10 + 2*coverageIndex);
			int firstValueSize = ((Gpos)lookupTable).getValueRecordSize(lookupTable.data.getuint16(stOffset + 4));
			int secondValueSize = ((Gpos)lookupTable).getValueRecordSize(lookupTable.data.getuint16(stOffset + 6));
			int pairValueCount = lookupTable.data.getuint16(pairSetOffset);
			for (int i = 0; i < pairValueCount; i++)
			{
				int secondGlyph = lookupTable.data.getuint16(pairSetOffset + 2 + (2 + firstValueSize + secondValueSize)*i);
				if (gids.getExistingSubsetGid(secondGlyph) != -1)
					return true;
			}
			  
			return false;
		}
	  }
	  
	  /**
	   * Multiple coverages need to be covered by the subset in order for the lookup to be applicable.
	   * This harvester handles lookups whose multiple coverages are at offsets that are adjacent in the subtable.
	   */
	  private class NCoveragesHarvester extends GposLookupHarvester implements CoverageConsumer {
		 private boolean covered;
		 private boolean subsequentCoveragesIterated = false;
		 private boolean subsequentCoveragesCovered;
		 private int offsetDeltaToCoverages;
		 protected int numCoverages;
		 private int coverageStartIndex;
		 
		 NCoveragesHarvester(int offsetDeltaToCoverages, int numCoverages, int coverageStartIndex)
		 {
			 this.offsetDeltaToCoverages = offsetDeltaToCoverages;
			 this.numCoverages = numCoverages;
			 this.coverageStartIndex = coverageStartIndex;
		 }
		 
		 protected void resetCoverageInfo(int offsetDeltaToCoverages, int numCoverages, int coverageStartIndex)
		 {
			 // Change the set of coverages to be iterated in lookupApplies. 
			 
			 subsequentCoveragesIterated = false;
			 this.offsetDeltaToCoverages = offsetDeltaToCoverages;
			 this.numCoverages = numCoverages;
			 this.coverageStartIndex = coverageStartIndex;
		 }
		 
		 public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) throws InvalidFontException, UnsupportedFontException {
			 // This method is called for each glyph in a coverage. The rest of this method is used to look at the
			 // rest of the coverages to see if they are covered by the Subset. Since that doesn't depend on 
			 // the glyph being iterated, only run this code 1x for a fixed set of coverages.
			 if (subsequentCoveragesIterated)
				 return subsequentCoveragesCovered;
			 
			subsequentCoveragesIterated = true;
			for (int i = coverageStartIndex; i < numCoverages; i++)
			{
				covered = false;
				lookupTable.iterateCoverage(lookupTable.data.getOffset(stOffset, offsetDeltaToCoverages + 2*i), gids, this);
				if (covered == false)
				{
					subsequentCoveragesCovered = false;
					return false;
				}
			}
			subsequentCoveragesCovered = true;
			return true;
		}
		 
		 public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			covered = true;
			return false;
		}
	  }
	  
	  private class ContextFormat3Harvester extends NCoveragesHarvester
	  {
		  private final Map lookups;
		  
		  ContextFormat3Harvester(Map lookups, int stOffset) throws InvalidFontException
		  {
			  super (6, lookupTable.data.getuint16(stOffset + 2), 1);
			  this.lookups = lookups;
		  }
		  
		  public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) throws InvalidFontException, UnsupportedFontException
		  {
			  boolean applies = super.lookupApplies(coverageGlyph, stOffset, coverageIndex, gids);
			  
			  if (applies)
			  {
				  int lookupCount = lookupTable.data.getuint16(stOffset+4);
				  
				  for (int j = 0; j < lookupCount; j++)
				  {
					  int lookupIndex = lookupTable.data.getuint16(stOffset + 6 + 2*numCoverages + 4*j + 2);
					  harvest(gids, lookupIndex, lookups);
				  }
			  }
			  
			  return applies;
		  }
	  }

	  private class ChainingFormat3Harvester extends NCoveragesHarvester
	  {
		  private final Map lookups;
		  private int backtrackCount;
		  private boolean subsequentCoveragesTested = false;
		  private boolean subsequentCoveragesApply;
		  
		  ChainingFormat3Harvester(Map lookups, int stOffset) throws InvalidFontException
		  {
			  
			  super (4, lookupTable.data.getuint16(stOffset + 2), 1);
			  backtrackCount = lookupTable.data.getuint16(stOffset + 2);
			  this.lookups = lookups;
		  }
		  
		  public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) throws InvalidFontException, UnsupportedFontException
		  {

			  // This method gets called for every glyph in the first coverage, but all of the code
			  // below here does the same thing regardless of what that glyph is. Only run this processing 1x...
			  if (subsequentCoveragesTested)
				  return subsequentCoveragesApply;
			  
			  subsequentCoveragesTested = true;
			  
			  // see if the backtrack is covered by the subset.
			  boolean applies = super.lookupApplies(coverageGlyph, stOffset, coverageIndex, gids);

			  if (!applies)
			  {
				  subsequentCoveragesApply = false;
				  return false;
			  }
			  
			  // see if the input is covered by the subset.
			  int inputCount = lookupTable.data.getuint16(stOffset + 4 + 2*backtrackCount);
			  resetCoverageInfo(4 + 2*backtrackCount + 2, inputCount, 0);
			  applies = super.lookupApplies(coverageGlyph, stOffset, coverageIndex, gids);

			  if (!applies)
			  {
				  subsequentCoveragesApply = false;
				  return false;
			  }

			  // see if the lookahead is covered by the subset.
			  int lookaheadCount = lookupTable.data.getuint16(stOffset + 6 + 2*backtrackCount + 2*inputCount);
			  resetCoverageInfo(8 + 2*backtrackCount + 2*inputCount, lookaheadCount, 0);
			  applies = super.lookupApplies(coverageGlyph, stOffset, coverageIndex, gids);

			  if (!applies)
			  {
				  subsequentCoveragesApply = false;
				  return false;
			  }

			  // If we've gotten here, we want to keep the lookup. Harvest the lookups that it refers to.
			  int lookupCount = lookupTable.data.getuint16(stOffset + 8 + 2*backtrackCount + 2*inputCount + 2*lookaheadCount);
			  for (int j = 0; j < lookupCount; j++)
			  {
				  int lookupIndex = lookupTable.data.getuint16(stOffset + 10 + 2*backtrackCount + 2*inputCount + 2*lookaheadCount + 4*j + 2);
				  harvest(gids, lookupIndex, lookups);
			  }

			  subsequentCoveragesApply = true;
			  return true;
		  }
	  }
	  
	  private List gatherPairPosLookups(int stOffset, Subset gids,  List subtables, int subtableIndex) throws InvalidFontException, UnsupportedFontException
	  {
		    int stFormat = lookupTable.data.getuint16(stOffset);
		    switch (stFormat) {
		    case 1:
		    	return gatherLookupsForCoverage(stOffset, 2, gids, subtables, pairPosHarvester, subtableIndex);
		    case 2:
		    	return gatherLookupsForCoverage(stOffset, 2, gids, subtables, noopHarvester, subtableIndex);
		    default:
		    	throw new InvalidFontException("Invalid pair pos lookup format (" + stFormat + ")");
		    }
	  }
	  
	  private List gatherPairPosLookupsForAllSubtables(Subset gids, int offset) throws InvalidFontException, UnsupportedFontException {
		  List subtables = null;
		  int subtableCount = lookupTable.data.getuint16 (offset+ 4);
		      
		  for (int i = 0; i < subtableCount; i++)
		  {
		    int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
		    subtables = gatherPairPosLookups(stOffset, gids, subtables, i);
		  }
		  
		  return subtables;
	  }
	  
	  private List gatherContextualLookup(int stOffset, Subset gids, Map lookups, List subtables, int subtableIndex) throws InvalidFontException, UnsupportedFontException
	  {	    
		  int stFormat = lookupTable.data.getuint16(stOffset);
		  switch (stFormat) {
		  case 1:
			  return gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ContextFormat1Harvester(this, lookups, false), subtableIndex);
			  
		  case 2:
			  return gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ContextFormat2Harvester(this, lookups), subtableIndex);
			  
		  case 3:
			  	int numGlyphs = lookupTable.data.getuint16(stOffset+2);
			  	if (numGlyphs > 0)
			  		return gatherLookupsForCoverage(stOffset, 6, gids, subtables, new ContextFormat3Harvester(lookups, stOffset), subtableIndex);
			  	return subtables;
			  	
		  default:
		  	throw new InvalidFontException("Invalid contextual lookup format (" + stFormat + ")");
		  }
	  }

	  private List gatherContextualLookupsForAllSubtables(Subset gids, int offset, Map lookups) throws InvalidFontException, UnsupportedFontException {
		  List subtables = null;
		  int subtableCount = lookupTable.data.getuint16 (offset+ 4);
		      
		  for (int i = 0; i < subtableCount; i++)
		  {
		    int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
		    subtables = gatherContextualLookup(stOffset, gids, lookups, subtables, i);
		  }
		  
		  return subtables;
	  }
	  

	  private List gatherChainingLookup(int stOffset, Subset gids, Map lookups, List subtables, int subtableIndex) throws InvalidFontException, UnsupportedFontException
	  {	    
		  int stFormat = lookupTable.data.getuint16(stOffset);
		  switch (stFormat) {
		  case 1:
			  return gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ChainingFormat1Harvester(this, lookups, false), subtableIndex);
			  
		  case 2:
			  return gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ChainingFormat2Harvester(this, lookups), subtableIndex);
			  
		  case 3:
			  int backtrackCount = lookupTable.data.getuint16(stOffset + 2);
		    	int inputCount = lookupTable.data.getuint16(stOffset + 4 + 2*backtrackCount);
		    	if (inputCount > 0)
		    		return gatherLookupsForCoverage(stOffset, 4 + 2*backtrackCount + 2, gids, subtables, new ChainingFormat3Harvester(lookups, stOffset), subtableIndex);
		    	return subtables;
		  default:
		  	throw new InvalidFontException("Invalid contextual lookup format (" + stFormat + ")");
		  }
	  }
	  
	  private List gatherChainingLookupsForAllSubtables(Subset gids, int offset, Map lookups) throws InvalidFontException, UnsupportedFontException {
		  List subtables = null;
		  int subtableCount = lookupTable.data.getuint16 (offset+ 4);
		      
		  for (int i = 0; i < subtableCount; i++)
		  {
		    int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
		    subtables = gatherChainingLookup(stOffset, gids, lookups, subtables, i);
		  }
		  
		  return subtables;
	  }
	  

	  private List gatherExtensionGlyphsForAllSubtables(Subset gids, int offset, Map lookups) throws InvalidFontException, UnsupportedFontException {
		  List subtables = null;
		  List dummySubtables = null;
		  int subtableCount = lookupTable.data.getuint16 (offset+ 4);
		      
		  for (int i = 0; i < subtableCount; i++)
		  {
		    int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
		    int format = lookupTable.data.getuint16(stOffset);
		    if (format != 1) // we only understand format 1.
		    	throw new InvalidFontException("Invalid extension format (" + format + ")");
		    int lookupType = lookupTable.data.getuint16(stOffset + 2);
		    int extOffset = lookupTable.data.getuint32asint(stOffset + 4, "Only signed extension values supported");
		    
			switch (lookupType) {
		      case 1: 
		      case 3: 
		    	  dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, noopHarvester, 0);
		    	  break;
		    	  
		      case 2: 
		    	  dummySubtables = gatherPairPosLookups(stOffset+extOffset, gids, dummySubtables, 0);
		    	  break;
		    	  
		      case 4: 
		      case 5:
		      case 6: 
		    	  twoCoveragesHarvester.resetCoverageInfo(2, 2, 1);
		    	  dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, twoCoveragesHarvester, 0);
		    	  break;

		      case 7:
		    	  dummySubtables = gatherContextualLookup(stOffset+extOffset, gids, lookups, dummySubtables, 0);
		    	  break; 
		    	  
		      case 8: 
		    	  dummySubtables = gatherChainingLookup(stOffset+extOffset, gids, lookups, dummySubtables, 0);
		    	  break;
		    	  
		      case 9: // Extension lookups can't have extension sublookups, per the spec.
		      default: 
		        throw new InvalidFontException ("Invalid GPOS lookup type (" + lookupType + ")"); 
		      
		    }
			
			if (dummySubtables != null) // if this sub-subtable applied, add the extension subtable to the list of relevant subtables.
			{
				if (subtables == null)
					subtables = new ArrayList();
		    	Integer j = new Integer(i);
		    	if (!subtables.contains(j))
		    		subtables.add(j);
			}
		  }
		  
		  return subtables;
	  }
	
	List harvestSubtables(Subset gids, int lookupOffset, int lookupType,
			int lookupFlag, Map lookups) throws InvalidFontException,
			UnsupportedFontException {
		  switch (lookupType) {
		      case 1: 
		      case 3:  
		    	  return gatherLookupsForCoverageAllSubtables(gids, lookupOffset, noopHarvester);

		      case 2: 
		    	  return gatherPairPosLookupsForAllSubtables(gids, lookupOffset);

		      case 4: 
		      case 5:
		      case 6:  
		    	  twoCoveragesHarvester.resetCoverageInfo(2, 2, 1);
		    	  return gatherLookupsForCoverageAllSubtables(gids, lookupOffset, twoCoveragesHarvester);

		      case 7: 
		    	  return gatherContextualLookupsForAllSubtables(gids, lookupOffset, lookups);

		      case 8: 
		    	  return gatherChainingLookupsForAllSubtables(gids, lookupOffset, lookups);
		    	  
		      case 9: 
		    	  return gatherExtensionGlyphsForAllSubtables(gids, lookupOffset, lookups);
		      default: {
		        throw new InvalidFontException ("Invalid GPOS lookup type (" + lookupType + ")"); }}
	}

}
