/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2012 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/

package com.adobe.granite.ui.components;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;

/**
 * Utility class for form field component.
 */
public class Field {
    /**
     * The config of the component
     */
    @Nonnull
    private Config cfg;

    /**
     * The name of the CSS class to apply in order to hide empty field components in
     * default mode.
     */
    @Nonnull
    public static String HIDE_IN_DEFAULT_CLASS = "foundation-field-hide-in-default";

    /**
     * The suffix that specifies if a property is a mixed value (for bulk editing).
     */
    @Nonnull
    public static String IS_MIXED_SUFFIX = ".granite.mixed";

    /**
     * The request attribute to be set when doing bulk editing
     */
    private static final String BULK_EDIT_MODE_ATTRIBUTE = "granite.ui.field.bulkedit";

    public Field() {
        /* Empty on purpose: don't use it! */
        throw new UnsupportedOperationException("Please use the other constructor");
    }

    /**
     * Creates a new Field object with the given config.
     *
     * @param cfg
     *            The config of the component
     */
    public Field(@Nonnull Config cfg) {
        this.cfg = cfg;
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows for
     * example to hide empty field components in read only mode.
     *
     * @param isEmpty
     *            Indicate if the field is empty
     * @return the CSS class (or space separated class values)
     */
    @Nonnull
    public String getRootClass(boolean isEmpty) {
        return Field.getRootClass(this.cfg, isEmpty);
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows for
     * example to hide empty field components in read only mode.
     *
     * @param value
     *            The value of the field component
     * @return the CSS class (or space separated class values)
     */
    @Nonnull
    public String getRootClass(@CheckForNull String value) {
        return Field.getRootClass(this.cfg, value);
    }

    /**
     * Returns {@code true} if the field is a mixed value; {@code false} otherwise.
     *
     * A field is a mixed value if it has a property named
     * {@code cfg.get("name") + IS_MIXED_SUFFIX} which has the value of
     * {@code true}.
     *
     * @param value
     *            The form value
     * @return True if the field component holds a mixed value, false otherwise
     */
    public boolean isMixed(@Nonnull Value value) {
        return Field.isMixed(this.cfg, value);
    }

    /**
     * Returns {@code true} if bulk edit is allowed on this field; {@code false}
     * otherwise.
     *
     * A field could be bulk edited if it has a property named {@code allowBulkEdit}
     * which has the value of {@code true}.
     *
     * @return True if the field component could be bulk edited, false otherwise
     */
    public boolean isBulkEditAllowed() {
        return this.cfg.get("allowBulkEdit", false);
    }

    /**
     * Returns {@code true} if bulk edit mode is set; {@code false} otherwise
     *
     * Bulk edit mode is set if the request has an attribute
     * {@code BULK_EDIT_MODE_ATTRIBUTE} which has the value of {@code true}
     *
     * @param request
     *            The request providing the parameter
     *
     * @return True if the bulk edit mode is set
     */
    public static boolean isBulkEditMode(@Nonnull HttpServletRequest request) {
        Boolean isBulkEditMode = (Boolean) request.getAttribute(BULK_EDIT_MODE_ATTRIBUTE);
        return isBulkEditMode == null ? false : isBulkEditMode;
    }

    /**
     * Sets the bulk edit mode
     *
     * @param request
     *            The request that would hold the parameter
     */
    public static void setBulkEditMode(@Nonnull HttpServletRequest request) {
        request.setAttribute(BULK_EDIT_MODE_ATTRIBUTE, true);
    }

    /**
     * Clears the bulk edit mode
     *
     * @param request
     *            The request that was holding the parameter
     */
    public static void clearBulkEditMode(@Nonnull HttpServletRequest request) {
        request.removeAttribute(BULK_EDIT_MODE_ATTRIBUTE);
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows for
     * example to hide empty field components in read only mode.
     *
     * @param cfg
     *            The config object of the field component
     * @param isEmpty
     *            Indicate if the field is empty
     * @return the CSS class (or space separated class values)
     */
    @Nonnull
    public static String getRootClass(@Nonnull Config cfg, boolean isEmpty) {
        String cls = "";

        // empty field is hidden in default mode if all of the following
        // conditions are true:
        // - value is empty
        // - config sets renderReadOnly = true (otherwise there is no rendering
        // for the default mode)
        // - config doesn't set showEmptyInReadOnly = true
        if (isEmpty && cfg.get("renderReadOnly", false) && !cfg.get("showEmptyInReadOnly", false)) {
            cls += HIDE_IN_DEFAULT_CLASS;
        }

        return cls;
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows for
     * example to hide empty field components in read only mode.
     *
     * @param cfg
     *            The config object of the field component
     * @param value
     *            The value of the field component
     * @return the CSS class (or space separated class values)
     */
    @Nonnull
    public static String getRootClass(@Nonnull Config cfg, @CheckForNull String value) {
        return getRootClass(cfg, value == null || "".equals(value));
    }

    /**
     * Returns {@code true} if the field is a mixed value; {@code false} otherwise.
     *
     * A field is a mixed value if it has a property named
     * {@code cfg.get("name") + IS_MIXED_SUFFIX} which has the value of
     * {@code true}.
     *
     * @param cfg
     *            The config object of the field component
     * @param value
     *            The form value
     * @return True if the field component holds a mixed value, false otherwise
     */
    public static boolean isMixed(@Nonnull Config cfg, @Nonnull Value value) {
        String name = cfg.get("name") + IS_MIXED_SUFFIX;
        return value.getContentValue(name, false);
    }
}
