/*
 * File: CCITTFaxOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

public class CCITTFaxOutputStream extends FilterOutputStream
{
	private int mCols = 1728;
	private int[] mRunsThisScan;
	private byte[] mbytesThisScan;
	private int mScanByteCounter;
	private int mLeftOverBits;
	private int mLeftOverCount;

	/*
	 * This is the normal constructor since "Columns" is effectively required,
	 * even though its value defaults to 1728. See section 3.3.5 of the PDF spec.
	 */
	public CCITTFaxOutputStream(OutputStream outStm, FilterParams params)
	{
		super(outStm);
		
		if(params != null)
		{
			Integer cols = (Integer)params.get(FilterParams.Columns_K);
			if (cols != null)
				mCols = cols.intValue();
		}
		
		mRunsThisScan = new int[mCols+4];
		mRunsThisScan[0] = 1;
		mRunsThisScan[1] = mCols;
		mbytesThisScan = new byte[(mCols+7) / 8];
	}

	/*
	 * Default constructor
	 */
	public CCITTFaxOutputStream(OutputStream outStm)
	{
		this(outStm, null);
	}

	/*
	 * Write data to the scan buffer and compress a scan when the
	 * scan buffer becomes full.
	 */
	@Override
	public void write(int b)
		throws IOException
	{
		mbytesThisScan[mScanByteCounter++] = (byte)b;
		if (mScanByteCounter == mbytesThisScan.length) {
			compressScan();
			mScanByteCounter = 0;
		}
	}

	/*
	 * Write data to the scan buffer and compress a scan when the
	 * scan buffer becomes full.
	 */
	@Override
	public void write(byte b[], int off, int len)
		throws IOException
	{
		int count = b.length - off;
		if (count > len)
			count = len;
		while (count > 0) {
			int curSpace = mbytesThisScan.length - mScanByteCounter;
			int curCount = (count < curSpace) ? count : curSpace;
			System.arraycopy(b, off, mbytesThisScan, mScanByteCounter, curCount);
			off += curCount;
			count -= curCount;
			if (curCount == curSpace) {
				compressScan();
				mScanByteCounter = 0;
			} else {
				mScanByteCounter += curCount;
			}
		}
	}

	/*
	 * Write data to the scan buffer and compress a scan when the
	 * scan buffer becomes full.
	 */
	@Override
	public void write(byte b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	/*
	 * Close the filter, after flushing out any residual encoded bits.
	 */
	@Override
	public void close()
		throws IOException
	{
		while (mLeftOverCount > 0) {
			out.write((mLeftOverBits >> 24) & 0xFF);
			mLeftOverBits <<= 8;
			mLeftOverCount -= 8;
		}
		super.close();
	}

	/*
	 * Compress one scan line from the full scan buffer. First move the
	 * run buffer to the previous one and convert the new scan buffer to
	 * runs.
	 */
	private void compressScan()
		throws IOException
	{
		int[] mRunsLastScan = mRunsThisScan;
		mRunsThisScan = new int[mCols+4];
		convertToRuns(mbytesThisScan, mRunsThisScan, mCols);
		code2DCompressed(mRunsThisScan, mRunsLastScan);
	}

	/*
	 * Convert the passed scan buffer to runs, which are placed in the
	 * passed run buffer. The "cols" variable is set at construction and
	 * is required for the FAX encoder to work properly.
	 */
	private static void convertToRuns(byte[] srcData, int[] dstRuns, int cols)
	{
		int runIdx = 1;
		int curLen = 0;
		boolean runIsWhite = true;
		for (int i = 0; i < srcData.length; i++) {
			byte curByte = srcData[i];
			int bitCnt = (i < srcData.length - 1) ? 8 : cols - (8 * (srcData.length - 1));
			if (curByte == (runIsWhite ? 0xFF : 0)) {
				curLen += bitCnt;
				continue;
			}
			for (int j = 0; j < bitCnt; j++) {
				boolean bitIsWhite = (curByte & 0x80) != 0;
				curByte <<= 1;
				if (bitIsWhite == runIsWhite) {
					curLen++;
				} else {
					dstRuns[runIdx++] = curLen;
					curLen = 1;
					runIsWhite ^= true;
				}
			}
		}
		dstRuns[runIdx++] = curLen;
		dstRuns[0] = runIdx - 1;
	}

	/*
	 * This is the core of the FAX encoder. We only ever encode two
	 * dimensional. There is no requirement to encode the old format,
	 * just to be able to read it. We force K to -1 in the stream parameters
	 * accordingly so it matches. This code was ported from Code2DCompressed
	 * in ccmpec.cpp in the Acrobat/PDFLib code base. It has been cleaned up
	 * and repurposed for Java, but the algorithm is unmodified in any way.
	 * If you are unfortunate enough to need to understand how this code
	 * really works, and why, see the Acrobat code and/or refer to ITU T6.
	 */
	private void code2DCompressed(int[] scanRuns, int[] prevRuns)
		throws IOException
	{
		int a0 = 0;
		int b1 = prevRuns[1];
		int prevIdx = 2;
		int runIdx = 1;
		int runLim = scanRuns[0] + 1;
		boolean isBlack = false;
		while (runIdx < runLim) {
			int a1 = a0 + scanRuns[runIdx];
			int passes = 0;
			while ((b1 <= a0) && ((0 < a0) || isBlack)) {
				b1 += prevRuns[prevIdx] + prevRuns[prevIdx+1];
				prevIdx += 2;
			}
			int b2;
			while ((b2 = (b1 + prevRuns[prevIdx])) < a1) {
				passes++;
				b1 = b2 + prevRuns[prevIdx+1];
				prevIdx += 2;
			}
			if ((a1 <= (b1 + 3)) && (b1 <= (a1 + 3)) && (passes < 6)) {
				while (0 < passes--) {
					nextCode(PASSCODE);
				}
				nextCode(vertCodes[3 + (a1 - b1)]);
				isBlack ^= true;
				if (b1 <= a1) {
					b1 += prevRuns[prevIdx++];
				}
				else
					b1 -= prevRuns[--prevIdx];
				runIdx++;
			} else {
				a1 += scanRuns[runIdx+1];
				nextCode(HORIZONTALCODE);
				code1DRun(scanRuns[runIdx], isBlack);
				code1DRun(scanRuns[runIdx+1], !isBlack);
				runIdx += 2;
			}
			a0 = a1;
		}
	}

	/*
	 * Convert one code short from the "standard" format and emit it.
	 * The low order four bits are the code length and the high twelve
	 * are the value. There are code values up to thirteen bits long, but
	 * the trick is that the high order bit of such values is always zero!
	 * Note that the code data bits actually used are left justified in
	 * code short.
	 */
	private void nextCode(int code)
		throws IOException
	{
		int codeLen = code & 0xF;
		code >>= (codeLen < 12) ? 16 - codeLen : 4;
		while (mLeftOverCount >= 8) {
			out.write((mLeftOverBits >> 24) & 0xFF);
			mLeftOverBits <<= 8;
			mLeftOverCount -= 8;
		}
		mLeftOverBits &= (-1 << (32 - mLeftOverCount));
		mLeftOverBits |= (code << (32 - mLeftOverCount - codeLen));
		mLeftOverCount += codeLen;
	}

	/*
	 * Code up a one dimensional run when we fall back to horizontal mode.
	 */
	private void code1DRun(int runLen, boolean isBlack)
		throws IOException
	{
		int[] termCodesTable = isBlack ? termCodesBlack : termCodesWhite;
		if (runLen >= NTERMCODES) {
			while (runLen > NTERMCODES*NMAKEUPCODES) {
				nextCode(EXTRALONGRUN);
				runLen -= NTERMCODES*NMAKEUPCODES;
			}
			if (runLen >= NTERMCODES) {
				int[] makeupCodesTable = isBlack ? makeupCodesBlack : makeupCodesWhite;
				nextCode(makeupCodesTable[(runLen / NTERMCODES) - 1]);
				runLen %= NTERMCODES;
			}
		}
		nextCode(termCodesTable[runLen]);
	}

	/*
	 * These are the magic code words whose encoding is explained at nextCode.
	 * They were cut and pasted from Acrobat, but their values are right out of
	 * the T6 spec.
	 */
	private static final int NTERMCODES = 64;
	private static final int NMAKEUPCODES = 40;
	private static final int PASSCODE = 0x1004;
	private static final int HORIZONTALCODE = 0x2003;
	private static final int EXTRALONGRUN = 0x01FC;

	private static final int[] vertCodes = {0x0407,	0x0806, 0x4003, 0x8001, 0x6003, 0x0C06, 0x0607};

	private static final int[] termCodesWhite = {
		0x3508, 0x1C06, 0x7004, 0x8004, 0xB004, 0xC004, 0xE004, 0xF004,
		0x9805, 0xA005, 0x3805, 0x4005, 0x2006, 0x0C06, 0xD006, 0xD406,
		0xA806, 0xAC06, 0x4E07, 0x1807, 0x1007, 0x2E07, 0x0607, 0x0807,
		0x5007, 0x5607, 0x2607, 0x4807, 0x3007, 0x0208, 0x0308, 0x1A08,
		0x1B08, 0x1208, 0x1308, 0x1408, 0x1508, 0x1608, 0x1708, 0x2808,
		0x2908, 0x2A08, 0x2B08, 0x2C08, 0x2D08, 0x0408, 0x0508, 0x0A08,
		0x0B08, 0x5208, 0x5308, 0x5408, 0x5508, 0x2408, 0x2508, 0x5808,
		0x5908, 0x5A08, 0x5B08, 0x4A08, 0x4B08, 0x3208, 0x3308, 0x3408
	};

	private static final int[] makeupCodesWhite = {
		0xD805, 0x9005, 0x5C06, 0x6E07, 0x3608, 0x3708, 0x6408, 0x6508,
		0x6808, 0x6708, 0x6609, 0x6689, 0x6909, 0x6989, 0x6A09, 0x6A89,
		0x6B09, 0x6B89, 0x6C09, 0x6C89, 0x6D09, 0x6D89, 0x4C09, 0x4C89,
		0x4D09, 0x6006, 0x4D89, 0x010B, 0x018B, 0x01AB, 0x012C, 0x013C,
		0x014C, 0x015C, 0x016C, 0x017C, 0x01CC, 0x01DC, 0x01EC, 0x01FC
	};

	private static final int[] termCodesBlack = {
		0x0DCA, 0x4003, 0xC002, 0x8002, 0x6003, 0x3004, 0x2004, 0x1805,
		0x1406, 0x1006, 0x0807, 0x0A07, 0x0E07, 0x0408, 0x0708, 0x0C09,
		0x05CA, 0x060A, 0x020A, 0x0CEB, 0x0D0B, 0x0D8B, 0x06EB, 0x050B,
		0x02EB, 0x030B, 0x0CAC, 0x0CBC, 0x0CCC, 0x0CDC, 0x068C, 0x069C,
		0x06AC, 0x06BC, 0x0D2C, 0x0D3C, 0x0D4C, 0x0D5C, 0x0D6C, 0x0D7C,
		0x06CC, 0x06DC, 0x0DAC, 0x0DBC, 0x054C, 0x055C, 0x056C, 0x057C,
		0x064C, 0x065C, 0x052C, 0x053C, 0x024C, 0x037C, 0x038C, 0x027C,
		0x028C, 0x058C, 0x059C, 0x02BC, 0x02CC, 0x05AC, 0x066C, 0x067C
	};

	private static final int[] makeupCodesBlack = {
		0x03CA, 0x0C8C, 0x0C9C, 0x05BC, 0x033C, 0x034C, 0x035C, 0x06CD,
		0x06DD, 0x04AD, 0x04BD, 0x04CD, 0x04DD, 0x072D, 0x073D, 0x074D,
		0x075D, 0x076D, 0x077D, 0x052D, 0x053D, 0x054D, 0x055D, 0x05AD,
		0x05BD, 0x064D, 0x065D, 0x010B, 0x018B, 0x01AB, 0x012C, 0x013C,
		0x014C, 0x015C, 0x016C, 0x017C, 0x01CC, 0x01DC, 0x01EC, 0x01FC
	};
}
