package com.adobe.internal.pdftoolkit.core.filter;

import java.io.OutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;


/**
 * RunLengthOutputStream
 *
 *	Important: This filter assumes the output stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1996 Adobe Systems Incorporated
 */
public class RunLengthOutputStream extends FilterOutputStream
{
	// Variables
	private static final int	ED = 128;	/* EOD */

	private boolean	repeat;
	private int		count;
	private byte	repeatChar;
	private byte	single[];
	private long    totalOut;



    // Constructors
    public RunLengthOutputStream(OutputStream  out)
	{
    	super(out);
		initOutput();
	}


    // Methods
	private void initOutput()
	{
		count = -1;	/* Starting new sequence */
		single = new byte[128];
		totalOut = 0;
	}


	private void newSequence(int b)
	{
		repeat = false;
		count = 0;
		repeatChar = (byte)b;
		single[0] = (byte)b;
	}


	private void dumpRun(int count) throws IOException
	{
		if (count >= 0) {
			out.write(count);
			for (int i = 0; i <= count; i++)
				out.write(single[i] & 0x0ff);
			totalOut += count+2;
		}
	}

    /**
     * Writes a byte. Will block until the byte is actually
     * written.
     * @param b the byte
     * @exception IOException If an I/O error has occurred.
     */
	@Override
	public void write(int b) throws IOException
	{
		if (count < 0) {	/* New Sequence Start */
			newSequence(b);
		}
		else if (repeat) {
			if ((byte)b != repeatChar || count == 127) {
				out.write(256 - count);
				out.write(repeatChar & 0x0ff);
				totalOut += 2;
				newSequence(b);
			}
			else
				++count;
		}
		else {
			if (count > 0 && (byte)b == single[count] && (byte)b == single[count-1]) {
				dumpRun(count-2);
				newSequence(b);
				repeat = true;
				count = 2;	/* Initial run of 3 repeated characters */
			}
			else if (count == 127 && (byte)b == single[count]) {
				dumpRun(count-1);
				newSequence(b);
				repeat = true;
				count = 1;	/* Initial run of 2 repeated characters */
			}
			else if (count == 127) {	/* Full sequence, dump it and start again */
				dumpRun(count);
				newSequence(b);
			}
			else {						/* Add one more to previous sequence */
				single[++count] = (byte)b;
			}
		}
	}

    /**
     * Writes a subarray of bytes.
     * @param b	the data to be written
     * @param off	the start offset in the data
     * @param len	the number of bytes that are written
     * @exception IOException If an I/O error has occurred.
     */
    @Override
	public void write(byte  b[], int  off, int  len) throws IOException
	{
		int	maxWrite = b.length-off;

		if (maxWrite > len)
			maxWrite = len;

		while (maxWrite-- > 0) {
			write(b[off++] & 0x0ff);
		}
	}

    /**
     * Writes an array of bytes. Will block until the bytes
     * are actually written.
     * @param b	the data to be written
     * @exception IOException If an I/O error has occurred.
     */
	@Override
	public void write(byte  b[]) throws IOException
	{
		write(b, 0, b.length);
	}

    /**
     * Closes the stream and writes the stream trailer "0x80".
     * This method must be called to release any resources
     * associated with the stream.
     * @exception IOException If an I/O error has occurred.
     */
	@Override
	public void close() throws IOException
	{
		if (repeat) {
			out.write(256 - count);
			out.write(repeatChar);
			totalOut += 2;
		}
		else
			dumpRun(count);

		out.write(ED);
		++totalOut;
		super.close();
	}

    /**
     * Counts the number of bytes written by this filter.
     * @return	actual number of bytes written
     */
	public long getTotalOut()
	{
	    return totalOut;
	}
}

