/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.mobile.api.device;

import com.day.cq.wcm.emulator.Emulator;
import com.day.cq.wcm.mobile.api.device.capability.DeviceCapability;

import javax.servlet.ServletException;
import javax.servlet.jsp.PageContext;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

/**
 * A <code>DeviceGroup</code> represents a segment of mobile devices. The segment is defined by the mobile device
 * capabilities expected by the implementation.
 *
 * @since CQ 5.4.0
 */
public interface DeviceGroup {

    final static String RESOURCE_TYPE = "wcm/mobile/components/devicegroup";

    /**
     * Returns all {@link DeviceCapability}s supported by this device group.
     *
     * @return A <code>Collection</code> containing all supported capabilities. If no capabilities are defined, an empty
     *         collection is returned.
     */
    Collection<DeviceCapability> getCapabilities();

    /**
     * Returns the {@link Emulator} configured for this device group.
     *
     * @return The emulator.
     */
    Emulator getDefaultEmulator();

    /**
     * Return the description of this device group as entered by the user.
     *
     * @return The description of this device group or <code>null</code> if not defined.
     */
    String getDescription();

    /**
     * Returns the emulators associated with this device.
     *
     * @return The emulators.
     */
    List<Emulator> getEmulators();

    /**
     * Returns the minimum screen width devices must support in order to belong to this group. The width is in pixels.
     *
     * @return The minimum screen width in pixels. If not defined, <code>0</code> is returned, meaning all widths are
     *         accepted.
     */
    int getMinimumScreenWidth();

    /**
     * Returns the minimum screen height devices must support in order to belong to this group. The height is in
     * pixels.
     *
     * @return The minimum screen height in pixels. If not defined, <code>0</code> is returned, meaning all heights are
     *         accepted.
     */
    int getMinimumScreenHeight();

    /**
     * Return the unique name of this device group.
     *
     * @return A <code>String</code> representing the name of this device group.
     */
    String getName();

    /**
     * Return the path of the content page representing this device group.
     *
     * @return A <code>String</code> representing the path of this device group.
     */
    String getPath();

    /**
     * Returns the title of this device group as entered by the user.
     *
     * @return The device group title or <code>null</code> if not defined.
     */
    String getTitle();

    /**
     * A device class may optionally specify a mobile client user-agent string which devices must match in order to
     * belong to this group. If no user-agent string is defined, <code>null</code> is returned.
     *
     * @return A <code>String</code> representing the user-agent string expected by this device group, or
     *         <code>null</code> if not defined.
     */
    String getUserAgent();

    /**
     * Checks whether the device group supports the given {@link DeviceCapability}.
     *
     * @param capability The capability to check whether supported by the device group.
     *
     * @return <code>true</code> if the device group supports the capability.
     */
    boolean hasCapability(DeviceCapability capability);

    /**
     * Checks whether the device group supports the given {@link DeviceCapability}.
     *
     * @param name The name of the capability to check whether supported by the device group.
     *
     * @return <code>true</code> if the device group supports the capability.
     */
    boolean hasCapability(String name);

    /**
     * This method is to be used in a JSP script drawing the HTML head. The method will provide all necessary JS and CSS
     * includes and other initialization required for the device group emulator to work (in authoring mode only) and
     * providing custom CSS for rendering.
     *
     * @param pageContext The JSP page context.
     *
     * @throws IOException      If an error occurred during writing to the response.
     * @throws ServletException If an error occurred including the emulator's init component.
     */
    void drawHead(final PageContext pageContext) throws IOException, ServletException;

    /**
     * Indicates whether the emulator shall be shown during editing of a page or not. Even if the emulator is disabled
     * for content editing, it will still be shown for preview modes.
     *
     * @return <code>true</code> if the emulator shall be shown.
     */
    boolean showEditEmulator();

    /**
     * Returns the regex pattern expected for the mobile device user agent to match.
     *
     * @return The pattern.
     */
    Pattern getUserAgentPattern();

    /**
     * Returns the path of "static.css" if the device group possesses one. The static.css is used for styling device
     * group specifically the content of a page.
     *
     * @return A <code>String</code> representing the static path of the CSS of this device group, or <code>null</code>
     *         if no CSS is present.
     */
    String getStaticCssPath();

    /**
     * Returns the names of filters as configured on a device group page.
     *
     * @return The names.
     *
     * @see DeviceMapper#getFilters()
     */
    String[] getFilterNames();
}
