package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.LayoutTableSubsetter.ClassCoveredBySubset;
import com.adobe.fontengine.font.opentype.LayoutTableSubsetter.ClassDefGenerator;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.LookupSubset;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.LookupSubtableGenerator;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.NewCoverage;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

class ChainingGenerator {
	private static final boolean warnAboutDroppedSublookups = false;
	
	static LookupSubtableGenerator newChainingInstance(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, Subset subset, 
			Map newCoverages, LookupSubset lookupSubset, int origNumGlyphs) throws InvalidFontException 
	{
  	  int format = origTable.data.getuint16(origSTOffset);
	  switch (format)
	  {
	  case 1:
		  return ChainingFormat1Generator.newInstance(origSTOffset, newSTOffset, origTable, subset, builder,  newCoverages,lookupSubset);
		  
	  case 2:
		  return ChainingFormat2Generator.newInstance(origSTOffset, newSTOffset, origTable, subset, builder, lookupSubset, origNumGlyphs);
		  
	  case 3:
		  return ChainingFormat3Generator.newInstance(origTable, builder, origSTOffset, newSTOffset, lookupSubset);
		  
	  default:
		  throw new InvalidFontException("Invalid chaining contextual lookup format: " + format);
	  }
	}
	
	static void gatherCoveragesForSubtable(LayoutTable origTable, int stOffset, int lookupType, int lookupIndex, Integer subtableIndex, 
			Map coverageMapToUse, Subset subset) throws InvalidFontException, UnsupportedFontException 
	{
  	  int stFormat = origTable.data.getuint16(stOffset);
	  switch (stFormat) {
	  case 1:
	  case 2:
	  {
		  int coverageOffset = origTable.data.getOffset (stOffset, 2);
		  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  break;
	  }
	  case 3:
	  {
		  int coverageCount = origTable.data.getuint16(stOffset + 2);
		  for (int j = 0; j < coverageCount; j++)
		  {
			  int coverageOffset = origTable.data.getOffset (stOffset, 4 + 2*j);
			  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  }
		  
		  int delta = 4 + 2*coverageCount;
		  
		  coverageCount = origTable.data.getuint16(stOffset + delta);
		  for (int j = 0; j < coverageCount; j++)
		  {
			  int coverageOffset = origTable.data.getOffset (stOffset, delta + 2 + 2*j);
			  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  }
		  
		  delta += 2 + 2*coverageCount;
		  
		  coverageCount = origTable.data.getuint16(stOffset + delta);
		  for (int j = 0; j < coverageCount; j++)
		  {
			  int coverageOffset = origTable.data.getOffset (stOffset, delta + 2 + 2*j);
			  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  }
		  break;
	  }
	  default:
		  throw new InvalidFontException ("Invalid contextual subtable format (" + stFormat + ")"); 
	  }
	}
	
	static void patchSubtableCoverage(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, 
			Map coverageInfo, int lookupType) throws InvalidFontException
	{
	int format = origTable.data.getuint16(origSTOffset);
	switch (format)
	{
	case 1:
	case 2:
		LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 2, coverageInfo);
		break;
	case 3:
		int backTrackCount = origTable.data.getuint16(origSTOffset+2);
		int inputCount = origTable.data.getuint16(origSTOffset + 4 + 2*backTrackCount);
		int lookaheadCount = origTable.data.getuint16(origSTOffset + 6 + 2*backTrackCount + 2*inputCount);
		for (int i = 0; i < backTrackCount; i++)
		{
			LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 4 + 2*i, coverageInfo);
		}
		
		for (int i = 0; i < inputCount; i++)
		{
			LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 6 + 2*backTrackCount + 2*i, coverageInfo);
		}
		
		for (int i = 0; i < lookaheadCount; i++)
		{
			LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 8 + 2*inputCount + 2*backTrackCount + 2*i, coverageInfo);
		}
		
		break;
		
	default:
			throw new InvalidFontException("Unrecognized lookup type 6 format: " + format);	
	}
	}
	
	private static class ChainingFormat1Generator extends SetGenerator implements CoverageConsumer
	{
		private final LookupSubset lookupSubset;
		private final int ruleCount;

		private ChainingFormat1Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, int origSTOffset, int subRuleSetCount, LookupSubset lookupSubset) throws InvalidFontException
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, subRuleSetCount, 4, false, true);
			this.lookupSubset = lookupSubset;
			this.ruleCount = subRuleSetCount;
		}
		
		static ChainingFormat1Generator newInstance(int origSTOffset, int newSTOffset, LayoutTable origTable, Subset subset, OTByteArrayBuilder builder, Map newCoverages, LookupSubset lookupSubset) throws InvalidFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			NewCoverage newCoverage = (NewCoverage)newCoverages.get(new Integer(coverageOffset));
			return new ChainingFormat1Generator(origTable, subset, builder, newSTOffset, origSTOffset, newCoverage.glyphCount,lookupSubset);
		}
		
		boolean[] computeMembersToKeep(int newRuleSetOffset, int origRuleSetOffset) throws InvalidFontException, UnsupportedFontException
		{
			int origRuleCount = origTable.data.getuint16(origRuleSetOffset);
			boolean[] keep = new boolean[origRuleCount];
			Arrays.fill(keep, true);
			for (int i = 0; i < origRuleCount; i++)
			{
				int ruleOffset = origTable.data.getOffset(origRuleSetOffset, 2 + 2*i);
				int backtrackCount = origTable.data.getuint16(ruleOffset);
				for (int j = 0; j < backtrackCount; j++)
				{
					// if backtrack requires glyphs not in the substitution, don't keep this subrule.
					if (subset.getExistingSubsetGid(origTable.data.getuint16(ruleOffset + 2 + 2*j)) == -1)
					{
						keep[i] = false;
						break;
					}
				}
				
				if (!keep[i]) continue;
				
				int inputCount = origTable.data.getuint16(ruleOffset + 2 + 2*backtrackCount);
				for (int j = 0; j < inputCount-1; j++)
				{
					// if input requires glyphs not in the substitution, don't keep this subrule.
					if (subset.getExistingSubsetGid(origTable.data.getuint16(ruleOffset + 4 + 2*backtrackCount + 2*j)) == -1)
					{
						keep[i] = false;
						break;
					}
				}
				
				if (!keep[i]) continue;
				
				int lookahead = origTable.data.getuint16(ruleOffset + 4 + 2*(inputCount-1) + 2*backtrackCount);
				for (int j = 0; j < lookahead; j++)
				{
					// if lookahead requires glyphs not in the substitution, don't keep this subrule.
					if (subset.getExistingSubsetGid(origTable.data.getuint16(ruleOffset + 6 + 2*(inputCount-1) + 2*backtrackCount + 2*j)) == -1)
					{
						keep[i] = false;
						break;
					}
				}
			}
			
			return keep;
		}
		
		int writeMember(int newSubRuleOffset, int origSubRuleOffset) throws InvalidFontException, UnsupportedFontException
		{
			int backTrackCount = origTable.data.getuint16(origSubRuleOffset);
			int inputCount = origTable.data.getuint16(origSubRuleOffset + 2 + 2*backTrackCount);
			int lookaheadCount = origTable.data.getuint16(origSubRuleOffset + 4 + 2*backTrackCount + 2*(inputCount - 1));
			int substCount = origTable.data.getuint16(origSubRuleOffset + 6 + 2*backTrackCount + 2*(inputCount - 1) + 2*lookaheadCount);
			int lSize = 8 + 2*backTrackCount + 2*(inputCount - 1) + 2*lookaheadCount;
			builder.ensureCapacity(newSubRuleOffset + lSize);
			builder.setuint16(newSubRuleOffset, backTrackCount);
			
			// write backtrack.
			for (int i = 0; i < backTrackCount; i++)
			{
				int gid = subset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 2 + 2*i));
				if (gid == -1)
					throw new RuntimeException("Backtrack gid not in subset? Can't happen!");
				builder.setuint16(newSubRuleOffset + 2 + 2*i, gid);
			}
			
			newSubRuleOffset += 2 + 2*backTrackCount;
			origSubRuleOffset += 2 + 2*backTrackCount;
			
			//write input
			builder.setuint16(newSubRuleOffset, inputCount);
			for (int i = 0; i < inputCount-1; i++)
			{
				int gid = subset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 2 + 2*i));
				if (gid == -1)
					throw new RuntimeException("Input gid not in subset? Can't happen!");
				builder.setuint16(newSubRuleOffset + 2 + 2*i, gid);
			}

			newSubRuleOffset += 2 + 2*(inputCount-1);
			origSubRuleOffset += 2 + 2*(inputCount-1);

			// write lookahead
			builder.setuint16(newSubRuleOffset, lookaheadCount);
			for (int i = 0; i < lookaheadCount; i++)
			{
				int gid = subset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 2 + 2*i));
				if (gid == -1)
					throw new RuntimeException("Lookahead gid not in subset? Can't happen!");
				builder.setuint16(newSubRuleOffset + 2 + 2*i, gid);
			}
			
			newSubRuleOffset += 2 + 2*lookaheadCount;
			origSubRuleOffset += 2 + 2*lookaheadCount;

			//write substitutions
			int newSubstCount = 0;
			for (int i = 0; i < substCount; i++)
			{
				int lookup = lookupSubset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 2 + 4*i + 2));
				if (lookup != -1)
				{
					lSize += 4;
					builder.ensureCapacity(newSubRuleOffset + lSize);
					builder.setuint16(newSubRuleOffset + 2 + 4*newSubstCount, origTable.data.getuint16(origSubRuleOffset + 2 + 4*i));
					builder.setuint16(newSubRuleOffset + 2 + 4*newSubstCount + 2, lookup);
					newSubstCount++;
				}else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for chaining format 1");
				}
			}
			builder.setuint16(newSubRuleOffset, newSubstCount);
			return lSize;
		}

		int whichRuleSetIndexApplies(int gid, int coverageIndex)
		{
			return coverageIndex;
		}

		int getOrigRecordSize() {
			return 0;
		}
		
		private void writeHeader(int newSTOffset, int glyphCount)
		{
			builder.ensureCapacity(newSTOffset + 6 + 2*glyphCount);
			builder.setuint16(newSTOffset, 1);
			builder.setuint16(newSTOffset + 4, glyphCount);
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException {
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			writeHeader(newSTOffset, ruleCount);
			origTable.iterateCoverage(coverageOffset, subset, this);
			return super.writeSubtable();
		}
	}
	
	private static class ChainingFormat2Generator extends SetGenerator implements CoverageConsumer
	{
		private final LookupSubset lookupSubset;
		private final ClassCoveredBySubset classCovered;
		private final int numGlyphs;
		private final int classCount;

		private ChainingFormat2Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, 
				int origSTOffset, int classCount, LookupSubset lookupSubset, int numGlyphs) 
		throws InvalidFontException
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, classCount, 10, true, true);
			this.lookupSubset = lookupSubset;
			this.classCovered = new ClassCoveredBySubset(subset);
			this.numGlyphs = numGlyphs;
			this.classCount = classCount;
		}
		
		static ChainingFormat2Generator newInstance(int origSTOffset, int newSTOffset, LayoutTable origTable, Subset subset, 
				OTByteArrayBuilder newData, LookupSubset lookupSubset, int numGlyphs) throws InvalidFontException
		{
			int classCount = origTable.data.getuint16(origSTOffset+10);
			return new ChainingFormat2Generator(origTable, subset, newData, newSTOffset, origSTOffset, classCount, lookupSubset, numGlyphs);
		}
		
		boolean[] computeMembersToKeep(int newRuleSetOffset, int origRuleSetOffset) throws InvalidFontException, UnsupportedFontException
		{			
			int origRuleCount = origTable.data.getuint16(origRuleSetOffset);
			boolean[] keep = new boolean[origRuleCount];
			Arrays.fill(keep, true);
			for (int i = 0; i < origRuleCount; i++)
			{
				int ruleOffset = origTable.data.getOffset(origRuleSetOffset, 2 + 2*i);
				int delta = 2;
				int glyphCount = origTable.data.getuint16(ruleOffset);
				int origClassDefOffset = origTable.data.getOffset(origSTOffset, 4);
				for (int j = 0; j < glyphCount; j++)
				{
					// if a subrule requires a class that isn't covered by our subset, the subrule isn't kept.
					if (!classCovered.classCoveredBySubset(origTable, origClassDefOffset, numGlyphs, origTable.data.getuint16(ruleOffset + delta + 2*j)))
					{
						keep[i] = false;
						break;
					}
				}
				
				delta += 2*glyphCount;
				origClassDefOffset = origTable.data.getOffset(origSTOffset, 6);
				glyphCount = origTable.data.getuint16(ruleOffset + delta);
				delta += 2;
				for (int j = 0; j < glyphCount-1; j++)
				{
					// if a subrule requires a class that isn't covered by our subset, the subrule isn't kept.
					if (!classCovered.classCoveredBySubset(origTable, origClassDefOffset, numGlyphs, origTable.data.getuint16(ruleOffset + delta + 2*j)))
					{
						keep[i] = false;
						break;
					}
				}
				

				delta += 2*(glyphCount-1);
				origClassDefOffset = origTable.data.getOffset(origSTOffset, 8);
				glyphCount = origTable.data.getuint16(ruleOffset + delta);
				delta += 2;
				for (int j = 0; j < glyphCount; j++)
				{
					// if a subrule requires a class that isn't covered by our subset, the subrule isn't kept.
					if (!classCovered.classCoveredBySubset(origTable, origClassDefOffset, numGlyphs, origTable.data.getuint16(ruleOffset + delta + 2*j)))
					{
						keep[i] = false;
						break;
					}
				}
			}
			
			return keep;
		}
		
		int writeMember(int newSubRuleOffset, int origSubRuleOffset) throws InvalidFontException, UnsupportedFontException
		{
			int backtrackCount = origTable.data.getuint16(origSubRuleOffset);
			int inputCount = origTable.data.getuint16(origSubRuleOffset + 2+ 2*backtrackCount);
			int lookahead = origTable.data.getuint16(origSubRuleOffset + 2+ 2*backtrackCount  + 2+ 2*(inputCount-1));
			int substCount = origTable.data.getuint16(origSubRuleOffset + 2+ 2*backtrackCount  + 2+ 2*(inputCount-1) + 2 + 2*lookahead);
			int lSize = 2 + 2*backtrackCount  + 2 + 2*(inputCount-1) + 2 + 2*lookahead + 2;
			builder.ensureCapacity(newSubRuleOffset + lSize);
			
			// write backtrack
			builder.setuint16(newSubRuleOffset, backtrackCount);
			int delta = 2;
			for (int i = 0; i < backtrackCount; i++)
			{
				builder.setuint16(newSubRuleOffset + delta, origTable.data.getuint16(origSubRuleOffset + delta));
				delta += 2;
			}
			
			// write input
			builder.setuint16(newSubRuleOffset + delta, inputCount);
			delta += 2;
			for (int i = 0; i < inputCount - 1; i++)
			{
				builder.setuint16(newSubRuleOffset + delta, origTable.data.getuint16(origSubRuleOffset + delta));
				delta += 2;
			}

			// write lookahead
			builder.setuint16(newSubRuleOffset + delta, lookahead);
			delta += 2;
			for (int i = 0; i < lookahead; i++)
			{
				builder.setuint16(newSubRuleOffset + delta, origTable.data.getuint16(origSubRuleOffset + delta));
				delta += 2;
			}
			
			// write substitutions
			int substcountDelta = delta;
			int newSubstCount = 0;
			delta += 2;
			for (int i = 0; i < substCount; i++)
			{
				int newLookup = lookupSubset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + delta + 2));
				if (newLookup != -1)
				{
					builder.ensureCapacity(newSubRuleOffset + lSize + 4);
					builder.setuint16(newSubRuleOffset + lSize, origTable.data.getuint16(origSubRuleOffset + delta));
					builder.setuint16(newSubRuleOffset + lSize + 2, newLookup);
					lSize += 4;
					newSubstCount++;
				}else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for chaining format 2");
				}
				delta += 4;
			}
			
			builder.setuint16(newSubRuleOffset + substcountDelta, newSubstCount);
			
			return lSize;
		}
		
		void writeClassDef(int stOffsetDelta) throws InvalidFontException, UnsupportedFontException
		{
			ClassDefGenerator generator = ClassDefGenerator.newInstance(origTable, origTable.data.getOffset(origSTOffset, stOffsetDelta), subset, numGlyphs);
			builder.setuint16(this.newSTOffset + stOffsetDelta, subtableSize);
			OTByteArray array = generator.generateClass().toOTByteArray();
			subtableSize += writeByteArrayAtOffset(newSTOffset + subtableSize, array);
		}

		int whichRuleSetIndexApplies(int gid, int coverageIndex) throws InvalidFontException
		{
			return origTable.getClassIndex(gid, origTable.data.getOffset(origSTOffset, 6));
		}

		int getOrigRecordSize() {
			return 0;
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException {
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			builder.ensureCapacity(newSTOffset + 12 + 2*classCount);
			builder.setuint16(newSTOffset, 2);
			builder.setuint16(newSTOffset + 10, classCount);
			
			origTable.iterateCoverage(coverageOffset, subset, this);
			writeClassDef(4);
			writeClassDef(6);
			writeClassDef(8);
			return subtableSize;
		}
	}
	
	private static class ChainingFormat3Generator implements LookupSubtableGenerator
	{
		private final LayoutTable origTable;
		private final OTByteArrayBuilder builder;
		private int origSTOffset;
		private int newSTOffset;
		private final LookupSubset lookupSubset;
		
		private ChainingFormat3Generator(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, LookupSubset lookupSubset)
		{
			this.origTable = origTable;
			this.builder = builder;
			this.origSTOffset = origSTOffset;
			this.newSTOffset = newSTOffset;
			this.lookupSubset = lookupSubset;
		}
		
		static ChainingFormat3Generator newInstance(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, LookupSubset lookupSubset)
		{
			return new ChainingFormat3Generator(origTable, builder, origSTOffset, newSTOffset, lookupSubset);
		}
		
		public int writeSubtable() throws InvalidFontException
		{
			int backTrackCount = origTable.data.getuint16(origSTOffset+2);
			origSTOffset += 4 + 2*backTrackCount;
			int inputCount = origTable.data.getuint16(origSTOffset);
			origSTOffset += 2 + 2*inputCount;
			int lookaheadCount = origTable.data.getuint16(origSTOffset);
			origSTOffset += 2 + 2*lookaheadCount;
			int substCount = origTable.data.getuint16(origSTOffset);
			origSTOffset += 2;
			
			int size = 10 + 2*backTrackCount + 2*inputCount + 2*lookaheadCount;

			builder.ensureCapacity(newSTOffset + size);
			builder.setuint16(newSTOffset, 3); // format
			builder.setuint16(newSTOffset+2, backTrackCount);// backtrack coverage count
			newSTOffset += 4 + 2*backTrackCount;
			builder.setuint16(newSTOffset, inputCount);// input coverage count
			newSTOffset += 2 + 2*inputCount;
			builder.setuint16(newSTOffset, lookaheadCount); // lookahead coverage count
			newSTOffset += 2 + 2*lookaheadCount;
			
			// write the new substitutions.
			int newSubstCount = 0;
			for (int i = 0; i < substCount; i++)
			{
				int newValue = lookupSubset.getExistingSubsetGid(origTable.data.getuint16(origSTOffset + 4*i + 2));
				if (newValue != -1)
				{
					size += 4;
					builder.ensureCapacity(newSTOffset + size);
					builder.setuint16(newSTOffset + 2 + 4*newSubstCount, origTable.data.getuint16(origSTOffset + 4*i));
					builder.setuint16(newSTOffset + 2 + 4*newSubstCount + 2, newValue);
					newSubstCount++;
				}else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for chaining format 3");
				}
			}
			
			builder.setuint16(newSTOffset, newSubstCount); //substitution count
			
			return size;
			
		}
	}
}
