/*
 *
 *	File: Glyf.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.Map;
import java.io.UnsupportedEncodingException;
import java.io.IOException;

import com.adobe.fontengine.font.OrigFontType;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.cff.AutoColor;
import com.adobe.fontengine.font.cff.CIDKeyedFont;
import com.adobe.fontengine.font.cff.CharStrings;
import com.adobe.fontengine.font.cff.Dict;
//import com.adobe.fontengine.font.cff.NameKeyedFont;
import com.adobe.fontengine.font.cff.CFFSubrize;
import com.adobe.fontengine.font.cff.Type2CStringGenerator;
//import com.adobe.fontengine.font.cff.NameKeyedFont.GlyphNameFetcher;
import com.adobe.fontengine.font.cff.NonOverlappingOutlineConsumer;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;
import com.adobe.fontengine.font.opentype.Name;
import com.adobe.fontengine.math.F16Dot16;
import com.adobe.fontengine.math.F2Dot14;

/** Gives access to the 'glyf' table.
 *
 * <p>Note that there is no public representation of the 'loca' table,
 * which is used only to access the 'glyf' table. Instead, 
 * the <code>Glyph</code> objects accept gids to identify glyphs and 
 * perform the necessary lookup in the 'loca' table.</p>
 * 
 * <h4>Version handling</h4>
 * 
 *  <p>'glyf' tables are not versionned.</p>
 *
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Glyf extends Table {

  // The 'glyf' table does not stand on its own; 
  // the descriptions of the glyphs are one after the other, and 
  // the locations of each glyph is in the 'loca' table.
  // The 'loca' table itself does not stand on its own, as 
  // one needs the 'head.indexToLocFormat' field to interpret it.

  // So this class collects all these pieces and gives the illusion
  // of a standalone 'glyf' table.

  protected final LocaRaw loca;
  protected final GlyfRaw glyf;
  protected final int indexToLocFormat;

  protected Glyf (GlyfRaw glyf, LocaRaw loca, int indexToLocFormat) {
	// In FontFactory we ignore 0 length tables but it is possible for
	// a valid font to not have glyf or have a zero length glyf table.
	// By constructing the empty GlyfRaw obj the Glyf class should 
	// handle calls appropriately.
	if (glyf == null) {
		glyf = new GlyfRaw();
	} 
    this.glyf = glyf;
    this.loca = loca;
    this.indexToLocFormat = indexToLocFormat;
  }

  protected int getGlyphLocation (int gid) 
  throws UnsupportedFontException, InvalidFontException {
    return loca.getGlyphLocation (gid, indexToLocFormat);
  }

  /** Get the bounding box for a glyph. */
  public Rect getGlyphBoundingBox (int gid) 
  throws UnsupportedFontException, InvalidFontException {
    return glyf.getGlyphBoundingBox (getGlyphLocation (gid), getGlyphLocation (gid+1));
  }

  /** Return whether is glyph is composite. */
  public boolean isComposite (int gid) 
  throws UnsupportedFontException, InvalidFontException {
    return glyf.isComposite (getGlyphLocation (gid), getGlyphLocation (gid+1));
  }

  public interface Flags {
    public static final int ARG_1_AND_2_ARE_WORDS      = 0x0001;
    public static final int ARGS_ARE_XY_VALUES   = 0x0002;
    public static final int ROUND_XY_TO_GRID           = 0x0004;
    public static final int WE_HAVE_A_SCALE            = 0x0008;
    public static final int MORE_COMPONENTS            = 0x0020;
    public static final int WE_HAVE_AN_X_AND_Y_SCALE   = 0x0040;
    public static final int WE_HAVE_A_TWO_BY_TWO       = 0x0080;
    public static final int WE_HAVE_INSTRUCTIONS       = 0x0100;
    public static final int USE_MY_METRICS             = 0x0200;
    public static final int OVERLAP_COMPOUND           = 0x0400;
    public static final int SCALED_COMPONENT_OFFSET    = 0x0800;
    public static final int UNSCALED_COMPONENT_OFFSET  = 0x1000;
  }

  public interface CoordFlags {
    public static final int ON_CURVE = 0x1;
    public static final int X_SHORT_VECTOR = 0x2;
    public static final int Y_SHORT_VECTOR = 0x4;
    public static final int REPEAT = 0x8; 
    public static final int THIS_X_IS_SAME = 0x10;
    public static final int THIS_Y_IS_SAME = 0x20;
  }


  public void subsetAndStream (Subset subset, Map tables) 
  throws UnsupportedFontException, InvalidFontException {	  
    OTByteArrayBuilder locaData 
    = OTByteArray.getOTByteArrayBuilderInstance((subset.getNumGlyphs () + 1) * 4);

    // calculate the size that the new glyf table needs to be
    int newGlyfTableSize = 0;
    for (int i = 0; i < subset.getNumGlyphs (); i++) {
      int fullGid = subset.getFullGid (i);
      long start = getGlyphLocation (fullGid);
      long limit = getGlyphLocation (fullGid + 1);
      if (limit < start) {
        throw new InvalidFontException ("gid " + fullGid 
            + " starts at " + start + " and ends at " + limit + "!");  }
      newGlyfTableSize += limit - start; }
    OTByteArrayBuilder glyfData 
        = OTByteArray.getOTByteArrayBuilderInstance(newGlyfTableSize);

    int position = 0;
    for (int i = 0; i < subset.getNumGlyphs (); i++) {
      locaData.setuint32(i*4, position);

      int fullGid = subset.getFullGid (i);
      long start = getGlyphLocation (fullGid);
      long limit = getGlyphLocation (fullGid + 1);
      if ((limit - start) > 0) {
		glyfData.replace (position, glyf.data, (int) start, (int) (limit - start));
        fixComponentGlyphs (subset, (int) start, glyfData, position); }
      position += limit - start; }

    locaData.setuint32(subset.getNumGlyphs ()*4, position); 

    tables.put (new Integer (Tag.table_loca), locaData); 
    tables.put (new Integer (Tag.table_glyf), glyfData);
  }  

  public void stream(Map tables) {
	  OTByteArrayBuilder newData = loca.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_loca), newData);
	  
	  newData = glyf.getDataAsByteArray();
	  tables.put(new Integer(Tag.table_glyf), newData);
  }
  
  public void pullComponentGlyphs (Subset subset, int gid)
  throws UnsupportedFontException, InvalidFontException {
    int start = getGlyphLocation (gid);
    int limit = getGlyphLocation (gid + 1);

    if (start == limit) {
      return; }

    if (glyf.data.getint16 (start) != -1) {
      return; }

    start += 10;
    boolean moreComponents = true;

    while (moreComponents) {
      subset.getSubsetGid (glyf.data.getuint16 (start + 2));

      int flags = glyf.data.getuint16 (start);
      int blockSize = 4;

      if ((flags & Flags.ARG_1_AND_2_ARE_WORDS) != 0) {
        blockSize += 4; }
      else {
        blockSize += 2; }

      if ((flags & Flags.WE_HAVE_A_SCALE) != 0) {
        blockSize += 2; }
      else if ((flags & Flags.WE_HAVE_AN_X_AND_Y_SCALE) != 0) {
        blockSize += 4; }
      else if ((flags & Flags.WE_HAVE_A_TWO_BY_TWO) != 0) {
        blockSize += 8; }

      start += blockSize;
      moreComponents = ((flags & Flags.MORE_COMPONENTS) != 0); }
  }

  void fixComponentGlyphs (Subset subset, int start, OTByteArrayBuilder data, int offset)
  throws UnsupportedFontException, InvalidFontException {
    if (glyf.data == null || glyf.data.getint16 (start) != -1) {
      return; }

    start += 10;
    offset += 10;
    boolean moreComponents = true;

    while (moreComponents) {
      data.setuint16 (offset + 2, subset.getSubsetGid (glyf.data.getuint16 (start + 2)));

      int flags = glyf.data.getuint16 (start);
      int blockSize = 4;

      if ((flags & Flags.ARG_1_AND_2_ARE_WORDS) != 0) {
        blockSize += 4; }
      else {
        blockSize += 2; }

      if ((flags & Flags.WE_HAVE_A_SCALE) != 0) {
        blockSize += 2; }
      else if ((flags & Flags.WE_HAVE_AN_X_AND_Y_SCALE) != 0) {
        blockSize += 4; }
      else if ((flags & Flags.WE_HAVE_A_TWO_BY_TWO) != 0) {
        blockSize += 8; }

      start += blockSize;
      offset += blockSize;

      moreComponents = ((flags & Flags.MORE_COMPONENTS) != 0); }
  }


  //==========================================================================
//font.hmtx.getLeftSideBearing (gid), 
//font.hmtx.getHorizontalAdvance (gid),
//font.os2.getTypoAscender (), 
//font.os2.getTypoDescender ()); 


  public TTOutline createTTOutline (OpenTypeFont font, int gid)
  throws UnsupportedFontException, InvalidFontException {

    int offset = getGlyphLocation (gid);
    int limit = getGlyphLocation (gid + 1);
    	
    if (offset == limit) {
    	return createTTSimpleOutline (font, gid, offset, limit); }
    
    int numberOfContours = glyf.data.getint16 (offset);

    if (numberOfContours < 0) {
      return createTTCompositeOutline (font, gid, offset, limit); }

    else if (numberOfContours == 0) {
      throw new UnsupportedFontException ("createTTOutline, numberOfContours == 0"); }

    else {
      return createTTSimpleOutline (font, gid, offset, limit); }
  }

  TTCompositeOutline createTTCompositeOutline (OpenTypeFont font, int gid, int offset, int limit)
  throws UnsupportedFontException, InvalidFontException {

    ArrayList components = new ArrayList ();

    offset += 2; // nbContours

    int xMin = glyf.data.getint16 (offset);
    offset += 8;

    boolean moreComponents = true;
    boolean haveInstructions = false;

    while (moreComponents) {
      TTCompositeOutline.TTComponent component = new TTCompositeOutline.TTComponent ();
      int flags = glyf.data.getuint16 (offset);
      offset += 2;

      moreComponents = (flags & Flags.MORE_COMPONENTS) != 0;
      haveInstructions = (flags & Flags.WE_HAVE_INSTRUCTIONS) != 0;

      int childGid = glyf.data.getuint16 (offset);
      offset += 2;

      component.outline = createTTOutline (font, childGid);

      component.alignByPosition = (flags & Flags.ARGS_ARE_XY_VALUES) != 0;
      component.roundXYToGrid = (flags & Flags.ROUND_XY_TO_GRID) != 0;
      component.useThisMetrics = (flags & Flags.USE_MY_METRICS) != 0;
      component.scaledOffsets = (flags & Flags.SCALED_COMPONENT_OFFSET) != 0;
      component.unscaledOffset = (flags & Flags.UNSCALED_COMPONENT_OFFSET) != 0;

      if ((flags & Flags.ARG_1_AND_2_ARE_WORDS) != 0) {
        component.argument1 = glyf.data.getint16 (offset);
        offset += 2;
        component.argument2 = glyf.data.getint16 (offset);
        offset += 2; }
      else {
        component.argument1 = glyf.data.getint8 (offset);
        offset++;
        component.argument2 = glyf.data.getint8 (offset);
        offset++; }

      if ((flags & Flags.WE_HAVE_A_SCALE) != 0) {
        component.a = glyf.data.getuint16 (offset);
        offset += 2;
        component.b = F2Dot14.ZERO;
        component.c = F2Dot14.ZERO;
        component.d = component.a; }
      else if ((flags & Flags.WE_HAVE_AN_X_AND_Y_SCALE) != 0) {
        component.a = glyf.data.getuint16 (offset);
        offset += 2;
        component.b = F2Dot14.ZERO;
        component.c = F2Dot14.ZERO;
        component.d = glyf.data.getuint16 (offset);
        offset += 2; }
      else if ((flags & Flags.WE_HAVE_A_TWO_BY_TWO) != 0) {
        component.a = glyf.data.getuint16 (offset);
        offset += 2;
        component.b = glyf.data.getuint16 (offset); 
        offset += 2;
        component.c = glyf.data.getuint16 (offset);
        offset += 2;
        component.d = glyf.data.getuint16 (offset);
        offset += 2; }
      else {
        component.a = F2Dot14.ONE;
        component.b = F2Dot14.ZERO;
        component.c = F2Dot14.ZERO;
        component.d = F2Dot14.ONE; }

      components.add (component); }

    int instructionsLength = 0;
    int instructionsOffset = 0;
    if (haveInstructions) {
      instructionsLength = glyf.data.getuint16(offset);
      offset += 2;
      instructionsOffset = offset; }

    int l = xMin - font.hmtx.getLeftSideBearing (gid);
    int r = l + font.hmtx.getHorizontalAdvance (gid);
    int t = font.os2.getTypoAscender ();
    int b = font.os2.getTypoDescender ();

    TTCompositeOutline ttOutline = new TTCompositeOutline (components,
        l, r, t, b,
        glyf.data, instructionsOffset, instructionsLength);

    return ttOutline;
  }

  TTSimpleOutline createTTSimpleOutline (OpenTypeFont font, int gid, int offset, int limit)
  throws UnsupportedFontException, InvalidFontException {

    TTSimpleOutline ttOutline = new TTSimpleOutline ();

    if (offset == limit) { // empty outline
      ttOutline.points = new TTPoint [4];
      ttOutline.contourEndPoints = new int [0];
      int numPoints = 0;
      int xMin = 0;
      int l = xMin - font.hmtx.getLeftSideBearing (gid);
      int r = l + font.hmtx.getHorizontalAdvance (gid);
      int t = font.os2.getTypoAscender ();
      int b = font.os2.getTypoDescender ();
      ttOutline.points [numPoints+0] = new TTPoint (l, 0, false);
      ttOutline.points [numPoints+1] = new TTPoint (r, 0, false);
      ttOutline.points [numPoints+2] = new TTPoint (0, t, false);
      ttOutline.points [numPoints+3] = new TTPoint (0, b, false);
      return ttOutline; }

    int numberOfContours = glyf.data.getint16 (offset);
    offset += 2;

    // bounding box
    int xMin = glyf.data.getint16 (offset);
    offset += 8;

    if (numberOfContours < 0) {
      throw new UnsupportedFontException ("composite glyph"); }

    else if (numberOfContours == 0) {
      throw new UnsupportedFontException (); }

    ttOutline.contourEndPoints = new int [numberOfContours];
    for (int i = 0; i < numberOfContours; i++) {
      ttOutline.contourEndPoints [i] = glyf.data.getuint16 (offset);
      offset += 2; }

    int numPoints = ttOutline.contourEndPoints [numberOfContours - 1] + 1; 

    int l = xMin - font.hmtx.getLeftSideBearing (gid);
    int r = l + font.hmtx.getHorizontalAdvance (gid);
    int t = font.os2.getTypoAscender ();
    int b = font.os2.getTypoDescender ();
    ttOutline.points = new TTPoint [numPoints + 4];	  
    ttOutline.points [numPoints+0] = new TTPoint (l, 0, false);
    ttOutline.points [numPoints+1] = new TTPoint (r, 0, false);
    ttOutline.points [numPoints+2] = new TTPoint (0, t, false);
    ttOutline.points [numPoints+3] = new TTPoint (0, b, false);

    // skip the instructions
    ttOutline.instructions = glyf.data;
    ttOutline.instructionsLength = glyf.data.getuint16 (offset);
    offset += 2;
    ttOutline.instructionsOffset = offset;
    offset += ttOutline.instructionsLength;

    // first pass over the flags, to discover the beginning of X and Y
    // coordinates
    int flagOffset = offset;
    int xSize = 0;

    int curPoint = 0;
    while (curPoint < numPoints) {
      int flag = glyf.data.getuint8 (offset++);
      int numRepeats = ((flag & Glyf.CoordFlags.REPEAT) == 0) ? 0 : glyf.data.getuint8 (offset++);
      for (int i = 0; i <= numRepeats; i++) {
        if ((flag & Glyf.CoordFlags.X_SHORT_VECTOR) != 0) {
          xSize += 1; }
        else if ((flag & Glyf.CoordFlags.THIS_X_IS_SAME) == 0) {
          xSize += 2; }
        curPoint++; }}

    int xOffset = offset;
    int yOffset = xOffset + xSize;
    int x = 0;
    int y = 0;

    curPoint = 0;
    while (curPoint < numPoints) {
      int flag = glyf.data.getuint8 (flagOffset++);
      int numRepeats = ((flag & Glyf.CoordFlags.REPEAT) == 0) ? 0 : glyf.data.getuint8 (flagOffset++);
      for (int i = 0; i <= numRepeats; i++) {

        if ((flag & Glyf.CoordFlags.X_SHORT_VECTOR) != 0) {
          if ((flag & Glyf.CoordFlags.THIS_X_IS_SAME) == 0) {
            x -= glyf.data.getuint8 (xOffset); }
          else {
            x += glyf.data.getuint8 (xOffset); }
          xOffset++; }
        else if ((flag & Glyf.CoordFlags.THIS_X_IS_SAME) == 0) {
          x += glyf.data.getint16 (xOffset);
          xOffset += 2; }

        if ((flag & Glyf.CoordFlags.Y_SHORT_VECTOR) != 0) {
          if ((flag & Glyf.CoordFlags.THIS_Y_IS_SAME) == 0) {
            y -= glyf.data.getuint8 (yOffset); }
          else {
            y += glyf.data.getuint8 (yOffset); }
          yOffset++; }
        else if ((flag & Glyf.CoordFlags.THIS_Y_IS_SAME) == 0) {
          y += glyf.data.getint16 (yOffset);
          yOffset += 2; }

        boolean onCurve = (flag & Glyf.CoordFlags.ON_CURVE) != 0;
        try {
        	ttOutline.points [curPoint] = new TTPoint (x, y, onCurve);
        } catch (ArrayIndexOutOfBoundsException e) {
        	throw new InvalidGlyphException(e);
        }

        curPoint++; }}

    return ttOutline; }
  
  
	private String nameGet(String src)
	{
		if (src == null)
			return null;
		StringBuffer dst = new StringBuffer();
		for (int i = 0; i < src.length(); i++) {
			char c = src.charAt(i);
			switch (c) {
			case 0x00A9: {
				dst.append("(C)");
				break;
			}
			case 0x00AE: {
				dst.append("(R)");
				break;
			}
			case 0x2122: {
				dst.append("(TM)");
				break;
			}
			default: {
				if ((c & 0xFF00) != 0) {
					dst.append('?');
				} else {
					dst.append(c);
				}
			}
			}
		}
		return dst.toString();
	}

	private Dict createTopDict(OpenTypeFont otf, boolean cidFont)
		throws InvalidFontException, UnsupportedFontException, UnsupportedEncodingException
  {
	  String registry, ordering;
	  String fullname, familyname, weight, basefontname, fontname;
	  String notice, version, copyright, postscript;
	  Double italicangle, underPos, underThick, strokewidth;
	  Rect bbox;
	  Boolean fixedPitch;
	  Integer paintType, uniqueID, fsType;
	  int[] xuid, bfBlend;
	  
	  registry = (cidFont) ? "Adobe" : null;
	  ordering = (cidFont) ? "Identity" : null;
	  Matrix fontMatrix = (cidFont) ? null : new Matrix(1.0/otf.getUnitsPerEmX(), 0, 0, 1.0/otf.getUnitsPerEmY(), 0, 0);

	  postscript = "/OrigFontType /" + OrigFontType.kTRUETYPE + " def";
	  fontname = otf.getXDCFontDescription(null).getPostscriptName();
	  if (fontname == null || fontname.length() == 0 || fontname.charAt(0) == 0)
		  fontname = "unknown";
	  	
	  Name otfName = otf.name;
	  if (otfName != null) {
		version = nameGet(otfName.selectName(Name.PredefinedNames.VERSION_STRING));
		notice = nameGet(otfName.selectName(Name.PredefinedNames.TRADEMARK));
		copyright = nameGet(otfName.selectName(Name.PredefinedNames.LICENSE_DESCRIPTION));
		fullname = nameGet(otfName.selectName(Name.PredefinedNames.FULL_FONT_NAME));
		familyname = nameGet(otfName.selectName(Name.PredefinedNames.FONT_FAMILY));
		weight = nameGet(otfName.selectName(Name.PredefinedNames.FONT_SUBFAMILY));
	  } else {
        version = notice = copyright = fullname = familyname = weight = null;
	  }

		
	  if (otf.post != null) {
		  fixedPitch = new Boolean(otf.post.isFixedPitch());
		  italicangle = new Double(F16Dot16.toDouble(-otf.post.getItalicAngle()));
		  underPos = new Double (otf.post.getUnderlinePosition());
		  underThick = new Double(otf.post.getUnderlineThickness());
	  } else {
		  fixedPitch = Boolean.FALSE;
		  italicangle = new Double(0);
		  underPos = new Double(-100);
		  underThick = new Double(50);
	  }
	  bbox = otf.head.getFontBBox();

	  //default values of the following are ok
	  paintType = null;
	  uniqueID = null; 
	  strokewidth = null;
	  xuid = null; 
	  fsType = null;
	  bfBlend = null;
	  basefontname = null;

	  
	  return new Dict(registry, ordering, 0, 
	          version, notice, copyright, fullname, fontname, familyname,
	          weight, fixedPitch, italicangle, underPos, underThick,
	          paintType, uniqueID, bbox, strokewidth, xuid,
	          postscript, fsType, bfBlend, basefontname, fontMatrix);

  }
  
	private Dict createPrivateDict(double nominalWidth, double defaultWidth, ZoneHint zoneHint)
  {
/*	  TODO to be filled in correctly after hinting work is completed - SDas
 * 
 * 	  Double blueScale, blueShift;
	  Double expansionFactor, stdHW, stdVW;
	  Boolean forceBold;
	  Integer bluefuzz, langGroup, initRandSeed;
	  double[] blues, otherBlues, familyBlues, familyOtherBlues;
	  double[] stemSnapH, stemSnapV;
	  
	  blues = (double[])getValue (Type1Keys.BlueValues);
	  otherBlues = (double[])getValue (Type1Keys.OtherBlues);
	  familyBlues = (double[])getValue (Type1Keys.FamilyBlues);
	  familyOtherBlues = (double[])getValue (Type1Keys.FamilyOtherBlues);
	  blueScale = (Double)getValue (Type1Keys.BlueScale);
	  blueShift = (Double)getValue (Type1Keys.BlueShift);
	  bluefuzz = (Integer)getValue(Type1Keys.BlueFuzz);
	  stdHW = (Double)getValue (Type1Keys.StdHW);
	  stdVW = (Double)getValue (Type1Keys.StdVW);
	  stemSnapH = (double[])getValue (Type1Keys.StemSnapH);
	  stemSnapV = (double[])getValue (Type1Keys.StemSnapV );
	  forceBold = (Boolean)getValue (Type1Keys.ForceBold);
	  langGroup = (Integer)getValue (Type1Keys.LanguageGroup);
	  expansionFactor = (Double)getValue (Type1Keys.ExpansionFactor);
	  initRandSeed = (Integer)getValue (Type1Keys.initialRandomSeed);
	  
	  return new Dict(blues, otherBlues, familyBlues, familyOtherBlues,
	          blueScale, blueShift, bluefuzz, stdHW, stdVW, stemSnapH, stemSnapV,
	          forceBold, langGroup, expansionFactor, initRandSeed,
	          new Double(nominalWidth), new Double(defaultWidth));
	          */
	  double[] blueValues = null;
	  double[] otherBlues = null;
	  Double blueScale = null;
	  Integer blueFuzz = null;
	  Double blueShift = null;
	  Double stdVW = null;
	  Double stdHW = null;
	  if (zoneHint != null) {
		  blueValues = zoneHint.getBlueValues();
		  if (blueValues != null) {
			  otherBlues = zoneHint.getOtherBlues();
			  blueScale = new Double(zoneHint.getBlueScale());
			  blueFuzz = new Integer(zoneHint.getBlueFuzz());
			  blueShift = new Double (zoneHint.getBlueShift());
		  }
		  stdHW = new Double(zoneHint.getStdHStem());
		  stdVW = new Double(zoneHint.getStdVStem());
	  }

	  return new Dict(blueValues, otherBlues, null, null,
	          blueScale, blueShift, blueFuzz, (stdHW == 0) ? null : stdHW, (stdVW == 0) ? null : stdVW, null, null,
	          null, null, null, null,
	          new Double(nominalWidth), new Double(defaultWidth));
	  
  }
  

	
  public void streamForCFF (OpenTypeFont otf, Map tables, boolean removeOverlaps) 
  throws UnsupportedFontException, InvalidFontException, IOException {
	  subsetAndStreamForCFF(null, otf, tables, removeOverlaps);
  }

	private static final int MAX_GLYF_SIZE_FOR_SUBRIZE = 614400;	// 600KB - same as CoolType

	private boolean checkSubrizeLimit(Subset subset, OpenTypeFont otf)
		throws InvalidFontException, UnsupportedFontException
	{
		if (glyf.data == null)
			return false;
		int glyfSize = glyf.data.getSize();
		if (glyfSize <= MAX_GLYF_SIZE_FOR_SUBRIZE)
			return true;
		if (subset == null)
			return false;
		if ((double)glyfSize * subset.getNumGlyphs() <= (double)MAX_GLYF_SIZE_FOR_SUBRIZE * otf.getNumGlyphs())
			return true;
		return false;
	}

	public void subsetAndStreamForCFF (Subset subset, OpenTypeFont otf, Map tables, boolean removeOverlaps)
		throws UnsupportedFontException, InvalidFontException, IOException
	{
		Type2CStringGenerator t2StringGenerator;
		NonOverlappingOutlineConsumer iSectOC;
		ZoneHint zoneHint = null;
		AutoColor autoColor = null;
		boolean enableHinting = true;
		if (subset == null)
			subset = new SubsetDefaultImpl(otf.getNumGlyphs(), false);
		boolean enableSubrizer = checkSubrizeLimit(subset, otf);
		if (!removeOverlaps) {
			t2StringGenerator = new Type2CStringGenerator(subset.getNumGlyphs(), 1, enableSubrizer);
			for (int i=0; i < subset.getNumGlyphs(); i++) {
				double width = otf.getHorizontalAdvance(subset.getFullGid(i));
				t2StringGenerator.newGlyph(i, 0,width, Type2CStringGenerator.DEFAULT_NOMINAL_WIDTH);
				otf.getGlyphOutline(subset.getFullGid(i), t2StringGenerator, TTParser.APPROX_PATH);
			}
		} else {
			if (enableHinting) {
				zoneHint = new ZoneHint(otf);
			}
			t2StringGenerator = new Type2CStringGenerator(subset.getNumGlyphs(), 1, enableSubrizer);
			if (enableHinting) {
				autoColor = new AutoColor(t2StringGenerator, otf.getUnitsPerEmY(),
							  AutoColor.AC_HINTSUB | AutoColor.AC_FIXWINDING,
							  true, false, zoneHint.getTopZones(), zoneHint.getBottomZones());
				iSectOC = new NonOverlappingOutlineConsumer(autoColor, otf.getUnitsPerEmY());
			} else {
				iSectOC = new NonOverlappingOutlineConsumer(t2StringGenerator, otf.getUnitsPerEmY());
			}
			for (int i=0; i < subset.getNumGlyphs(); i++) {
				double width = otf.getHorizontalAdvance(subset.getFullGid(i));
				if (enableHinting) {
					autoColor.newGlyph(i, width, 0);
				}
				t2StringGenerator.newGlyph(i, 0,width, Type2CStringGenerator.DEFAULT_NOMINAL_WIDTH);
				otf.getGlyphOutline(subset.getFullGid(i), iSectOC, TTParser.APPROX_PATH);
			}
		}
		String fontName = otf.getXDCFontDescription(null).getPostscriptName();
		if (fontName == null || fontName.length() == 0 || fontName.charAt(0) == 0)
			fontName = "unknown";
		double nominalWidth = t2StringGenerator.calculateNominalWidth(0);
		double defaultWidth = t2StringGenerator.calculateDefaultWidth(0);
		CharStrings charStrings = t2StringGenerator.getCharstringIndex();
		CharStrings lSubrStrings = null;
		CharStrings gSubrStrings = null;
		if (enableSubrizer) {
			CFFSubrize subrizer = new CFFSubrize();
			charStrings = subrizer.subrize(charStrings);
			lSubrStrings = subrizer.getLSubrs();
			gSubrStrings = subrizer.getGSubrs();
		}
		Dict topDict = createTopDict(otf, true);
		Dict privateDict = createPrivateDict(nominalWidth, defaultWidth, zoneHint);
		OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor stream = new OTByteArrayBuilder.OTByteArrayBuilderOutputStreamAdaptor();
		Dict cidDict = new Dict(new Matrix(1.0/otf.getUnitsPerEmX(), 0, 0, 1.0/otf.getUnitsPerEmY(), 0, 0));
		CIDKeyedFont ckFont = new CIDKeyedFont(fontName, topDict, charStrings, subset.getNumGlyphs(), cidDict, privateDict, lSubrStrings, gSubrStrings);
		ckFont.subsetAndStream(subset, stream, false, null, false, false);
		tables.put(new Integer(Tag.table_CFF), stream.getBuilder());
	}
}
