/*
*
*	File: LayoutTable.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;

/**
 * A LayoutTable is the base class for GSUB/GPOS/GDEF/BASE tables, and
 * captures the common types: coverages, class definitions, etc.
 */
public abstract class LayoutTable extends Table {

  protected LayoutTable (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
  }
  
  /** Determines if a value of lookupFlag covers a glyph.
   * 
   * @param lookupFlag the value of lookupFlag
   * @param gdef the GDEF table from which to get glyph classes
   * @param glyphID the glyph to test
   * @return true iff <code>glyphID</code> is covered by <code>lookupFlag</code>.
   * @throws InvalidFontException if the font is invalid
   */
  protected boolean lookupFlagCovers (int lookupFlag, Gdef gdef, int glyphID)
  throws InvalidFontException {
    
    if (glyphID == -1) {
      return false; }
    
    if ((lookupFlag & 0x0E) != 0) {
      if (gdef == null) {
        throw new InvalidFontException ("lookupflag value requires a GDEF table, which is absent"); }
      int glyphClass = gdef.getGlyphClass (glyphID);
      if (   ((lookupFlag & 0x2) != 0 && glyphClass == 1)
          || ((lookupFlag & 0x4) != 0 && glyphClass == 2)
          || ((lookupFlag & 0x8) != 0 && glyphClass == 3))
        return true; }
    
    if ((lookupFlag & 0xff00) != 0) {
      if (gdef == null) {
        throw new InvalidFontException ("lookupflag value requires a GDEF table, which is absent"); }
      if (gdef.getGlyphClass (glyphID) == 3) {
        int markClass = gdef.getMarkAttachClass (glyphID);
        if (((lookupFlag & 0xff00) >> 8) != markClass) {
          return true; }}}
    
    return false;
  }

  private void iterateClassFormat1(int classDefOffset, int numGlyphs, ClassConsumer consumer, int theClass) throws InvalidFontException, UnsupportedFontException {
	  int start = LayoutTable.this.data.getuint16(classDefOffset+2);
	  int end = start + LayoutTable.this.data.getuint16(classDefOffset+4);
	  int index;
	  
	  // iterate the inital hole if we are doing class 0.
	  if (theClass == 0 || theClass == -1)
	  {
		  for (index = 0; index < start; index++)
			  if (!consumer.glyph(index, 0))
				  return;
	  }
	  
	  // iterate the range looking for glyphs with the right class.
	  for (index = start; index < end; index++)
	  {
		  int thisClass = this.data.getuint16(classDefOffset + 6 + 2*(index-start));
		  if (thisClass == theClass || theClass == -1) 
		  {
			  if (!consumer.glyph(index, thisClass))
				  return;
		  }
	  }
	  
	  // iterate the final hole if we are doing class 0.
	  if (theClass == 0 || theClass == -1)
	  {
		  for (index = end; index < numGlyphs; index++)
			  if (!consumer.glyph(index, 0))
				  return;
	  }
	  
  }
  
  
  private void iterateClassFormat2(int classDefOffset, int numGlyphs, ClassConsumer consumer, int theClass) throws InvalidFontException, UnsupportedFontException {
	  int index = 0;
	  int numRanges = LayoutTable.this.data.getuint16(classDefOffset + 2);
	  int start;
	  
	  // iterate the initial "hole" if we are looking for class 0
	  if (theClass == 0 || theClass == -1)
	  {
		  if (numRanges == 0)
		  {
			  while (index < numGlyphs)
			  {
				  if (!consumer.glyph(index++, 0))
					  return;
			  }
			  return;
		  }
	
		  start = LayoutTable.this.data.getuint16(classDefOffset + 4);
		  while (index < start) {
			  if (!consumer.glyph(index++, 0))
				  return;
		  }
	  }
	  
	  for (int i = 0; i < numRanges; i++)
	  {
		  int thisClass = this.data.getuint16(classDefOffset + 4 + 6*i + 4);
		  int end = this.data.getuint16(classDefOffset + 4 + 6*i + 2);
		  
		  // iterate the range if it is the right class.
		  if (thisClass == theClass || theClass == -1)
		  {
			  start = this.data.getuint16(classDefOffset + 4 + 6*i);
			  end = this.data.getuint16(classDefOffset + 4 + 6*i + 2);
			  for (index = start; index <= end; index++)
				  if (!consumer.glyph(index, thisClass))
					  return;
		  }
		  index = end+1;
		  
		  // iterate the hole between ranges if it exists and we are looking at class 0.
		  if (theClass == 0 || theClass == -1) 
		  {
			  if (i == numRanges - 1)
				  end = numGlyphs;
			  else 
				  end = this.data.getuint16(classDefOffset + 4 + 6*(i+1));
			  
			  while (index < end)
				  if (!consumer.glyph(index++, 0))
					  return;
		  }
	  }
  }

  interface ClassConsumer {
	  /** The next glyphID in this class. If false is returned from this method, the iterator stops 
	 * @throws InvalidFontException 
	 * @throws UnsupportedFontException */
	  public boolean glyph(int glyphID, int classID) throws UnsupportedFontException, InvalidFontException;
  }
  
  /**
   * Iterate all of the glyphs in a given class definition and send each glyph in a specified class to a consumer.
   * 
   * @param classDefOffset The offset to a class definition structure to be iterated
   * @param numGlyphs The number of glyphs in the font
   * @param consumer The consumer of the glyph ids
   * @param theClass The class that should be iterated. -1 if all classes should be iterated
   * @throws InvalidFontException
 * @throws UnsupportedFontException 
   */
  void iterateClass(int classDefOffset, int numGlyphs, ClassConsumer consumer, int theClass) throws InvalidFontException, UnsupportedFontException
  {
	  int format = this.data.getuint16(classDefOffset);
	  switch (format) {
	  case 1: 
		  iterateClassFormat1(classDefOffset, numGlyphs, consumer, theClass);
		  break;
	  case 2:
		  iterateClassFormat2(classDefOffset, numGlyphs, consumer, theClass);
		  break;
      default:
    	  throw new InvalidFontException("Invalid class def format (" + format + ")");
	  }
  }
  
  private void iterateFormat1Coverage(int coverageOffset, Subset glyphsToInclude, CoverageConsumer consumer) throws InvalidFontException, UnsupportedFontException {
	  int numGlyphs = LayoutTable.this.data.getuint16 (coverageOffset + 2);
	  int currentGlyph = 0;
	  
	  for (; currentGlyph < numGlyphs; currentGlyph++)
	  {
		  int g = LayoutTable.this.data.getuint16 (coverageOffset + 4 + 2*currentGlyph);
		  if (glyphsToInclude != null && glyphsToInclude.getExistingSubsetGid(g) == -1) {
			  continue;
		  }
		  
		  if (!consumer.glyphInfo(g, currentGlyph))
			  return;
		  
	  }
  }
  
  private void iterateFormat2Coverage(int coverageOffset, Subset glyphsToInclude, CoverageConsumer consumer) throws InvalidFontException, UnsupportedFontException {
	  final int rangeCount = LayoutTable.this.data.getuint16 (coverageOffset + 2);
	  int indicesInRange;
	  int indexInRange = 0;
	  int currentRange = 0;
	  
	  if (rangeCount > 0)
		  indicesInRange = LayoutTable.this.data.getuint16 (coverageOffset + 4 + 2) - LayoutTable.this.data.getuint16 (coverageOffset + 4) + 1;
	  else
		  indicesInRange = 0;
	  
	  while (indexInRange < indicesInRange || currentRange < rangeCount-1)
	  {
		  if (indexInRange == indicesInRange) {
			  currentRange++;
			  indexInRange = 0;
			  indicesInRange = LayoutTable.this.data.getuint16 (coverageOffset + 4 + 6*currentRange + 2) 
				 					- LayoutTable.this.data.getuint16 (coverageOffset + 4 + 6*currentRange) + 1;
		  }
		  
		  int gid = LayoutTable.this.data.getuint16 (coverageOffset + 4 + 6*currentRange) + indexInRange;
		  if (glyphsToInclude == null || glyphsToInclude.getExistingSubsetGid(gid) != -1)
		  {
			  int coverageIndex = LayoutTable.this.data.getuint16 (coverageOffset + 4 + 6*currentRange + 4) + indexInRange;
			  if (!consumer.glyphInfo(gid, coverageIndex))
				  return;
		  }
		  indexInRange++;
	  }  
  }
  
  interface CoverageConsumer {
	  /** The next glyph id and coverage index in this coverage. If false is returned from glyphInfo, the iterator stops. 
	 * @throws UnsupportedFontException */
	  boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException;
  }
  
  /**
   * Iterates over the specified coverage, returning gids and coverage indices in the order they
   * appear in the coverage. Only glyphs in glyphsToInclude are sent to the consumer, unless glyphsToInclude
   * is null (in which case, all glyphs are sent to the consumer).
 * @throws UnsupportedFontException 
   */
  void iterateCoverage(int coverageOffset, Subset glyphsToInclude, CoverageConsumer consumer) throws InvalidFontException, UnsupportedFontException {
	  int format = LayoutTable.this.data.getuint16 (coverageOffset);
	  switch (format) {
      case 1: {
    	  iterateFormat1Coverage(coverageOffset, glyphsToInclude, consumer);
    	  break;
      }
      case 2: {
    	  iterateFormat2Coverage(coverageOffset, glyphsToInclude, consumer);
    	  break;
      }
      default:
    	  throw new InvalidFontException("Invalid coverage format " + format);
	  } 
  }  
  
  /**
   * Returns the coverage index of glyphID using the coverage at
   * coverageOffset. If the glyph is not covered, returns -1.
   *
   * @param glyphID the glyph of interest
   * @param coverageOffset the offset the coverage table in the current table
   * @return -1 if the glyph is not covered, its coverage index  otherwise. 
   * @throws InvalidFontException if the font is invalid
   */
  protected int getCoverageIndex (int glyphID, int coverageOffset)
  throws InvalidFontException {
    
    if (glyphID == -1) {
      return -1; }
    
    int format = this.data.getuint16 (coverageOffset);
    
    switch (format) {
      case 1: { 
        int min = 0; 
        int max = this.data.getuint16 (coverageOffset + 2) - 1;
        
        while (min <= max) {
          int mid = (min + max) / 2;
          int g = this.data.getuint16 (coverageOffset + 4 + 2*mid);
          
          if (glyphID < g) {
            max = mid - 1; }
          else if (g < glyphID) {
            min = mid + 1; }
          else {
            return mid; }}
        
        return -1; }

      case 2: { 
        int min = 0;
        int max = this.data.getuint16 (coverageOffset + 2) - 1;
        
        while (min <= max) {
          int mid = (min + max) / 2;
          int start = this.data.getuint16 (coverageOffset + 4 + 6*mid);
          int stop = this.data.getuint16 (coverageOffset + 4 + 6*mid + 2);
          
          if (glyphID < start) {
            max = mid - 1; }
          else if (stop < glyphID) {
            min = mid + 1; }
          else {
            return this.data.getuint16 (coverageOffset + 4 + 6*mid + 4) 
                     + (glyphID - start); }}
        
        return -1; }

      default: {
        throw new InvalidFontException ("invalid coverage format (" + format + ")"); }}
  }
  
  
  /**
   * Returns the class index of glyphID using the class definition at
   * classdefOffset. If the glyph is not covered, returns -1.
   *
   * @param glyphID the glyph of interest
   * @param classdefOffset the offset the coverage table in the current table
   * @return the class index 
   * @throws InvalidFontException if the font is invalid
   */
  protected int getClassIndex (int glyphID, int classdefOffset)
  throws InvalidFontException {
    
    if (glyphID == -1) {
      return -1; }
    
    int format = this.data.getuint16 (classdefOffset);
    
    switch (format) {
      case 1: { 
        int startGlyph = this.data.getuint16 (classdefOffset + 2);
        int glyphCount = this.data.getuint16 (classdefOffset + 4);
        
        if (startGlyph <= glyphID && glyphID < startGlyph + glyphCount) {
          return this.data.getuint16 (classdefOffset + 6 + (glyphID - startGlyph)*2); }
        
        return 0; }

      case 2: { 
        int rangeCount = this.data.getuint16 (classdefOffset + 2);
        int min = 0;
        int max = rangeCount - 1;
        
        while (min <= max) {
          int mid = (min + max) / 2;
          int start = this.data.getuint16 (classdefOffset + 4 + mid*6);
          int stop = this.data.getuint16 (classdefOffset + 4 + mid*6 + 2);
          
          if (glyphID < start) {
            max = mid - 1; }
          else if (stop < glyphID) {
            min = mid + 1; }
          else {
            return this.data.getuint16 (classdefOffset + 4 + mid*6 + 4); }}
        
        return 0; }

      default: {
        throw new InvalidFontException ("invalid classdef format (" + format + ")"); }}
  }
}
