/**
 * 
 */
package com.adobe.fontengine.fontmanagement.fxg;

import java.util.HashMap;
import java.util.Iterator;

import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.fontmanagement.FontResolutionPriority;
import com.adobe.fontengine.fontmanagement.IntelligentResolver;

/**
 * @author sgill
 *
 */
public class FXGFontResolverImpl implements FXGFontResolver 
{
	private static final class FXGKey
	{
		private final String familyName;
		private final boolean isBold;
		private final boolean isItalic;

		protected FXGKey(FXGFontDescription fxgDescription)
		{
			this.familyName = fxgDescription.getFamilyName();
			this.isBold = fxgDescription.isBold();
			this.isItalic = fxgDescription.isItalic();
		}

		protected FXGKey(FXGFontSearchAttributes fxgSearchAttributes)
		{
			this.familyName = fxgSearchAttributes.getFamilyName();
			this.isBold = fxgSearchAttributes.isBold();
			this.isItalic = fxgSearchAttributes.isItalic();
		}


		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		public int hashCode() 
		{
			final int prime = 31;
			int result = 1;
			result = prime * result	+ ((familyName == null) ? 0 : familyName.hashCode());
			result = prime * result + (isBold ? 1231 : 1237);
			result = prime * result + (isItalic ? 1231 : 1237);
			return result;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		public boolean equals(Object obj) 
		{
			if (this == obj)
			{
				return true;
			}
			if (obj == null)
			{
				return false;
			}
			if (!(obj instanceof FXGKey))
			{
				return false;
			}
			final FXGKey other = (FXGKey) obj;
			if (familyName == null) 
			{
				if (other.familyName != null)
				{
					return false;
				}
			} else if (!familyName.equals(other.familyName))
			{
				return false;
			}
			if (isBold != other.isBold)
			{
				return false;
			}
			if (isItalic != other.isItalic)
			{
				return false;
			}
			return true;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		public String toString() 
		{
			return new String(this.familyName + (this.isBold ? ", bold" : "") + (this.isItalic ? ", italic" : ""));
		}
	}

	private static final class FXGValue
	{
		private final FXGFontDescription fxgDescription;
		private final Font font;

		FXGValue(FXGFontDescription fxgDescription, Font font)
		{
			this.fxgDescription = fxgDescription;
			this.font = font;
		}

		FXGFontDescription getFXGDescription()
		{
			return this.fxgDescription;
		}

		Font getFont()
		{
			return this.font;
		}
		
		public String toString()
		{
			return new String (this.fxgDescription.toString() + "\n\t" + this.font.toString());
		}
	}

	private HashMap/*<FXGKey, FXGValue>*/ fxgFonts;
	private FontResolutionPriority resolutionPriority = FontResolutionPriority.FIRST;

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public FXGFontResolverImpl() 
	{
		this.fxgFonts = new HashMap/*<FXGFontDescription,Font>*/();
	}

	public FXGFontResolverImpl(FXGFontResolverImpl original) 
	{
		this.fxgFonts = (HashMap/*<FXGFontDescription, Font>*/) original.fxgFonts.clone();
		this.resolutionPriority = original.resolutionPriority;
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.fxg.FXGNameResolver#addFont(com.adobe.fontengine.font.Font)
	 */
	public void addFont(Font font) 
	throws UnsupportedFontException, InvalidFontException, FontLoadingException 
	{
		FXGFontDescription[] fxgDesc = font.getFXGFontDescription();
		for (int i = 0; i < fxgDesc.length; i++)
		{
			this.addFont(fxgDesc[i], font);
		}
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.fxg.FXGNameResolver#addFont(com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription, com.adobe.fontengine.font.Font)
	 */
	public void addFont(FXGFontDescription fxgDesc, Font font)
	throws UnsupportedFontException, InvalidFontException, FontLoadingException 
	{
		FXGKey searchKey = new FXGKey(fxgDesc);
		FXGValue oldValue = (FXGValue) fxgFonts.get(searchKey);
		if (oldValue != null) 
		{
			Font oldFont = oldValue.getFont();
			if (this.resolutionPriority == FontResolutionPriority.FIRST) 
			{
				// don't add as the old one has priority
				return; 
			}
			else if (this.resolutionPriority == FontResolutionPriority.INTELLIGENT_LAST
					|| this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST) 
			{
				Font preferredFont = IntelligentResolver.choosePreferredFont(oldFont, font,
						this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST);
				if (preferredFont == oldFont) 
				{
					return; 
				}
			}
		}
		fxgFonts.put (searchKey, new FXGValue(fxgDesc, font));
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.fxg.FXGNameResolver#findFont(com.adobe.fontengine.fontmanagement.fxg.FXGFontSearchAttributes)
	 */
	public Font findFont(FXGFontSearchAttributes searchAttributes) 
	{
		FXGValue value = (FXGValue) this.fxgFonts.get(new FXGKey(searchAttributes));
		if (value == null)
		{
			return null;
		}
		return value.getFont();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.fxg.FXGNameResolver#isEmpty()
	 */
	public boolean isEmpty() 
	{
		return fxgFonts.isEmpty();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.fxg.FXGNameResolver#setResolutionPriority(com.adobe.fontengine.fontmanagement.FontResolutionPriority)
	 */
	public FontResolutionPriority setResolutionPriority(FontResolutionPriority priority) 
	{
		FontResolutionPriority oldPriority = this.resolutionPriority;

		this.resolutionPriority = priority;
		return oldPriority;		
	}

	public int hashCode() 
	{
		final int prime = 31;
		int result = 1;
		result = prime * result	+ ((fxgFonts == null) ? 0 : fxgFonts.hashCode());
		result = prime * result	+ ((resolutionPriority == null) ? 0 : resolutionPriority.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
		{
			return true;
		}
		if (obj == null)
		{
			return false;
		}
		if (!(obj instanceof FXGFontResolverImpl))
		{
			return false;
		}
		final FXGFontResolverImpl other = (FXGFontResolverImpl) obj;
		if (fxgFonts == null) 
		{
			if (other.fxgFonts != null)
			{
				return false;
			}
		} 
		else if (!fxgFonts.equals(other.fxgFonts))
		{
			return false;
		}
		if (resolutionPriority == null) 
		{
			if (other.resolutionPriority != null)
			{
				return false;
			}
		} else if (!resolutionPriority.equals(other.resolutionPriority))
		{
			return false;
		}
		return true;
	}

	public String toString()
	{
		StringBuffer sb = new StringBuffer ();

		sb.append ("FXG Fonts:\n");
		for (Iterator it = this.fxgFonts.keySet().iterator(); it.hasNext (); ) 
		{
			FXGKey k = (FXGKey) it.next();
			sb.append ("  ");
			sb.append (k);
			sb.append (" = ");
			sb.append (((FXGValue)this.fxgFonts.get(k)).getFont().toString());
			sb.append ("\n");
		}
		return sb.toString();
	}
}
