/**
 * 
 */
package com.adobe.internal.io;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

/**
 * An extension of the <code>DataInputStream</code> providing more operations.
 *
 */
public class ExtendedDataInputStream extends DataInputStream 
{
	/**
	 * Constructor.
	 * @param in the stream to wrap
	 */
	public ExtendedDataInputStream(InputStream in) 
	{
		super(in);
	}

	/**
	 * Read an unsigned integer from the contained input stream.
	 * @return the next four bytes of this input stream, interpreted as an unsigned 32-bit integer
	 * @throws IOException if an I/O error occurs
	 * @throws EOFException if this input stream reaches the end before reading four bytes
	 */
	public long readUnsignedInt() 
	throws IOException
	{
		long value = 0;
		for (int i = 0; i < 4; i++)
		{
			int b = this.read();
			if (b == -1)
			{
				throw new EOFException("Unexpected end of stream.");
			}
			value = (value << 8) + (b & 0xff);
		}
		return value;
	}

	/**
	 * Read an unsigned integer from the contained input stream.
	 * @return the next three bytes of this input stream, interpreted as an unsigned 24-bit integer
	 * @throws IOException if an I/O error occurs
	 * @throws EOFException if this input stream reaches the end before reading three bytes
	 */
	public int readUnsigned3ByteInt() 
	throws IOException 
	{
		int value = 0;
		for (int i = 0; i < 3; i++)
		{
			int b = this.read();
			if (b == -1)
			{
				throw new EOFException("Unexpected end of stream.");
			}
			value = (value << 8) + (b & 0xff);
		}
		return value;
	}
	
	/**
	 * Skips over and discards <code>n</code> bytes of data from the input stream.  This method attempts to
	 * fully skip the number of bytes specified.
	 * @param n the number of bytes to be skipped
	 * @throws IOException if an I/O error occurs
	 * @throws EOFException if this input stream reaches the end before skipping the given number of bytes
	 */
	public void skipFully(long n) 
	throws IOException 
	{
		long totalBytesSkipped = 0;
		while (n > 0)
		{	
			long bytesSkipped = super.skip(n);
			if (bytesSkipped == 0)
			{
				throw new EOFException("Unable to skip.");
			}
			n -= bytesSkipped;
			totalBytesSkipped += bytesSkipped;
		}
	}
}
