/*
* File: BitInputStream.java
*
*************************************************************************
** ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2012 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
  from Adobe Systems Incorporated.
**************************************************************************/
package com.adobe.internal.io.stream;

import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidParameterException;

/**
 * This class reads bit by bit data from a back-end stream which is passed in constructor.
 * @author hkansal
 *
 */
public class BitInputStream{
	private static final int BUFFER_SIZE = 512;
	
	private InputStream stream = null;
	private int currentBitPos = 0;
	private byte[] buffer = null; 
	private int currentBytePos = 0;
	private int maxBytesLimit = BUFFER_SIZE;
	
	/**
	 * Reads chunk of data on demand which is cached for further 
	 * bit by bit reading.
	 * @throws IOException
	 */
	private void fillBuffer() throws IOException{
		if(stream.available() > 0)
			maxBytesLimit = stream.read(buffer);
		else
			maxBytesLimit = -1;
		currentBytePos = 0;
	}
	/**
	 * Constructor for this stream.
	 * @param stream This stream is used as back-end stream to read bit by bit data.
	 * @throws IOException
	 */
	public BitInputStream(InputStream stream) throws IOException{
		this.stream = stream;
	}
	
	private int readData(int nob) throws IOException{
		if(maxBytesLimit == -1)
			throw new IOException("End of stream has been reached so no data can be read now.");
		if(currentBytePos == maxBytesLimit)
			fillBuffer();// buffer more data to perform this read operation.
		int value = (255&(buffer[currentBytePos]<<currentBitPos)) >> (Byte.SIZE-nob);
		currentBitPos+=nob;
		if(currentBitPos >=Byte.SIZE)
			currentBytePos++;
		currentBitPos%=Byte.SIZE;
		return value;
	}
	/**
	 * Reads nob number of bits from available data and returns an integer value.
	 * Throws {@link IOException} if end of stream has been reached.
	 * @param nob
	 * @return int
	 * @throws IOException
	 */
	public int read(int nob) throws IOException {
		if(buffer == null){
			buffer = new byte[BUFFER_SIZE];
			fillBuffer();
		}
		int value = 0;
		int bitsToRead = 0;
		while(nob > 0){
			bitsToRead = (nob <= (Byte.SIZE-currentBitPos)?nob:(Byte.SIZE-currentBitPos));
			nob-=bitsToRead;
			value = (value | (readData(bitsToRead)<<nob));
		}
		return value;
	}
	
	/**
	 * Reads next 1 bit of available data.
	 * Throws {@link IOException} if end of stream has been reached.
	 */
	public int read() throws IOException {
		
		return read(1);
	}
	
	/**
	 * Releases the resources hold by this instance.
	 */
	public void close() throws IOException{
		if(this.stream != null)
			this.stream.close();
		this.buffer = null;
	}
	
	/**
	 * This method skips n number of bits from available data.
	 */
	public long skip(long n) throws IOException {
		read((int) n);
		return n;
	}
	
	/**
	 * This method breaks the back-end stream into data samples
	 * each having "bitsPerValue" number of bits and writes length number of samples
	 * to the byte array passed.
	 * Throws an exception if bitsPerValue is greater than 8 OR byte array passed null OR it's length < "length"
	 * Returns the number of values actually written to the byte array.
	 * @param b
	 * @param bitsPerValue
	 * @return int
	 */
	public int read(byte b[], int bitsPerValue, int length) {
		if(b == null || length > b.length){
			throw new InvalidParameterException("Byte array passed is either null or insufficient to read the number of bytes actually asked to read.");
		}
		if(bitsPerValue > Byte.SIZE)
			throw new InvalidParameterException("BitsPerValue is " + bitsPerValue + " which is greater than 8 and cann't be fit into byte.");
		int i=0;
		try{
			for(i=0;i<length;i++){
				if(maxBytesLimit == -1)
					break;
				b[i] = (byte) read(bitsPerValue);
			}
		}catch(IOException e){
		}
		return i;
	}
	
	/**
	 * This method breaks the back-end stream into data samples
	 * each having "bitsPerValue" number of bits and writes length number of samples
	 * to the byte array passed.
	 * Throws an exception if byte array passed null OR it's length < "length"
	 * Returns the number of values actually written to the byte array.
	 * @param b
	 * @param bitsPerValue
	 * @return int
	 */
	public int read(int b[], int bitsPerValue, int length) {
		if(b == null || length > b.length){
			throw new InvalidParameterException("Byte array passed is either null or insufficient to read the number of bytes actually asked to read.");
		}
		int i=0;
		try{
			for(i=0;i<length;i++){
				if(maxBytesLimit == -1)
					break;
				b[i] = read(bitsPerValue);
			}
		}catch(IOException e){
		}
		return i;
	}
	
	/**
	 * This method returns approximate number of bits which are available to read.
	 * Actual number of bits may be more than the count returned by this method but not less than that. 
	 * @return int
	 * @throws IOException 
	 */
	public int available() throws IOException{
		if(maxBytesLimit == -1)
			return 0;
		return (Byte.SIZE - currentBitPos) + ((maxBytesLimit - currentBytePos) + this.stream.available())*Byte.SIZE;
	}

}
